/*
 * speed.c - (c) 1998 Andreas Beck   becka@ggi-project.org
 *
 * This is a demonstration of LibGGI's functions and can be used as a
 * reference programming example.
 *
 *   This software is placed in the public domain and can be used freely
 *   for any purpose. It comes without any kind of warranty, either
 *   expressed or implied, including, but not limited to the implied
 *   warranties of merchantability or fitness for a particular purpose.
 *   Use it at your own risk. the author is not responsible for any damage
 *   or consequences raised by use or inability to use this program.
 */

/*
 * This is a GGI speed- and consistency-test application.
 * Policy : Speed-Tests send their output to stdout,
 *          Consistency-Tests report to stderr.
 */

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/times.h>

#ifdef _POSIX_PRIORITY_SCHEDULING
#  include <sched.h>
#else
#  define sched_yield() ;
#endif

#ifdef __mips
#define CLOCKS_PER_SEC CLK_TCK
#endif

#include <ggi/ggi.h>


/* What to test : */

enum flags {
	CONSISTENCY=1,		/* Check if the functions work "right" */
	BREAK_UNCONSISTENT=2,	/* Stop consistency tests, when a few failures have been recorded */
	SPEED=4,			/* Check speed */
	VERBOSE=8			/* Verbose messages */
} flags = 0;

struct {
	ggi_visual_t vis;
	int sx,sy,vx,vy,fx,fy;
	ggi_graphtype type;
} mode;

/**********************************************************************/

void waitabit(void)
{
	int key;

	while (!ggiKbhit(mode.vis)) {
		sched_yield();
	}

	key=ggiGetc(mode.vis);

	if ((key == 'q') || (key == 'Q')) { /* Q pressed */
		ggiExit();
	}
}

struct tms timer;
double u_time,s_time;

void
time_start(void)
{
	times(&timer);
}

void
time_offset(void)
{
	struct tms end; 
	times(&end);
	u_time=-(end.tms_utime-timer.tms_utime)/(double)CLOCKS_PER_SEC;
	s_time=-(end.tms_stime-timer.tms_stime)/(double)CLOCKS_PER_SEC;
	time_start();
}

void
time_stop(void)
{
	struct tms end; 
	times(&end);
	u_time+=(end.tms_utime-timer.tms_utime)/(double)CLOCKS_PER_SEC;
	s_time+=(end.tms_stime-timer.tms_stime)/(double)CLOCKS_PER_SEC;
}

ggi_uint white_pixel;

void
TestName(const char *name)
{
	ggiSetGCForeground(mode.vis,white_pixel);
	ggiPuts(mode.vis,0,0,name);
}

void
nothing(ggi_visual_t vis, int foo, int bar, int foo2, int bar2)
{}

int
getnumpixels(int x1,int y1,int x2,int y2)
{
	int x,y,c; 
	ggi_uint i;
	c=0;
	for (y=y1; y<=y2; y++) {
		for (x=x1; x<=x2; x++) {
	  		ggiGetPixel(mode.vis,x,y,&i);
	  		if (i != 0) c++;
		}
	}
	return c;
}

void
BasicConsistency(void)
{
	int c,x,y;
	ggi_uint i;
	
	/* Get/Put/DrawPixel consistency test */
	ggiSetGCForeground(mode.vis,0);
	ggiFillscreen(mode.vis);

	/* The screen should now be all zero. */
	c=0;
	for (y=0;y<mode.vy;y++)
		for (x=0;x<mode.vx;x++)
		{
	  		ggiGetPixel(mode.vis,x,y,&i);
	  		if ( i != 0 ) 
	  		{
	  			printf("Warning: Screen not blank at %d,%d (%x)\n",x,y,i);
	  			c++;
	  			if (c>16)
	  			{
	  				flags&=~CONSISTENCY;
		  			fprintf(stderr,"Error: Screen not blank or GetPixel not working. Disabled consitency checks.\n");
					y=mode.vy;
		  			break;
	  			}
	  		}
		}
}

void
ColorWrap(void)
{
	ggi_uint i,c;
	/* Now we check when we have a color wraparound. */
	if (flags&CONSISTENCY)
		for (i=0;i!=0xffffffff;i++)
		{
			ggiPutPixel(mode.vis,0,0,i);
			ggiGetPixel(mode.vis,0,0,&c);
			if ( i != c ) 
			{	fprintf(stderr,"Info:Color wraps at %d\n",i);
				break;
			}
		}
}

void
Hline(void)
{
	int x,y,c;

	TestName("Hline");
	ggiSetGCForeground(mode.vis,0);
	ggiFillscreen(mode.vis);
	if (flags&CONSISTENCY) {
		for (x=0;x<mode.vx;x++) {
			for (y=0;y<=mode.vx-x;y++) {
				ggiSetGCForeground(mode.vis,white_pixel);
				ggiDrawHLine(mode.vis,x,0,y);
				ggiSetGCForeground(mode.vis,0);
				/* for speed reasons, we hope for no stray pixels far away ... */
				if ((c=getnumpixels(0,0,mode.vx-1,1))!=y || getnumpixels(x,0,x+y-1,0)!=y ) {
					ggiFillscreen(mode.vis);
					fprintf(stderr,"Error:Hline(%d,0,%d); consistency failed (%d pixels measured).\n",x,y,c);
				} else {
					ggiDrawHLine(mode.vis,x,0,y);
				}
			}
		}
	}
}

void
Vline(void)
{
	int x,y,c;

	if (flags&CONSISTENCY)
		for (y=0;y<mode.vy;y++)
		{
			ggiSetGCForeground(mode.vis,0);
			ggiFillscreen(mode.vis);
			ggiSetGCForeground(mode.vis,white_pixel);
			for (x=0;x<=mode.vy-y;x++)
			{
				ggiDrawVLine(mode.vis,0,y,x);
				/* for speed reasons, we hope for no stray pixels far away ... */
				if ((c=getnumpixels(0,0,1,mode.vy-1))!=x || (c=getnumpixels(0,y,0,x+y-1))!=x )
				{
					fprintf(stderr,"Error:Vline(0,%d,%d); consistency failed (%d pixels measured).\n",y,x,c);
				}
			}
		}
}

void
CheckLine(ggi_visual_t vis,int x1,int y1,int x2,int y2)
{
	int xx,max,cx,cy;
	ggi_uint c;
	ggiSetGCForeground(vis,white_pixel);
	ggiDrawLine(vis,x1,y1,x2,y2);
	ggiSetGCForeground(vis,0);
	max=abs(x2-x1);if (abs(y2-y1)>max) max=abs(y2-y1);
	if (max==0) return;
	for (xx=0;xx<=max;xx++)
	{
		cx=(x1*xx+x2*(max-xx)+max/2)/max;
		cy=(y1*xx+y2*(max-xx)+max/2)/max;
		ggiGetPixel (vis,cx,cy,&c);
	  	if (c==0) printf("Line: Unset pixel %d,%d in line(%d,%d,%d,%d).\n",
	  			  cx,cy,x1,y1,x2,y2);
		ggiDrawPixel(vis,cx,cy);
	}
	if ((c=getnumpixels(0,0,120,120))!=0)
	{	printf("Line: %d surplus pixels in line(%d,%d,%d,%d).\n",
  			  c,x1,y1,x2,y2);
		ggiFillscreen(vis);
	}
}

void
Line(void)
{
	int x,c,mxcnt;

	if (flags&CONSISTENCY)
	{
		ggiSetGCForeground(mode.vis,0);
		ggiFillscreen(mode.vis);

		for (x=10;x<110;x++)
			CheckLine(mode.vis, 10, 10,x,110);
		for (x=10;x<110;x++)
			CheckLine(mode.vis, 10, 10,110,x);
		for (x=10;x<110;x++)
			CheckLine(mode.vis,110, 10,x,110);
		for (x=10;x<110;x++)
			CheckLine(mode.vis,110, 10, 10,x);
		for (x=10;x<110;x++)
			CheckLine(mode.vis, 10,110,x, 10);
		for (x=10;x<110;x++)
			CheckLine(mode.vis, 10,110,110,x);
		for (x=10;x<110;x++)
			CheckLine(mode.vis,110,110,x, 10);
		for (x=10;x<110;x++)
			CheckLine(mode.vis,110,110, 10,x);
	}
	if (flags&SPEED) {
		for (mxcnt=1000;mxcnt<1000000000;mxcnt*=10) {
			time_start();
			for (c=mxcnt;c>0;c--)
				nothing(mode.vis,10,10,10,10);
			time_offset();
			for (c=mxcnt;c>0;c--)
				ggiDrawLine(mode.vis,10,10,10,10);
			time_stop();
			if ( u_time+s_time > 1.0 ) break;
		}
		printf("Line   1: %f %f %f\n",mxcnt/(u_time+s_time),u_time,s_time);
		for (mxcnt=1;mxcnt<1000000000;mxcnt*=10)
		{
			time_start();
			for (c=mxcnt/10;c>0;c--)
				for (x=10;x<20;x++)
					nothing(mode.vis,10,10,19,x);
			time_offset();
			for (c=mxcnt/10;c>0;c--)
				for (x=10;x<20;x++)
					ggiDrawLine(mode.vis,10,10,19,x);
			time_stop();
			if ( u_time+s_time > 1.0 ) break;
		}
		printf("Line  10: %f %f %f\n",mxcnt/(u_time+s_time),u_time,s_time);
		for (mxcnt=1;mxcnt<1000000000;mxcnt*=10)
		{
			time_start();
			for (c=mxcnt/100;c>0;c--)
				for (x=10;x<110;x++)
					nothing(mode.vis,10,10,109,x);
			time_offset();
			for (c=mxcnt/100;c>0;c--)
				for (x=10;x<110;x++)
					ggiDrawLine(mode.vis,10,10,109,x);
			time_stop();
			if ( u_time+s_time > 1.0 ) break;
		}
		printf("Line 100: %f %f %f\n",mxcnt/(u_time+s_time),u_time,s_time);
	}
}

void
Box(void)
{
	int x,y,c,mxcnt;

	if (flags&CONSISTENCY) {
		ggiSetGCForeground(mode.vis,0);
		ggiFillscreen(mode.vis);

		for (x=10;x<110;x++) {
			for (y=10;y<110;y++) ;	/* Fixme go through all alignments */
		}
	}
	if (flags&SPEED) {
		for (mxcnt=1;mxcnt<1000000000;mxcnt*=10)	{
			time_start();
			for (c=0;c<mxcnt;c++) {
					nothing(mode.vis,10,10,1,1);
			}
			time_offset();
			for (c=0;c<mxcnt;c++) {
					ggiDrawBox(mode.vis,10,10,1,1);
			}
			time_stop();
			if ( u_time+s_time > 1.0 ) break;
		}
		printf("Box    1: %f %f %f\n",mxcnt/(u_time+s_time),u_time,s_time);
		for (mxcnt=1;mxcnt<1000000000;mxcnt*=10)
		{
			time_start();
			for (c=0;c<mxcnt;c++)
				nothing(mode.vis,10,10,10,10);
			time_offset();
			for (c=0;c<mxcnt;c++)
				ggiDrawBox(mode.vis,10,10,10,10);
			time_stop();
			if ( u_time+s_time > 1.0 ) break;
		}
		printf("Box   10: %f %f %f\n",mxcnt/(u_time+s_time),u_time,s_time);
		for (mxcnt=1;mxcnt<1000000000;mxcnt*=10)
		{
			time_start();
			for (c=0;c<mxcnt;c++)
				nothing(mode.vis,10,10,100,100);
			time_offset();
			for (c=0;c<mxcnt;c++)
				ggiDrawBox(mode.vis,10,10,100,100);
			time_stop();
			if ( u_time+s_time > 1.0 ) break;
		}
		printf("Box  100: %f %f %f\n",mxcnt/(u_time+s_time),u_time,s_time);
	}
}

struct test 
{	char *name;
	void (*func)(void);
	int active;
} tests[]=
{	{"BasicConsistency", BasicConsistency, 1},	
		/* Enabled by default to keep it from stupid tries when
		   getpixel is broken. */
	{"ColorWrap",	ColorWrap,	0},
	{"Hline",	Hline,		0},
	{"Vline",	Vline,		0},
	{"Line",	Line,		0},
	{"Box",		Box,		0},
	{ NULL,NULL,0 }
};

void
usage(const char *prog)
{
	fprintf(stderr,"Usage:\n\n"
		       "%s [-flags] <bpp> <xsize> <ysize> [<virtx> <virty>]\n\n"
		       "Default: %s --all 8 320 200 320 200\n"
		       "Supported flags are :\n",prog,prog);

	fprintf(stderr,	"-c check consistency\n"
			"-b bail out if any consistency check fails\n"
			"-s do speed tests\n"
			"-v be verbose\n"
			"-? list available tests\n");

	exit(1);
}

void
list_tests(void)
{
	int testnum;
	
	fprintf(stderr,"Available tests are :\n");
	
	for (testnum=0;tests[testnum].name;testnum++)
	{
		fprintf(stderr,"--%s\n",tests[testnum].name);
	}
}

int
parse_args(int argc,char **argv)
{
	int x,testnum;
	enum numstate { GET_DEPTH,GET_VISX,GET_VISY,GET_VIRX,GET_VIRY,GOT_ALL }
	   numstate=GET_DEPTH;
	char *prog=argv[0];

	mode.vx=mode.sx=GGI_AUTO;
	mode.vy=mode.sy=GGI_AUTO;
	mode.fx=8;mode.fy=16;	/* default font. */
	mode.type=GT_AUTO;
	
	for (x=1;x<argc;x++) {

		if ( *argv[x]>='0' && *argv[x]<='9' ) { /* a number ... */

			switch(numstate) {

				case GET_DEPTH:
					switch(atoi(argv[x])) {
						case 1:	mode.type=GT_1BIT;break;
						case 4: mode.type=GT_4BIT;break;
						case 8:	mode.type=GT_8BIT;break;
						case 15: mode.type=GT_15BIT;break;
						case 16: mode.type=GT_16BIT;break;
						case 24: mode.type=GT_24BIT;break;
						case 32: mode.type=GT_32BIT;break;
						default:
							fprintf(stderr,"%s: Invalid depth!\n\n",prog);
							usage(prog);
							return 1;
					}
					numstate++;
					break;
				case GET_VISX:
					mode.sx=mode.vx=atoi(argv[x]);
					numstate++;
					break;
				case GET_VISY:
					mode.sy=mode.vy=atoi(argv[x]);
					numstate++;
					break;
				case GET_VIRX:
					mode.vx=atoi(argv[x]);
					numstate++;
					break;
				case GET_VIRY:
					mode.vy=atoi(argv[x]);
					numstate++;
					break;
				case GOT_ALL:
					fprintf(stderr,"%s: too many dimensions !\n\n",prog);
					usage(prog);
					return 1;
			}
		} 
		else if ((*argv[x]=='t' || *argv[x]=='T') && 
			  numstate==GET_DEPTH ) {

			  switch(atoi(argv[x]+1)) {

				case 16:mode.type=GT_TEXT16;break;
				case 32:mode.type=GT_TEXT32;break;
				default:
					fprintf(stderr,"%s: Invalid text depth!\n\n",prog);
					usage(prog);
					return 1;
			  }
		}
		else if (*argv[x]=='-') {

			switch(argv[x][1]) {

				case 'c': flags|=CONSISTENCY;break;
				case 'b': flags|=BREAK_UNCONSISTENT;break;
				case 's': flags|=SPEED;break;
				case 'v': flags|=VERBOSE;break;
				case '?': list_tests();return 1;
				case '-':
					for (testnum=0;tests[testnum].name;testnum++)
					{
						if (strcmp(tests[testnum].name,argv[x]+2)==0 ||
						    strcmp("all",              argv[x]+2)==0 )
							tests[testnum].active=1;
					} break;
				default:
					fprintf(stderr,"%s: Unknown switch '%s' !\n\n",prog,argv[x]);
					usage(prog);
					return 1;
			}
		}
		else {
			fprintf(stderr,"%s: Can't parse '%s'.\n\n",prog,argv[x]);
			usage(prog);
			return 1;
		}
	}	

	printf("Using mode (%dx%d [%dx%d])\n",
		mode.sx,mode.sy,mode.vx,mode.vy);
	return 0;

}

int
setup_mode(void)
{
	int err, i, j;
	ggi_color map[256];
	const ggi_directbuffer *buf;
	const ggi_pixelformat *pixfmt;

	if ((mode.vis=ggiOpen(NULL)) == NULL) {
		fprintf(stderr,
			"unable to open default visual, exiting.\n");
		exit(1);
	}

	if (mode.type==GT_TEXT16 || mode.type==GT_TEXT32) {
		err=ggiSetTextMode(mode.vis,mode.sx,mode.sy,mode.vx,mode.vy,
					mode.fx,mode.fy,mode.type);
		printf("Text mode %dx%d (%dx%d virt), %dc%d font.\n",
			mode.sx,mode.sy,mode.vx,mode.vy,mode.fx,mode.fy);
	} else {
		err=ggiSetGraphMode(mode.vis,mode.sx,mode.sy,mode.vx,mode.vy,
					mode.type);
		printf("Graph mode %dx%d (%dx%d virt)\n",
			mode.sx,mode.sy,mode.vx,mode.vy);
	}

	if (err) {
		fprintf(stderr,"Can't set mode\n");
		return 1;
	}

	map[0].r=0xFFFF;
	map[0].g=0xFFFF;
	map[0].b=0xFFFF;
	white_pixel=ggiMapColor(mode.vis, &map[0]);
	printf("white=%d\n",white_pixel);

	pixfmt = ggiGetPixelFormat(mode.vis);

	printf("Pixel format: depth=%d, size=%d\n"
		"red_mask=%x, green_mask=%x, blue_mask=%x, alpha_mask=%x\n"
		"clut_mask=%x, fg_mask=%x, bg_mask=%x, texture_mask=%x\n\n",
		pixfmt->depth, pixfmt->size, 
		pixfmt->red_mask, pixfmt->green_mask, pixfmt->blue_mask, pixfmt->alpha_mask,
		pixfmt->clut_mask, pixfmt->fg_mask, pixfmt->bg_mask, pixfmt->texture_mask);

	for (i = 0; ; i++) {
		buf = ggiDBGetBuffer(mode.vis, i);
		
		if (buf==NULL)
			break;

		printf("Direct buffer (frame #%d):\n", buf->frame);

		printf("Mapped at read:%p, write:%p (paged %d)\n",
			buf->read, buf->write, buf->page_size);

		switch(buf->layout)
		{
			case blPixelLinearBuffer: 
				printf("Layout: Linear Pixel Buffer\n");
				printf("Stride=%d\n", buf->buffer.plb.stride);
				printf("Pixelformat: (%d/%d bits) flags=%x stdform=%x\n",
					buf->buffer.plb.pixelformat->depth,
					buf->buffer.plb.pixelformat->size,
					buf->buffer.plb.pixelformat->flags,
					buf->buffer.plb.pixelformat->stdformat);
				printf("Mask red   : %8x Shift:%2d\n",
					buf->buffer.plb.pixelformat->red_mask,
					buf->buffer.plb.pixelformat->red_shift);
				printf("Mask green : %8x Shift:%2d\n",
					buf->buffer.plb.pixelformat->green_mask,
					buf->buffer.plb.pixelformat->green_shift);
				printf("Mask blue  : %8x Shift:%2d\n",
					buf->buffer.plb.pixelformat->blue_mask,
					buf->buffer.plb.pixelformat->blue_shift);
				printf("Mask alpha : %8x Shift:%2d\n",
					buf->buffer.plb.pixelformat->alpha_mask,
					buf->buffer.plb.pixelformat->alpha_shift);
				printf("Mask clut  : %8x Shift:%2d\n",
					buf->buffer.plb.pixelformat->clut_mask,
					buf->buffer.plb.pixelformat->clut_shift);
				printf("Mask fg    : %8x Shift:%2d\n",
					buf->buffer.plb.pixelformat->fg_mask,
					buf->buffer.plb.pixelformat->fg_shift);
				printf("Mask bg    : %8x Shift:%2d\n",
					buf->buffer.plb.pixelformat->bg_mask,
					buf->buffer.plb.pixelformat->bg_shift);
				printf("Mask textur: %8x Shift:%2d\n",
					buf->buffer.plb.pixelformat->texture_mask,
					buf->buffer.plb.pixelformat->texture_shift);
				for(j=0;j<buf->buffer.plb.pixelformat->size;j++)
					printf("Bit %d: %08x\n",j,
						buf->buffer.plb.pixelformat->bitmeaning[j]);
				break;
			default: 
				printf("Layout: Unknown\n");
				continue;	/* Skip it. Don't know it. */
		}
	}

	

	return 0;
}

int
main(int argc,char **argv)
{
	int testnum;

	if (parse_args(argc,argv)) return 1;

	srandom(time(NULL));
	if (ggiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize libggi, exiting.\n",
			argv[0]);
		exit(1);
	}

	if (setup_mode()) return 2;
	
	for (testnum=0;tests[testnum].name;testnum++) {
		if (!tests[testnum].active) continue;
		TestName(tests[testnum].name);
		tests[testnum].func();
	}

	ggiClose(mode.vis);
	ggiExit();	

	return 0;
}
