/* $Id: keyboard.inc,v 1.7 1998/10/29 11:58:54 ajapted Exp $
***************************************************************************

   Linux_common/keyboard.c
   Linux keyboard handling code.

   Copyright (C) 1998  Andrew Apted  <andrew.apted@ggi-project.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   ---------------------------------------------------------------------

   USAGE:

	1. If vis->selectfd is not the keyboard file descriptor (it
	usually will be), then just #define KEYBOARD_FD(vis) to
	something appropriate.
	
	2. Define KEYBOARD_PRIV(vis) to be a variable (type: void *)
	where some private information can be stored.  Initialize the
	pointer to NULL for good measure.

	3. Open the appropriate keyboard file (e.g. a tty), and then
	call keyboard_init(vis).  This will setup the keyboard (such as
	going into MEDIUMRAW mode).  Returns 0 if successful.

	4. When keyboard data is available (e.g. by using select on
	KEYBOARD_FD(vis)), then call keyboard_handle_data(vis).  This 
	will read in the keycodes, convert them to GGI key events, and
	add them to the event queue.

	5. When finished, call keyboard_exit(vis), which will restore
	the keyboard to normal.  Then you can close the handle.  

***************************************************************************
*/

#include <unistd.h>
#include <termios.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#include <linux/kd.h>
#include <linux/vt.h>
#include <linux/keyboard.h>


typedef struct keyboard_hook
{
	struct termios old_termios;
	int old_mode;
	
	unsigned char keydown_buf[128];

} KeyboardHook;

#define KEYBOARD_HOOK(vis)  ((KeyboardHook *) KEYBOARD_PRIV(vis))

#ifndef KEYBOARD_FD
#define KEYBOARD_FD(vis)  LIBGGI_SELECT_FD(vis)
#endif


int keyboard_init(ggi_visual *vis)
{
	struct termios new;

	/* allocate keyboard hook */

	KEYBOARD_PRIV(vis) = _ggi_malloc(sizeof(KeyboardHook));

	/* Put tty into "straight through" mode.  Note that the keyboard
	 * file-descriptor has already been opened for us.
	 */

	if (tcgetattr(KEYBOARD_FD(vis), 
		      &(KEYBOARD_HOOK(vis)->old_termios)) < 0) {
		perror("L/keyboard: tcgetattr failed");
	}

	new = KEYBOARD_HOOK(vis)->old_termios;

	new.c_lflag &= ~(ICANON | ECHO | ISIG);
	new.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
	new.c_cc[VMIN]  = 0;
	new.c_cc[VTIME] = 0;

	if (tcsetattr(KEYBOARD_FD(vis), TCSANOW, &new) < 0) {
		perror("L/keyboard: tcsetattr failed");
	}

	/* Now set the tty into mediumraw mode.  Despite the name, this
	 * is really "mostly raw", with the kernel just taking care of
	 * folding long scancode sequences (e.g. E0 XX) onto single
	 * keycodes.
	 */

	if (ioctl(KEYBOARD_FD(vis), KDGKBMODE, 
		  &(KEYBOARD_HOOK(vis)->old_mode)) < 0) {

		perror("L/keyboard: couldn't get mode");
		KEYBOARD_HOOK(vis)->old_mode = K_XLATE;
	}

	if (ioctl(KEYBOARD_FD(vis), KDSKBMODE, K_MEDIUMRAW) < 0) {
		perror("L/keyboard: couldn't set mode");
	}
	
	DPRINT("L/keyboard: init OK.\n");

	return 0;
}


void keyboard_exit(ggi_visual *vis)
{
	if (KEYBOARD_FD(vis) >= 0) {

		if (ioctl(KEYBOARD_FD(vis), KDSKBMODE,
			  KEYBOARD_HOOK(vis)->old_mode) < 0) {
			perror("L/keyboard: couldn't set mode");
		}

		if (tcsetattr(KEYBOARD_FD(vis), TCSANOW, 
			  &(KEYBOARD_HOOK(vis)->old_termios)) < 0) {
			perror("L/keyboard: tcsetattr failed");
		}

		/* free keyboard hook */

		free(KEYBOARD_PRIV(vis));
	}

	DPRINT("L/keyboard: exit OK.\n");
}


void keyboard_handle_data(ggi_visual *vis)
{
	ggi_event kbd_event;

	struct kbentry entry;

	unsigned char buf[32];

	int code, shift;


	/* first get the keycode */

	if (read(KEYBOARD_FD(vis), buf, 1) < 1) {
		DPRINT("L/keyboard: Error reading keyboard.\n");
		return;
	}

	code = buf[0];

	if (code & 0x80) {
		code &= 0x7f;
		kbd_event.key.type = evKeyRelease;
		KEYBOARD_HOOK(vis)->keydown_buf[code] = 0;

	} else if (KEYBOARD_HOOK(vis)->keydown_buf[code] == 0) {
		kbd_event.key.type = evKeyPress;
		KEYBOARD_HOOK(vis)->keydown_buf[code] = 1;

	} else {
		kbd_event.key.type = evKeyRepeat;
	}

	kbd_event.key.button = code;
	
	
	/* Get the kernel's shift state.  This way is easier than
	 * keeping track of it ourselves.
	 */

	buf[0] = 6;

	if (ioctl(KEYBOARD_FD(vis), TIOCLINUX, buf) < 0) {
		DPRINT("L/keyboard: Couldn't read shift state.\n");
		entry.kb_table = 0;
	}

	shift = buf[0];

	kbd_event.key.effect = translate_shift(shift);
	

	/* Look up the keysym without modifiers, which will give us
	 * the key label (more or less).
	 */

	entry.kb_table = 0;
	entry.kb_index = code;
	
	if (ioctl(KEYBOARD_FD(vis), KDGKBENT, &entry) < 0) {
		DPRINT("L/keyboard: ioctl(KDGKBENT) failed.\n");
		return;
	}

	kbd_event.key.label = translate_label(entry.kb_value, shift);


	/* Now look up the full keysym in the kernel's table */

	entry.kb_table = shift;
	entry.kb_index = code;
	
	if (ioctl(KEYBOARD_FD(vis), KDGKBENT, &entry) < 0) {
		DPRINT("L/keyboard: ioctl(KDGKBENT) failed.\n");
		return;
	}

	switch (entry.kb_value) {
		case K_NOSUCHMAP: case K_HOLE:
			return;
	}

	kbd_event.key.sym = translate_sym(entry.kb_value, shift);


	DPRINT("KEY%s button=0x%02x shift=0x%02x sym=0x%04x label=0x%04x\n",
		(kbd_event.key.type == evKeyRelease) ? "UP" : "DOWN",
		kbd_event.key.button, kbd_event.key.effect, 
		kbd_event.key.sym,    kbd_event.key.label);


	/* Check for console switch.  Unfortunately, the kernel doesn't
	 * recognize KT_CONS when the keyboard is in RAW or MEDIUMRAW
	 * mode, so *we* have to.  Sigh.
	 */
	
	if (KTYP(entry.kb_value) == KT_CONS) {

		int new_cons = 1+KVAL(entry.kb_value);

		DPRINT("Switching to console %d.\n", new_cons);
		
		if (ioctl(KEYBOARD_FD(vis), VT_ACTIVATE, new_cons) < 0) {
			  perror("ioctl(VT_ACTIVATE)");
		}
		
		/* No need to wait till it's active, the vt_process
		 * signal handler does that for us.
		 */
		 
		return;
	}
	

	/* finally queue the key event */

	kbd_event.key.size   = sizeof(ggi_key_event);
	kbd_event.key.origin = EV_ORIGIN_NONE;
	kbd_event.key.target = EV_TARGET_NONE;

	EV_TIMESTAMP(& kbd_event);

	_ggiEvQueueAdd(vis, &kbd_event);
}
