/* $Id: mode.c,v 1.26 1998/10/18 15:52:52 becka Exp $
***************************************************************************

   Display-monotext: mode management

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <ggi/internal/ggi-dl.h>

#include "monotext.h"


static void _GGIfreedbs(ggi_visual *vis) 
{
	int i;

	for (i=LIBGGI_PRIVLIST(vis)->num-1; i >= 0; i--) {
		_ggi_db_free(LIBGGI_PRIVBUFS(vis)[i]);
		_ggi_db_del_buffer(LIBGGI_PRIVLIST(vis), i);
	}
}

int GGI_monotext_getapi(ggi_visual *vis, int num, char *apiname, char *arguments)
{
	strcpy(arguments, "");

	switch(num) {

	case 0:
		strcpy(apiname, "display-monotext");
		return 0;

	case 1:
		strcpy(apiname, "generic-stubs");
		return 0;

	case 2:
		strcpy(apiname, "generic-linear-8");
		return 0;

	case 3: strcpy(apiname, "generic-color");
		return 0;
	}

	return -1;
}


/*
 * Attempt to get the default framebuffer.
 */

static int do_dbstuff(ggi_visual *vis)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);

	mt->fb_size = LIBGGI_FB_SIZE(LIBGGI_MODE(vis));
	mt->fb_ptr  = _ggi_malloc(mt->fb_size);
	
	DPRINT("display-monotext: fb=%p size=%d\n", 
		mt->fb_ptr, mt->fb_size);

	if (mt->fb_ptr == NULL) {
		fprintf(stderr, "display-monotext: Out of memory.\n");
		return -1;
	}

	/* set up direct buffers */

	_ggi_db_add_buffer(LIBGGI_PRIVLIST(vis), _ggi_db_get_new());

	LIBGGI_PRIVBUFS(vis)[0]->type  = GGI_DB_NORMAL | GGI_DB_SIMPLE_PLB;
	LIBGGI_PRIVBUFS(vis)[0]->frame = 0;
	LIBGGI_PRIVBUFS(vis)[0]->read  = mt->fb_ptr;
	LIBGGI_PRIVBUFS(vis)[0]->write = mt->fb_ptr;
	LIBGGI_PRIVBUFS(vis)[0]->layout = blPixelLinearBuffer;
	LIBGGI_PRIVBUFS(vis)[0]->buffer.plb.stride = 
		(LIBGGI_VIRTX(vis) * GT_SIZE(LIBGGI_GT(vis)) + 7) / 8;
	LIBGGI_PRIVBUFS(vis)[0]->buffer.plb.pixelformat = LIBGGI_PIXFMT(vis);

	/* Set up palette */

	if (GT_SCHEME(LIBGGI_GT(vis)) == GT_PALETTE) {
		vis->palette = _ggi_malloc((1 << GT_DEPTH(LIBGGI_GT(vis)))*
					sizeof(ggi_color));
	}

	return 0;
}

static int do_setmode(ggi_visual *vis)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);

	char libname[256], libargs[256];

	int err, id;


	_GGIfreedbs(vis);

	if ((err = do_dbstuff(vis)) != 0) {
		return err;
	}
	

	/* load libraries */

	for(id=1; GGI_monotext_getapi(vis, id, libname, libargs) == 0; id++) {

		if (_ggiOpenDL(vis, libname, libargs, NULL) == NULL) {

			fprintf(stderr, "display-monotext: Error opening "
				" %s (%s) library.\n", libname, libargs);
			return -1;
		}

		DPRINT("Success in loading %s (%s)\n", libname, libargs);
	}


	/* Backup the current 2D operations, and override them with our
	 * own (which then call on these backups to do the work).
	 */

	mt->mem_opdraw = _ggi_malloc(sizeof(struct ggi_visual_opdraw));

	*mt->mem_opdraw = *vis->opdraw;

	vis->opdraw->drawpixel_nc=GGI_monotext_drawpixel_nc;
	vis->opdraw->drawpixel=GGI_monotext_drawpixel;
	vis->opdraw->drawhline_nc=GGI_monotext_drawhline_nc;
	vis->opdraw->drawhline=GGI_monotext_drawhline;
	vis->opdraw->drawvline_nc=GGI_monotext_drawvline_nc;
	vis->opdraw->drawvline=GGI_monotext_drawvline;
	vis->opdraw->drawline=GGI_monotext_drawline;

	vis->opdraw->putc=GGI_monotext_putc;
	vis->opdraw->putpixel_nc=GGI_monotext_putpixel_nc;
	vis->opdraw->putpixel=GGI_monotext_putpixel;
	vis->opdraw->puthline=GGI_monotext_puthline;
	vis->opdraw->putvline=GGI_monotext_putvline;
	vis->opdraw->putbox=GGI_monotext_putbox;

	vis->opdraw->drawbox=GGI_monotext_drawbox;
	vis->opdraw->copybox=GGI_monotext_copybox;
	vis->opdraw->crossblit=GGI_monotext_crossblit;
	vis->opdraw->fillscreen=GGI_monotext_fillscreen;

	vis->opdraw->setorigin=GGI_monotext_setorigin;
	vis->opcolor->setpalvec=GGI_monotext_setpalvec;
	
	ggiIndicateChange(vis, GGI_CHG_APILIST);

	DPRINT("display-monotext: Attempting to setmode on parent "
		"visual...\n");

	return _ggi_monotextOpen(vis);
}

int GGI_monotext_setmode(ggi_visual *vis, ggi_mode *mode)
{ 
	int err;

	if ((vis == NULL) || (mode == NULL) || (LIBGGI_MODE(vis) == NULL)) {
		DPRINT("display-monotext: vis/mode == NULL\n");
		return -1;
	}
	
	DPRINT("display-monotext: setmode %dx%d (gt=%d)\n",
		mode->visible.x, mode->visible.y, mode->graphtype);

	if ((err = GGI_monotext_checkmode(vis, mode)) != 0) {
		return err;
	}

	_ggiZapMode(vis, ~GGI_DL_OPDISPLAY);

	memcpy(LIBGGI_MODE(vis), mode, sizeof(ggi_mode));
	
	if ((err = do_setmode(vis)) != 0) {
		DPRINT("display-monotext: setmode failed (%d).\n", err);
		return err;
	}

	DPRINT("display-monotext: setmode succeeded.\n", vis, mode);

	return 0;
}

static int calc_squish(MonotextHook *mt, ggi_mode *mode, 
			int target_width, int target_height)
{
	int sq_x, sq_y;

	target_width  *= mt->accuracy.x;
	target_height *= mt->accuracy.y;

#if 0   /* Preliminary mode-improvement code */
	while ((mode->visible.x % target_width)  != 0) mode->visible.x++;
	while ((mode->visible.y % target_height) != 0) mode->visible.y++;
	mode->virt.x = mode->visible.x;
	mode->virt.y = mode->visible.y;
#endif

	if (((mode->visible.x % target_width)  != 0) ||
	    ((mode->visible.y % target_height) != 0)) {
	
		DPRINT("display-monotext: visible size is not a multible "
			"of the target size.\n");
		return -1;
	}
	
	sq_x = mode->visible.x / target_width;
	sq_y = mode->visible.y / target_height;

	if ((sq_x <= 0) || (sq_y <= 0)) {
	
		DPRINT("display-monotext: visible size is not a multible "
			"of the target size.\n");
		return -1;
	}

	mt->squish.x = sq_x;
	mt->squish.y = sq_y;

	return 0;
}

int GGI_monotext_checkmode(ggi_visual *vis, ggi_mode *mode)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);

	int target_width  = 80;  /* query target ??? */
	int target_height = 25;


	if ((vis == NULL) || (mode == NULL)) {
		DPRINT("display-monotext: vis/mode == NULL\n");
		return -1;
	}

	DPRINT("display-monotext: checkmode %dx%d (gt=%d)\n",
		mode->visible.x, mode->visible.y, mode->graphtype);


	/* Handle GGI_AUTO */

	if (mode->graphtype == GGI_AUTO) {

		mode->graphtype = GT_8BIT;
	}

	if ((mode->visible.x == GGI_AUTO) &&
	    (mode->virt.x    == GGI_AUTO)) {

	    mode->visible.x = mode->virt.x = target_width * mt->accuracy.x;
	    
	} else if (mode->virt.x == GGI_AUTO) {

	    mode->virt.x = mode->visible.x;

	} else if ((mode->visible.x == GGI_AUTO) ||
		   (mode->visible.x > mode->virt.x)) {

	    mode->visible.x = mode->virt.x;
	}

	if ((mode->visible.y == GGI_AUTO) &&
	    (mode->virt.y == GGI_AUTO)) {

	    mode->visible.y = mode->virt.y = target_height * mt->accuracy.y;
	    
	} else if (mode->virt.y == GGI_AUTO) {

	    mode->virt.y = mode->visible.y;

	} else if ((mode->visible.y == GGI_AUTO) ||
		   (mode->visible.y > mode->virt.y)) {

	    mode->visible.y = mode->virt.y;
	}

	if (mode->dpp.x == GGI_AUTO) {
		mode->dpp.x = 1;
	}

	if (mode->dpp.y == GGI_AUTO) {
		mode->dpp.y = mode->dpp.x;
	}

	if ((sint32) mode->frames == GGI_AUTO) {
		mode->frames = 1;
	}

	/* check stuff */

	if (mode->graphtype != GT_8BIT) {

		DPRINT("display-monotext: checkmode: graphtype "
			"not supported.\n");
		return -1;
	}

	if ((mode->visible.x != mode->virt.x) ||
	    (mode->visible.y != mode->virt.y)) {

		DPRINT("display-monotext: checkmode: "
			"Larger virtual area not yet supported.\n");
		return -1;
	}

	if (calc_squish(mt, mode, target_width, target_height) < 0) {
		return -1;
	}

	if (((mode->visible.x / mt->accuracy.x / mt->squish.x) != 
		target_width) || 
	    ((mode->visible.y / mt->accuracy.y / mt->squish.y) != 
		target_height)) {

		DPRINT("display-monotext: checkmode: "
			"visible size not supported.");
		return -1;
	}

	return 0;
}

int GGI_monotext_getmode(ggi_visual *vis, ggi_mode *mode)
{
	DPRINT("display-monotext: getmode.\n");

	memcpy(mode, LIBGGI_MODE(vis), sizeof(ggi_mode));

	return 0;
}

int GGI_monotext_setflags(ggi_visual *vis, ggi_flags flags)
{
	LIBGGI_FLAGS(vis) = flags;

	return ggiFlush(vis);
}

int GGI_monotext_flush(ggi_visual *vis, int tryflag)
{
	MonotextHook *mt = LIBGGI_PRIVATE(vis);
	int err;

	if ((err = _ggi_monotextFlush(vis)) < 0) {
		return err;
	}

	return _ggiInternFlush(mt->parent, tryflag);
}
