/* $Id: mode.c,v 1.17 1998/10/18 15:52:55 becka Exp $
***************************************************************************

   Display-palemu: mode management

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "palemu.h"

#include "../common/pixfmt-setup.inc"
#include "../common/gt-auto.inc"


static void _GGI_palemu_freedbs(ggi_visual *vis) 
{
	int i;

	for (i=LIBGGI_PRIVLIST(vis)->num-1; i >= 0; i--) {
		_ggi_db_free(LIBGGI_PRIVBUFS(vis)[i]);
		_ggi_db_del_buffer(LIBGGI_PRIVLIST(vis), i);
	}
}

int GGI_palemu_getapi(ggi_visual *vis, int num, char *apiname, char *arguments)
{
	strcpy(arguments, "");

	switch(num) {

	case 0: strcpy(apiname, "display-palemu");
		return 0;

	case 1: strcpy(apiname, "generic-stubs");
		return 0;

	case 2: sprintf(apiname, "generic-linear-%d%s",
			GT_DEPTH(LIBGGI_GT(vis)),
			(LIBGGI_GT(vis) & GT_SUB_HIGHBIT_RIGHT) ? "-r" : "");
		return 0;

	case 3: strcpy(apiname, "generic-color");
		return 0;
	}

	return -1;
}


/*
 * Attempt to get the default framebuffer.
 */

static int do_dbstuff(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	int i;


	/* allocate memory */

	ph->frame_size = LIBGGI_FB_SIZE(LIBGGI_MODE(vis));
	ph->fb_size = ph->frame_size * LIBGGI_MODE(vis)->frames;
	ph->fb_ptr  = malloc(ph->fb_size);
	
	DPRINT_MODE("display-palemu: fb=%p size=%d frame=%d\n", 
		ph->fb_ptr, ph->fb_size, ph->frame_size);

	if (ph->fb_ptr == NULL) {
		fprintf(stderr, "display-palemu: Out of memory.\n");
		return -1;
	}


        /* clear all frames */

	memset(ph->fb_ptr, 0, ph->fb_size);


	/* set up pixel format */

	setup_pixfmt(LIBGGI_PIXFMT(vis), LIBGGI_GT(vis));


	/* set up direct buffers */

	for (i=0; i < LIBGGI_MODE(vis)->frames; i++) {

		ggi_directbuffer *buf;
		
		_ggi_db_add_buffer(LIBGGI_PRIVLIST(vis), _ggi_db_get_new());

		buf = LIBGGI_PRIVBUFS(vis)[i];

		buf->frame = i;
		buf->type  = GGI_DB_NORMAL | GGI_DB_SIMPLE_PLB;
		buf->read  = (char *) ph->fb_ptr + i * ph->frame_size;
		buf->write = buf->read;
		buf->layout = blPixelLinearBuffer;

		buf->buffer.plb.stride = 
			(LIBGGI_VIRTX(vis) * GT_SIZE(LIBGGI_GT(vis)) + 7) / 8;
		buf->buffer.plb.pixelformat = LIBGGI_PIXFMT(vis);
	}


	/* set up palette */

	if (GT_SCHEME(LIBGGI_GT(vis)) == GT_PALETTE) {
		vis->palette = _ggi_malloc((1 << GT_DEPTH(LIBGGI_GT(vis))) *
					sizeof(ggi_color));
	}

	return 0;
}

static int do_setmode(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	char libname[256], libargs[256];

	int err, id;


	_GGI_palemu_freedbs(vis);

	if ((err = do_dbstuff(vis)) != 0) {
		return err;
	}
	

	/* load libraries */

	for(id=1; GGI_palemu_getapi(vis, id, libname, libargs) == 0; id++) {

		if (_ggiOpenDL(vis, libname, libargs, NULL) == NULL) {

			fprintf(stderr, "display-palemu: Error opening "
				" %s (%s) library.\n", libname, libargs);
			return -1;
		}

		DPRINT("Success in loading %s (%s)\n", libname, libargs);
	}


	/* Backup the current 2D operations, and override them with our
	 * own (which then call on these backups to do the work).
	 */

	ph->mem_opdraw = _ggi_malloc(sizeof(struct ggi_visual_opdraw));

	*ph->mem_opdraw = *vis->opdraw;

	vis->opdraw->drawpixel_nc=GGI_palemu_drawpixel_nc;
	vis->opdraw->drawpixel=GGI_palemu_drawpixel;
	vis->opdraw->drawhline_nc=GGI_palemu_drawhline_nc;
	vis->opdraw->drawhline=GGI_palemu_drawhline;
	vis->opdraw->drawvline_nc=GGI_palemu_drawvline_nc;
	vis->opdraw->drawvline=GGI_palemu_drawvline;
	vis->opdraw->drawline=GGI_palemu_drawline;

	vis->opdraw->putc=GGI_palemu_putc;
	vis->opdraw->putpixel_nc=GGI_palemu_putpixel_nc;
	vis->opdraw->putpixel=GGI_palemu_putpixel;
	vis->opdraw->puthline=GGI_palemu_puthline;
	vis->opdraw->putvline=GGI_palemu_putvline;
	vis->opdraw->putbox=GGI_palemu_putbox;

	vis->opdraw->drawbox=GGI_palemu_drawbox;
	vis->opdraw->copybox=GGI_palemu_copybox;
	vis->opdraw->crossblit=GGI_palemu_crossblit;
	vis->opdraw->fillscreen=GGI_palemu_fillscreen;

	vis->opdraw->setorigin=GGI_palemu_setorigin;
	vis->opcolor->setpalvec=GGI_palemu_setpalvec;

	vis->opdraw->setreadframe=GGI_palemu_setreadframe;
	vis->opdraw->setwriteframe=GGI_palemu_setwriteframe;
	vis->opdraw->setdisplayframe=GGI_palemu_setdisplayframe;

	ggiIndicateChange(vis, GGI_CHG_APILIST);


	/* set initial frames */

	ph->mem_opdraw->setreadframe(vis, 0);
	ph->mem_opdraw->setwriteframe(vis, 0);

	return 0;
}

int GGI_palemu_setmode(ggi_visual *vis, ggi_mode *mode)
{ 
	PalemuHook *ph = PALEMU_PRIV(vis);

	int err;

	if ((vis == NULL) || (mode == NULL) || (LIBGGI_MODE(vis) == NULL)) {
		DPRINT_MODE("display-palemu: vis/mode == NULL\n");
		return -1;
	}
	

	DPRINT_MODE("display-palemu: setmode %dx%d#%dx%dF%d[0x%02x]\n",
			mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);

	MANSYNC_ignore(vis);

	if ((err = ggiCheckMode(vis, mode)) != 0) {
		return err;
	}

	_ggiZapMode(vis, ~GGI_DL_OPDISPLAY);

	*LIBGGI_MODE(vis) = *mode;
	
	ph->mode.visible = mode->visible;
	ph->mode.virt    = mode->virt;
	ph->mode.dpp     = mode->dpp;
	ph->mode.frames  = 1;

	if ((err = do_setmode(vis)) != 0) {
		DPRINT_MODE("display-palemu: setmode failed (%d).\n", err);
		return err;
	}

	DPRINT_MODE("display-palemu: Attempting to setmode on parent "
		"visual...\n");

	if ((err = _ggi_palemu_Open(vis)) != 0) {
		return err;
	}
	
	MANSYNC_cont(vis);

	DPRINT_MODE("display-palemu: setmode succeeded.\n");

	return 0;
}

int GGI_palemu_resetmode(ggi_visual *vis)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	DPRINT("display-palemu: GGIresetmode(%p)\n", vis);

	if (ph->fb_ptr != NULL) {

		_ggi_palemu_Close(vis);

		_GGI_palemu_freedbs(vis);

		free(ph->fb_ptr);
		ph->fb_ptr = NULL;
	}

	return 0;
}

int GGI_palemu_checkmode(ggi_visual *vis, ggi_mode *mode)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	ggi_mode par_mode;

	int err = 0;


	if ((vis == NULL) || (mode == NULL)) {
		DPRINT_MODE("display-palemu: vis/mode == NULL\n");
		return -1;
	}

	DPRINT_MODE("display-palemu: checkmode %dx%d#%dx%dF%d[0x%02x]\n",
			mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);


	/* Handle graphtype */

	if (GT_SCHEME(mode->graphtype) == GT_AUTO) {
		GT_SETSCHEME(mode->graphtype, GT_PALETTE);
	}

	mode->graphtype = _GGIhandle_gtauto(mode->graphtype);

	if (GT_SCHEME(mode->graphtype) != GT_PALETTE) {
		GT_SETSCHEME(mode->graphtype, GT_PALETTE);
		err--;
	}

	if (GT_DEPTH(mode->graphtype) > 8) {
		GT_SETDEPTH(mode->graphtype, 8);
		err--;
	}

	if (GT_SIZE(mode->graphtype) != GT_DEPTH(mode->graphtype)) {
		GT_SETSIZE(mode->graphtype, GT_DEPTH(mode->graphtype));
		err--;
	}


	/* Handle geometry */
	
	if (mode->visible.x == GGI_AUTO) {
		mode->visible.x = ph->mode.visible.x;
	}
	if (mode->visible.y == GGI_AUTO) {
		mode->visible.y = ph->mode.visible.y;
	}
	if (mode->virt.x == GGI_AUTO) {
		mode->virt.x = ph->mode.virt.x;
	}
	if (mode->virt.y == GGI_AUTO) {
		mode->virt.y = ph->mode.virt.y;
	}
	if (mode->dpp.x == GGI_AUTO) {
		mode->dpp.x = ph->mode.dpp.x;
	}
	if (mode->dpp.y == GGI_AUTO) {
		mode->dpp.y = ph->mode.dpp.y;
	}
	if ((sint32) mode->frames == GGI_AUTO) {
		mode->frames = 1;
	}


	/* Now check mode against the parent target (letting the parent
	 * target handle any remaining GT_AUTO and GGI_AUTO values).
	 */

	par_mode = *mode;

	par_mode.graphtype = ph->mode.graphtype;

	if (ggiCheckMode(ph->parent, &par_mode) != 0) {
		err--;
	}

	mode->visible = par_mode.visible;
	mode->virt    = par_mode.virt;
	mode->dpp     = par_mode.dpp;

	/* When the parent is palettized, we must limit the
	 * resulting depth to be <= the parent depth.
	 */

	if ((GT_SCHEME(par_mode.graphtype) == GT_PALETTE) &&
	    (GT_DEPTH(par_mode.graphtype) < 
	     GT_DEPTH(mode->graphtype))) {

		GT_SETDEPTH(mode->graphtype, 
			GT_DEPTH(par_mode.graphtype));
		GT_SETSIZE(mode->graphtype, 
			GT_DEPTH(par_mode.graphtype));
		err--;
	}

	
	DPRINT_MODE("display-palemu: result %d %dx%d#%dx%dF%d[0x%02x]\n",
			err, mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);
	return err;
}

int GGI_palemu_getmode(ggi_visual *vis, ggi_mode *mode)
{
	if ((vis == NULL) || (mode == NULL) || (LIBGGI_MODE(vis) == NULL)) {
		DPRINT_MODE("display-palemu: vis/mode == NULL\n");
		return -1;
	}
	
	DPRINT_MODE("display-palemu: getmode.\n");

	memcpy(mode, LIBGGI_MODE(vis), sizeof(ggi_mode));

	return 0;
}

int GGI_palemu_setflags(ggi_visual *vis, ggi_flags flags)
{
	LIBGGI_FLAGS(vis) = flags;

	MANSYNC_SETFLAGS(vis, flags);

	return 0;
}

int GGI_palemu_flush(ggi_visual *vis, int tryflag)
{
	PalemuHook *ph = PALEMU_PRIV(vis);

	int err;

	MANSYNC_ignore(vis);

	_ggiLock(ph->flush_lock);

	err = _ggi_palemu_Flush(vis);

	if (! err) {
		err = _ggiInternFlush(ph->parent, tryflag);
	}

	_ggiUnlock(ph->flush_lock);

	MANSYNC_cont(vis);

	return err;
}
