/* $Id: kbd.c,v 1.8 1998/10/29 11:58:58 ajapted Exp $
***************************************************************************

   Display-SUID

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <fcntl.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#include <linux/kd.h>
#include <linux/vt.h>

#include "debug.h"
#include <ggi/internal/ggi-dl.h>


extern int GGIevqueueadd(ggi_visual *vis, ggi_event *ev);  /* !!! */

static int kbd_fd = -1;
static int kbd_mode;
static struct termios kbd_termios;

#include "../Linux_common/key_trans.inc"

void fbdev_init_keyboard(ggi_visual *vis, int fd)
{
	struct termios new;

	kbd_fd = fd;

	/* Put tty into "straight through" mode.  Note that the keyboard
	 * file-descriptor 'kbd_fd' has already been opened for us
	 * (by mode.c).
	 */

	tcgetattr(kbd_fd, &kbd_termios);

	new = kbd_termios;

	new.c_lflag &= ~(ICANON | ECHO | ISIG);
	new.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
	new.c_cc[VMIN]  = 0;
	new.c_cc[VTIME] = 0;

	tcsetattr(kbd_fd, TCSANOW, &new);

	/* Now set the tty into mediumraw mode.  Despite the name, this
	 * is really "mostly raw", with the kernel just taking care of
	 * folding long scancode sequences (e.g. E0 XX) onto single
	 * keycodes.
	 */

	if (ioctl(kbd_fd, KDGKBMODE, &kbd_mode) < 0) {
		DPRINT("display-fbdev: couldn't get keyboard mode.\n");
		kbd_mode = K_XLATE;
	}

	ioctl(kbd_fd, KDSKBMODE, K_MEDIUMRAW);
	
	DPRINT("display-fbdev: kbd init OK.\n");
}


void fbdev_exit_keyboard(ggi_visual *vis)
{
	if (kbd_fd >= 0) {
		ioctl(kbd_fd, KDSKBMODE, kbd_mode);
		tcsetattr(kbd_fd, TCSANOW, &kbd_termios);
	}

	DPRINT("display-fbdev: kbd exit OK.\n");
}


void fbdev_handle_kbd_data(ggi_visual *vis)
{
	ggi_event kbd_event;

	struct kbentry entry;

	unsigned char buf[32];


	/* first get the keycode */

	if (read(kbd_fd, buf, 1) < 1) {
		DPRINT("display-fbdev: Error reading keyboard.\n");
		return;
	}

	kbd_event.key.button = buf[0];

	if (kbd_event.key.button & 0x80) {
		kbd_event.key.type = evKeyRelease;
		kbd_event.key.button &= 0x7f;
	} else {
		kbd_event.key.type = evKeyPress;
	}
	
	
	/* Get the kernel's shift state.  This is easier than keeping
	 * track of it ourselves.
	 */

	buf[0] = 6;

	if (ioctl(kbd_fd, TIOCLINUX, buf) < 0) {
		DPRINT("display-fbdev: Couldn't read shift state.\n");
		entry.kb_table = 0;
	}

	kbd_event.key.effect = buf[0];
	

	/* look up the keysym in the kernel's table */

	entry.kb_table = kbd_event.key.effect;
	entry.kb_index = kbd_event.key.button;
	
	if (ioctl(kbd_fd, KDGKBENT, &entry) < 0) {
		DPRINT("display-fbdev: ioctl(KDGKBENT) failed.\n");
		return;
	}

	switch (entry.kb_value) {
		case K_NOSUCHMAP: case K_VOID:
			return;
	}

	kbd_event.key.sym = translate_sym(entry.kb_value, 0);
	
	DPRINT("KEYBOARD 0x%04x\n", kbd_event.key.sym);


	/* Check for console switch.  Unfortunately, the kernel doesn't
	 * recognize KT_CONS when the keyboard is in RAW or MEDIUMRAW
	 * mode, so *we* have to.  Sigh.
	 */
	
	if (KTYP(entry.kb_value) == KT_CONS) {
		ioctl(kbd_fd, VT_ACTIVATE, 1+KVAL(entry.kb_value));
		
		/* No need to wait till it's active, the vt_process
		 * signal handler does that for us.
		 */
		 
		return;
	}
	

	/* finally queue the key event */

	kbd_event.key.size   = sizeof(ggi_key_event);
	kbd_event.key.origin = EV_ORIGIN_NONE;
	kbd_event.key.target = EV_TARGET_NONE;

	EV_TIMESTAMP(& kbd_event);

	GGIevqueueadd(vis, &kbd_event);
}
