/* $Id: mode.c,v 1.27 1998/10/18 15:53:09 becka Exp $
***************************************************************************

   Display-trueemu : mode management

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include "trueemu.h"

#include "../common/pixfmt-setup.inc"
#include "../common/gt-auto.inc"


static void _GGI_trueemu_freedbs(ggi_visual *vis) 
{
	int i;

	for (i=LIBGGI_PRIVLIST(vis)->num-1; i >= 0; i--) {
		_ggi_db_free(LIBGGI_PRIVBUFS(vis)[i]);
		_ggi_db_del_buffer(LIBGGI_PRIVLIST(vis), i);
	}
}

int GGI_trueemu_getapi(ggi_visual *vis, int num, char *apiname, char *arguments)
{
	strcpy(arguments, "");

	switch(num) {

	case 0: strcpy(apiname, "display-trueemu");
		return 0;

	case 1: strcpy(apiname, "generic-stubs");
		return 0;

	case 2: sprintf(apiname, "generic-linear-%d%s",
			GT_SIZE(LIBGGI_GT(vis)),
			(LIBGGI_GT(vis) & GT_SUB_HIGHBIT_RIGHT) ? "-r" : "");
		return 0;

	case 3: strcpy(apiname, "generic-color");
		return 0;
	}

	return -1;
}

static int do_dbstuff(ggi_visual *vis)
{
	TrueemuHook *th = TRUEEMU_PRIV(vis);

	int i;


	/* allocate memory */

	th->frame_size = LIBGGI_FB_SIZE(LIBGGI_MODE(vis));
	th->fb_size = th->frame_size * LIBGGI_MODE(vis)->frames;
	th->fb_ptr  = malloc(th->fb_size);
	
	DPRINT("display-trueemu: fb=%p size=%d frame=%d\n", 
		th->fb_ptr, th->fb_size, th->frame_size);

	if (th->fb_ptr == NULL) {
		fprintf(stderr, "display-trueeemu: Out of memory.\n");
		return -ENOMEM;
	}

	
        /* clear all frames */

	memset(th->fb_ptr, 0, th->fb_size);


	/* set up pixel format */

	setup_pixfmt(LIBGGI_PIXFMT(vis), LIBGGI_GT(vis));

	
	/* set up direct buffers */

        for (i=0; i < LIBGGI_MODE(vis)->frames; i++) {

		ggi_directbuffer *buf;

		_ggi_db_add_buffer(LIBGGI_PRIVLIST(vis), _ggi_db_get_new());

		buf = LIBGGI_PRIVBUFS(vis)[i];

		buf->frame = i;
		buf->type  = GGI_DB_NORMAL | GGI_DB_SIMPLE_PLB;
		buf->read  = (char *) th->fb_ptr + i * th->frame_size;
		buf->write = buf->read;
		buf->layout = blPixelLinearBuffer;

		buf->buffer.plb.stride = 
			(LIBGGI_VIRTX(vis) * GT_SIZE(LIBGGI_GT(vis)) + 7) / 8;
		buf->buffer.plb.pixelformat = LIBGGI_PIXFMT(vis);
	}

	return 0;
}

static int do_setmode(ggi_visual *vis)
{
	TrueemuHook *th = TRUEEMU_PRIV(vis);

	char libname[256], libargs[256];

	int err, id;


	_GGI_trueemu_freedbs(vis);

	if ((err = do_dbstuff(vis)) != 0) {
		return err;
	}
	

	/* load libraries */

	for(id=1; GGI_trueemu_getapi(vis, id, libname, libargs) == 0; id++) {

		if (_ggiOpenDL(vis, libname, libargs, NULL) == NULL) {

			fprintf(stderr,"display-tryeeny: Error opening the "
				"%s (%s) library.\n", libname, libargs);
			return -1;
		}

		DPRINT("Success in loading %s (%s)\n", libname, libargs);
	}


	/* Backup the current 2D operations, and override them with our
	 * own (which then call on these backups to do the work).
	 */

	th->mem_opdraw = _ggi_malloc(sizeof(struct ggi_visual_opdraw));

	*th->mem_opdraw = *vis->opdraw;

	vis->opdraw->drawpixel_nc=GGI_trueemu_drawpixel_nc;
	vis->opdraw->drawpixel=GGI_trueemu_drawpixel;
	vis->opdraw->drawhline_nc=GGI_trueemu_drawhline_nc;
	vis->opdraw->drawhline=GGI_trueemu_drawhline;
	vis->opdraw->drawvline_nc=GGI_trueemu_drawvline_nc;
	vis->opdraw->drawvline=GGI_trueemu_drawvline;
	vis->opdraw->drawline=GGI_trueemu_drawline;

	vis->opdraw->putc=GGI_trueemu_putc;
	vis->opdraw->putpixel_nc=GGI_trueemu_putpixel_nc;
	vis->opdraw->putpixel=GGI_trueemu_putpixel;
	vis->opdraw->puthline=GGI_trueemu_puthline;
	vis->opdraw->putvline=GGI_trueemu_putvline;
	vis->opdraw->putbox=GGI_trueemu_putbox;

	vis->opdraw->drawbox=GGI_trueemu_drawbox;
	vis->opdraw->copybox=GGI_trueemu_copybox;
	vis->opdraw->crossblit=GGI_trueemu_crossblit;
	vis->opdraw->fillscreen=GGI_trueemu_fillscreen;
	vis->opdraw->setorigin=GGI_trueemu_setorigin;

	vis->opdraw->setreadframe=GGI_trueemu_setreadframe;
	vis->opdraw->setwriteframe=GGI_trueemu_setwriteframe;
	vis->opdraw->setdisplayframe=GGI_trueemu_setdisplayframe;

	ggiIndicateChange(vis, GGI_CHG_APILIST);


	/* set initial frames */

	th->mem_opdraw->setreadframe(vis, 0);
	th->mem_opdraw->setwriteframe(vis, 0);

	return 0;
}

int GGI_trueemu_setmode(ggi_visual *vis, ggi_mode *mode)
{ 
	TrueemuHook *th = TRUEEMU_PRIV(vis);

	int err;

	if ((vis == NULL) || (mode == NULL) || (LIBGGI_MODE(vis) == NULL)) {
		DPRINT("display-trueemu: vis/mode == NULL\n");
		return -1;
	}
	

	DPRINT_MODE("display-trueemu: setmode %dx%d#%dx%dF%d[0x%02x]\n",
			mode->visible.x, mode->visible.y,
			mode->virt.x, mode->virt.y, 
			mode->frames, mode->graphtype);

	MANSYNC_ignore(vis);

	if ((err = ggiCheckMode(vis, mode)) != 0) {
		return err;
	}

        _ggiZapMode(vis, ~GGI_DL_OPDISPLAY);

        *LIBGGI_MODE(vis) = *mode;

        th->mode.visible = mode->visible;
        th->mode.virt    = mode->virt;
        th->mode.dpp     = mode->dpp;
        th->mode.frames  = 1;

	if ((err = do_setmode(vis)) != 0) {
		DPRINT_MODE("display-trueemu: setmode failed (%d).\n", err);
		return err;
	}

	DPRINT_MODE("display-trueemu: Attempting to setmode on parent "
		"visual...\n");

	if ((err = _ggi_trueemu_Open(vis)) != 0) {
		return err;
	}

	MANSYNC_cont(vis);

	DPRINT("display-trueemu: setmode succeeded.\n");

	return 0;
}

int GGI_trueemu_checkmode(ggi_visual *vis, ggi_mode *mode)
{
	TrueemuHook *th = TRUEEMU_PRIV(vis);

	ggi_mode par_mode;

	int err = 0;

	
	if ((vis == NULL) || (mode == NULL)) {
		DPRINT("display-trueemu: vis/mode == NULL\n");
		return -1;
	}

        DPRINT_MODE("display-trueemu: checkmode %dx%d#%dx%dF%d[0x%02x]\n",
                        mode->visible.x, mode->visible.y,
                        mode->virt.x, mode->virt.y,
                        mode->frames, mode->graphtype);


	/* Handle graphtype */

	if (GT_SCHEME(mode->graphtype) == GT_AUTO) {
		GT_SETSCHEME(mode->graphtype, GT_TRUECOLOR);
	}

	mode->graphtype = _GGIhandle_gtauto(mode->graphtype);

	if (GT_SCHEME(mode->graphtype) != GT_TRUECOLOR) {
		GT_SETSCHEME(mode->graphtype, GT_TRUECOLOR);
		err--;
	}

	if (GT_DEPTH(mode->graphtype) != 24) {
		GT_SETDEPTH(mode->graphtype, 24);
		err--;
	}

	if ((GT_SIZE(mode->graphtype) != GT_DEPTH(mode->graphtype)) &&
	    (GT_SIZE(mode->graphtype) != 32)) {
		GT_SETSIZE(mode->graphtype, GT_DEPTH(mode->graphtype));
		err--;
	}

	
	/* Handle geometry */

	if (mode->visible.x == GGI_AUTO) {
		mode->visible.x = th->mode.visible.x;
	}
	if (mode->visible.y == GGI_AUTO) {
		mode->visible.y = th->mode.visible.y;
	}
	if (mode->virt.x == GGI_AUTO) {
		mode->virt.x = th->mode.virt.x;
	}
	if (mode->virt.y == GGI_AUTO) {
		mode->virt.y = th->mode.virt.y;
	}
	if (mode->dpp.x == GGI_AUTO) {
		mode->dpp.x = th->mode.dpp.x;
	}
	if (mode->dpp.y == GGI_AUTO) {
		mode->dpp.y = th->mode.dpp.y;
	}
	if ((sint32) mode->frames == GGI_AUTO) {
		mode->frames = 1;
	}


	/* Now let the parent target have it's say in the checkmode
	 * process.  It can deal with any remaining GGI_AUTO or GT_AUTO
	 * values that came from th->mode.
	 */

	par_mode = *mode;

	par_mode.graphtype = th->mode.graphtype;

	if (ggiCheckMode(th->parent, &par_mode) != 0) {
		err--;
	}

	mode->visible = par_mode.visible;
	mode->virt    = par_mode.virt;
	mode->dpp     = par_mode.dpp;


        DPRINT_MODE("display-trueemu: upgraded to %dx%d#%dx%dF%d[0x%02x]\n",
                        mode->visible.x, mode->visible.y,
                        mode->virt.x, mode->virt.y,
                        mode->frames, mode->graphtype);

	return err;
}

int GGI_trueemu_getmode(ggi_visual *vis, ggi_mode *mode)
{
	if ((vis == NULL) || (mode == NULL) || (LIBGGI_MODE(vis) == NULL)) {
		DPRINT("display-trueemu: vis/mode == NULL\n");
		return -1;
	}
	
	DPRINT("display-trueemu: getmode.\n");

	memcpy(mode, LIBGGI_MODE(vis), sizeof(ggi_mode));

	return 0;
}

int GGI_trueemu_resetmode(ggi_visual *vis)
{
	TrueemuHook *th = TRUEEMU_PRIV(vis);

	_ggi_trueemu_Close(vis);

	free(th->fb_ptr); 
	th->fb_ptr = NULL;

	return 0;
}


int GGI_trueemu_setflags(ggi_visual *vis, ggi_flags flags)
{
	LIBGGI_FLAGS(vis) = flags;

	MANSYNC_SETFLAGS(vis, flags);

	return 0;
}


int GGI_trueemu_flush(ggi_visual *vis, int tryflag)
{
	TrueemuHook *th = TRUEEMU_PRIV(vis);
	
	int err;

	MANSYNC_ignore(vis);

	_ggiLock(th->flush_lock);

	err = _ggi_trueemu_Flush(vis);

	if (! err) {
		err = _ggiInternFlush(th->parent, tryflag);
	}

	_ggiUnlock(th->flush_lock);

	MANSYNC_cont(vis);

	return err;
}
