/* $Id: visual.c,v 1.26 1998/10/18 15:53:13 becka Exp $
***************************************************************************

   XF86DGA display target.

   Copyright (C) 1997,1998 Steve Cheng   [steve@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/


#include <unistd.h>
#include <stdlib.h>
#include <signal.h>

#include "Xvisual.h"

void _GGI_xf86dga_freedbs(ggi_visual *vis)
{
	int i;
	int first = LIBGGI_APPLIST(vis)->first_targetbuf;
	int last = LIBGGI_APPLIST(vis)->last_targetbuf;
	
	if (first < 0) {
		return;
	}
	for (i = (last - first); i >= 0; i--) {
		_ggi_db_free(LIBGGI_APPBUFS(vis)[i+first]);
		_ggi_db_del_buffer(LIBGGI_APPLIST(vis), i+first);
	}
	LIBGGI_APPLIST(vis)->first_targetbuf = -1;
}


static int _GGI_xf86dga_getbpp(ggi_visual *vis)
{
	XImage *bppcheckimage;
	unsigned int bits_per_pixel=0;
	
	if ((bppcheckimage=XGetImage(XLIB_PRIV(vis)->display,
				     RootWindow(XLIB_PRIV(vis)->display,
						XLIB_PRIV(vis)->screen),
				     0, 0, 1, 1, AllPlanes, ZPixmap)) != NULL) {
		bits_per_pixel=bppcheckimage->bits_per_pixel;
		XDestroyImage(bppcheckimage);
	}
	return bits_per_pixel;
}

/* DGA-installed signal handlers.  I don't know it does with them, but they 
   don't handle all the error conditions. Thus we also need to fork
   to prevent dead apps from locking up the console. */
static int dga_signals[] = {
	SIGSEGV, SIGBUS, SIGHUP, SIGFPE};
static struct sigaction old_signals[sizeof(dga_signals)];

int GGIdlinit(ggi_visual *vis,const char *args,void *argptr)
{
	struct Xhooks *xhook;

	/* Just dummy variables */
	int x, y;
	unsigned z;
	Window root;

	if (geteuid()) {
		fprintf(stderr, "Need root privs to do XF86DGA.\n");
	        return GGI_DL_ERROR;
	}

	GIVE_UP_ROOT;

	for(x = 0; x<sizeof(dga_signals); x++)
		sigaction(dga_signals[x], NULL, old_signals+x);

	DPRINT("XF86DGA starting.\n");

	LIBGGI_GC(vis) = _ggi_malloc(sizeof(ggi_gc));
	LIBGGI_PRIVATE(vis)=xhook=_ggi_malloc(sizeof(struct Xhooks));

	xhook->cmap=0;
	xhook->cmap2=0;
	xhook->nocols=0;

	_ggiLockInit(xhook->XLibLock);

	DPRINT("XF86DGA wants display %s.\n",args);
	xhook->display = XOpenDisplay(args);
	if (xhook->display == NULL)
		return GGI_DL_ERROR;

	DPRINT("XF86DGA has display %s.\n",args);

	xhook->screen = DefaultScreen(xhook->display);

	LIBGGI_SELECT_FD(vis)=LIBGGI_FD(vis)=ConnectionNumber(xhook->display);

	XF86DGAQueryVersion(xhook->display, &x, &y);
	DPRINT("XF86DGA version %d.%d\n", x, y);
	if(x < 1) {
		fprintf(stderr, "Your XF86DGA is too old (%d.%d).\n", x, y);
		return GGI_DL_ERROR;
	}

	XF86VidModeQueryVersion(xhook->display, &x, &y);
	DPRINT("XF86VidMode version %d.%d\n", x, y);
	
	XF86DGAQueryDirectVideo(xhook->display, xhook->screen, &x);
	if (!(x && XF86DGADirectPresent)) {
		fprintf(stderr, "No direct video capability available!\n");
		return GGI_DL_ERROR;
	}

	GET_ROOT;
	XF86DGAGetVideo(xhook->display, xhook->screen, (char**) &xhook->fb,
		&xhook->stride, &xhook->bank_size, &xhook->mem_size);
	PERM_GIVE_UP_ROOT;

	DPRINT("fb: %p, stride: %d, bank_size: %d, mem_size: %d\n",
	       xhook->fb, xhook->stride, xhook->bank_size, xhook->mem_size);

	if (xhook->bank_size != xhook->mem_size*1024) {
		fprintf(stderr,"DGA-target: Sorry, banked framebuffer layout not supported.\n");
		return GGI_DL_ERROR;
	}

	/* Get virtual dimensions */
	XGetGeometry(xhook->display,
		RootWindow(xhook->display, xhook->screen),
		&root, &x, &y,
		&xhook->width, &xhook->height,
		(unsigned int*)&z,
		&xhook->depth);
	
	xhook->size = _GGI_xf86dga_getbpp(vis);

	DPRINT("Virtwidth: %d, depth: %d, size: %d\n", 
		xhook->width, xhook->depth, xhook->size);

	/* Get XF86VidMode modelines */
	XF86VidModeGetAllModeLines(xhook->display, xhook->screen,
		&xhook->num_modes, &xhook->dgamodes);

	xhook->modes = _ggi_malloc((xhook->num_modes+1)*sizeof(ggi_modelistmode));
	
	for(x = 0; x<xhook->num_modes; x++) {
		xhook->modes[x].x = xhook->dgamodes[x]->hdisplay;
		xhook->modes[x].y = xhook->dgamodes[x]->vdisplay;
		xhook->modes[x].bpp = xhook->depth;
		xhook->modes[x].gt = GT_CONSTRUCT(xhook->depth,
			(xhook->depth <= 8) ? GT_PALETTE : GT_TRUECOLOR,
			xhook->size);
		DPRINT("Found mode: %dx%d\n",
			xhook->modes[x].x,
			xhook->modes[x].y);
	}
	xhook->modes[xhook->num_modes].bpp=0;

	XF86DGAForkApp(xhook->screen);

	/* Has mode management */
	vis->opdisplay->getmode=GGI_xf86dga_getmode;
	vis->opdisplay->setmode=GGI_xf86dga_setmode;
	vis->opdisplay->checkmode=GGI_xf86dga_checkmode;
	vis->opdisplay->getapi=GGI_xf86dga_getapi;
	vis->opdisplay->setflags=GGI_xf86dga_setflags;

	/* Has Event management */
	vis->opdisplay->eventpoll=GGI_xf86dga_eventpoll;
	vis->opdisplay->eventread=GGI_xf86dga_eventread;
	vis->opdisplay->seteventmask=GGIseteventmask;

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	struct Xhooks *xhook;
	int i;

	_GGI_xf86dga_freedbs(vis);

	if ((xhook=LIBGGI_PRIVATE(vis)) != NULL)
	{
		XF86DGADirectVideo(xhook->display, xhook->screen, 0);

		XUngrabPointer(xhook->display, CurrentTime);
		XUngrabKeyboard(xhook->display, CurrentTime);

		/* Get rid of DGA's signals */
		for(i = 0; i<sizeof(dga_signals); i++)
			sigaction(dga_signals[i], old_signals+i, NULL);

#ifndef ALWAYS_SEGFAULT_ON_EXIT
		/* prevent-segfault hack */
		if(!_ggiDebugState)
			signal(SIGSEGV, _exit);
#endif

		if (xhook->cmap)
			XFreeColormap(xhook->display,xhook->cmap);
		if (xhook->cmap2)
			XFreeColormap(xhook->display,xhook->cmap2);

		if (xhook->dgamodes) {
			XF86VidModeSwitchToMode(xhook->display, xhook->screen,
				xhook->dgamodes[0]);

			/* Free the modelines */
			for (i = 0; i < xhook->num_modes; i++)
				if (xhook->dgamodes[i]->privsize > 0)
					XFree(xhook->dgamodes[i]->private);

			XFree(xhook->dgamodes);
		}
		if (xhook->modes)
			free(xhook->modes);
		
		if (xhook->display)
			XCloseDisplay(xhook->display);

		_ggiLockDestroy(xhook->XLibLock);

		free(LIBGGI_PRIVATE(vis));
	}

	if(vis->palette)
		free(vis->palette);
	free(LIBGGI_GC(vis));

	return 0;
}

#include <ggi/internal/ggidlinit.h>
