/*
 * demo.c - (c) 1998 Thomas Tanner   tanner@ggi-project.org
 *          (c) 1998 Andreas Beck   andreas.beck@ggi-project.org
 *
 * This is a demonstration of LibGGI2D's functions and can be used as a 
 * reference programming example.
 *
 * Thus it is heavily commented to explain you every single step to make
 * your own applications.
 *
 * Permission is granted to use this source as a reference for your own
 * applications. 
 *
 */

/* Include the necessary headers used for e.g. error-reporting.
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

/* Include the LibGGI declarations.
 */
#include <ggi/ggi.h>
#include <ggi/ggi2d.h>

/* We are running on a single primary visual which is made accessible
 * globally so subroutines like "waitabit" can access it.
 * See its allocation in main() for details.
 */
ggi_visual_t vis;


/* In case we were called with wrong parameters, give an explanation.
 */
void usage(const char *prog)
{
	fprintf(stderr,"Usage:\n\n"
		       "%s <bpp> <xsize> <ysize> [<virtx> <virty>]\n\n"
		       "Default: %s 8 320 200 320 200\n",prog,prog);
	exit(1);
}

/* Wait for a keypress. Shut down everything, if "q" is pressed.
 */
void waitabit(void)
{
	int key;

	/* Make sure the display is in sync with the drawing command queue. 
	 */
	ggiFlush(vis);

	/* ggiGetc will blocking-wait for a keypress. 
	 */
	key=ggiGetc(vis);

	/* This is not really clean. One should use the correct KTYP/KVAL macros.
	 * however the &0xff heuristic is useful for porting old applications
	 * quickly.
	 * Have a look at libggi/vr/vr.c to see how this is done correctly.
	 */
	if ((key&0xff)=='q') /* Q pressed */
	{	ggiExit();
		exit(0);
	}
	
}

/* Pixel value for white and grey. See main() on how to get at it.
 */
ggi_uint white, grey, darkgrey;

/* Print the name of the current test in the top left corner.
 */
void TestName(const char *name)
{
	ggiSetGCForeground(vis,white);
	ggiPuts(vis,0,0,name);
}

/* The main routine.
 * It will set up a graphics mode as requested on the commandline and 
 * do an extensive test using about all graphics primitives LibGGI
 * knows.
 */
int main(int argc,char **argv)
{
	/* First we define a bunch of variables we will access throughout the
	 * main() function. Most of them are pretty meaningless loop-counters
	 * and helper-variables.
	 */

	const char *prog=argv[0];	/* Make an alias for the program name */

	ggi_graphtype type;		/* This is an enum holding the requested
					 * type of graphics mode. See the mode setting
					 * code below for details.
					 */

	ggi_color pal[256];		/* This holds the palette we will soon set for 
					 * GT_8BIT type graphics modes
					 */

	int depth;			/* The depth is used for chosing a graphics type.
					 * it is mapped to the right GT_* by a switch-statement.
					 */

	int sx,sy,vx,vy;		/* These holde the visible screen size (sx,sy) and
					 * the virtual size (vx,vy). On most targets you can 
					 * request a certain larger area on which the visible
					 * area can be placed.
					 */

	int fx=0,fy=0;			/* These hold the font sizes for textmodes */

	int c,err;	/* Pretty meaningless helpers used everywhere. */

	ggi_color map;		/* Yet another array of colors used for color mapping
					 */

	int textmode=0;			/* Flag to indicate we want a textmode. */


	/* Get the arguments from the command line. 
	 * Set defaults for optional arguments.
	 */
	if (argc==1)
		depth=GGI_AUTO;		/* No depth give. Let the driver select. */
	else {
		if ((argv[1][0] | 0x20)=='t') {
			textmode=1;		/* a 't' before the depth indicates we */
			depth=atoi(argv[1]+1);	/* want a textmode */
		} else
			depth=atoi(argv[1]);	/* Read in the depth. */
		argc--;
		argv++;
	}

	if (argc==1) {			/* No size given. Let the driver select. */
		sx=GGI_AUTO;
		sy=GGI_AUTO;
		vx=sx;
		vy=sy;
	} else if (argc==3) {		/* Size given - read it. */
		sx=atoi(argv[1]);
		sy=atoi(argv[2]);
		vx=sx;
		vy=sy;
	} else if (argc==5) {		/* Size and virtual size given. Read them. */
		sx=atoi(argv[1]);
		sy=atoi(argv[2]);
		vx=atoi(argv[3]);
		vy=atoi(argv[4]);
	} else {
		usage(prog);		/* This is not an allowed call format. */
		return 1;		/* Tell the user how to call and fail. */
	}

	/* sx,sy,vx,vy now hold the requested visible 
	 * and virtual size of the screen.
	 */
	printf("Asking for mode (%dx%d [%dx%d] %c%dbpp)\n",
		sx,sy,vx,vy,textmode ? 'T' : ' ',depth);

	/* Now find the right GGI graphics type.
	 * for this we have to map the "depth" to the right GT_* enum value.
	 */
	if (textmode==1) {	/* In case we are requesting a textmode, there are */
		fx=vx;
		fy=vy;
		vx=sx;
		vy=sy;
		if (depth==16) {	/* two possibilities : */
			type=GT_TEXT16;	/* VGA style */
		} else if (depth==32) {
			type=GT_TEXT32;	/* four byte wide extended mode. */
		} else {
			 fprintf(stderr,"%s: Invalid textmode depth!\n\n",prog);
			 usage(prog);
			 return 1;
		}
	} else {
	   switch (depth) {	/* Else we want a graphics mode. */
		case GGI_AUTO:	/* Note this is "-1". */
			type=GGI_AUTO;	/* Let the driver select. */
			break;
		case 1:
			type=GT_1BIT;	/* Each case now maps a depth to the right enum */
			break;
		case 4: type=GT_4BIT;
			break;
		case 8:	type=GT_8BIT;
			break;
		case 15: type=GT_15BIT;
			 break;
		case 16: type=GT_16BIT;
			 break;
		case 24: type=GT_24BIT;
			 break;
		case 32: type=GT_32BIT;
			 break;
		default:
			 fprintf(stderr,"%s: Invalid depth!\n\n",prog);
			 usage(prog);
			 return 1;
	   }
	}

	/* Set up the random number generator. */
	srandom(time(NULL));

	/* Initialize the GGI2D library. This must be called before any other 
	 * GGI2D function. Otherwise behaviour is undefined. Note that this 
	 * implicitly calls ggiInit(). No need for explicit call (won't hurt
	 * if you do).
	 */
	ggiInit();
	ggi2dInit();

	/* Open the default visual. This is automatically selected depending
	 * on the calling environment (Linux console, X, ...). You can 
	 * explicitly set this in the LIBGGI_DISPLAY environment variable.
	 */
	vis=ggiOpen(NULL);

	/* Set up the text or graphics mode as requested */
	if (textmode) 
		err=ggiSetTextMode(vis,sx,sy,vx,vy,fx,fy,type);
	else
		/* Set up a graphics screen, sx*sy visible, 
		 * vx*vy virtual, mode type 
		 */
		err=ggiSetGraphMode(vis,sx,sy,vx,vy,type);


	/* Now we read back the set mode, as it might have been autoselected.
	 */

	{ ggi_mode mode;
	  ggiGetMode(vis,&mode);
	  vx=mode.virt.x;   vy=mode.virt.y;
	  sx=mode.visible.x;sy=mode.visible.y;
	  switch (mode.graphtype) {
		case GT_1BIT:
			depth=1;break;
		case GT_4BIT:
			depth=4;break;
		case GT_8BIT:
			depth=8;break;
		case GT_15BIT:
			depth=15;break;
		case GT_16BIT:
			depth=16;break;
		case GT_24BIT:
			depth=24;break;
		case GT_32BIT:
			depth=32;break;
		default:
			break;
	   }
	}
	printf("Got mode (%dx%d [%dx%d])\n",
		sx,sy,vx,vy);

	/* Check if there were errors. Almost all LibGGI functions return 0 on success
	 * and an error code otherwise. No messing with errno, as this is not thread-
	 * friendly. The functions that are supposed to return a pointer normally return
	 * NULL on error.
	 */
	if (err) {
		fprintf(stderr,"Can't set mode\n");
		return 2;
	}
	
	/* Find the color "white".
	 * LibGGI makes no assumptions on how a color value (which is an uint) is 
	 * actually mapped to a color on the screen. If you want to write strictly
	 * compatible programs, you need to ask LibGGI for the color value associated
	 * with a given color. We will do this now:
	 */
	for(c=0;c<256;c++)
	 pal[c].r = pal[c].g = pal[c].b = c*256; 
        ggiSetPalette(vis,0,256,pal);

	map.r=0xFFFF;
	map.g=0xFFFF;
	map.b=0xFFFF;
	white=ggiMapColor(vis,&map);
	printf("white=%d\n",white);

	map.r=0xF000;
	map.g=0xF000;
	map.b=0xF000;
	grey=ggiMapColor(vis,&map);
	printf("grey=%d\n",grey);
	
	map.r=0xE000;
	map.g=0xE000;
	map.b=0xE000;
	darkgrey=ggiMapColor(vis,&map);
	printf("darkgrey=%d\n",darkgrey);
	
	/* Initialize the visual for 2D drawing */
	ggi2dOpen(vis);
	
	/* Set the drawing color to white and display some text at 0/0. 
	 */
	ggiSetGCForeground(vis,white);
	ggiPuts(vis,0,0,"Press any key to begin tests...");
#if 0	/* Activate this for clipping tests ... */
	ggiDrawHLine(vis,0,39,sx);
	ggiDrawHLine(vis,0,sy-40,sx);
	ggiDrawVLine(vis,39,0,sy);
	ggiDrawVLine(vis,sx-40,0,sy);
	ggiSetGCClipping(vis,40,40,sx-40,sy-40);
	ggi2dSetClip(vis,40,40,sx-40,sy-40);
#endif

	/* Wait for any keypress. 
	 * This is a blocking call which returns a 16 bit wide unicode character.
	 * See the comments at waitabit() for details. Do not just do things like
	 * if (ggiGetc=='y') yes();.
	 */
	ggiGetc(vis);

	/* Set a colorful palette for the other tests.
	 * Please note that GGI always uses 16 bit palettes, so
	 * stretch the values accordingly when porting from DOS 
	 * or other libs that make assumptions about palette size.
	 *
	 * On paletteless modes the ggiSetPalette() call will fail.
	 * We silently ignore that.
	 */

	ggi2dSetDrawColor(vis, white);
	ggi2dSetFillColor(vis, grey);
	ggi2dFillRect(vis, 10, 10, 40, 50);
	ggi2dSetFillColor(vis, 150);
	ggi2dTriangle(vis, 30, 40, 60, 110, 120, 25);
	ggi2dDrawCircle(vis, 100, 100, 50);
	ggi2dDrawEllipse(vis, 100, 100, 70, 40);
	ggi2dLine(vis, 10, 10, 70, 90);
	ggi2dLine(vis, 20, 10, 100, 70);
	ggi2dSetOperator(vis, GGI2D_INVERT);
	ggi2dFillRect(vis, 100, 100, 320, 200);
	ggiGetc(vis);

	/* O.K. - all done. we will now cleanly shut down LibGGI2D.
	 */

	/* First close down the visual we requested. Any further calls
	 * using vis have undefined behaviour. LibGGI2D is still up and 
	 * you could open another visual
	 */
	ggi2dClose(vis);
	
	/* Now close down LibGGI2D. Every LibGGI2D call except ggi2dInit has 
	 * undefined behaviour now. It is not recommended to needlessly
	 * deinit-reinit LibGGI, but it should work.
	 */
	ggi2dExit();
	ggiExit();

	/* Terminate the program.
	 */
	return 0;
}
