/* $Id: parse.c,v 1.1 1998/10/24 04:04:10 marcus Exp $
***************************************************************************

   LibGG - Parsing code

   Copyright (C) 1998  Andrew Apted     [andrew.apted@ggi-project.org]

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <ggi/gg.h>
#include "plat.h"

/*
***************************************************************************
   char *
   ggParseTarget(char *str, char *target, int max);
 
 	Parses a target descriptor out of a string, and returns a
 	pointer to after the last character of the target.  Handles
 	bracketized forms appropriately.  Returns NULL if an error
 	occurs.
 
 	For example:
 
               "display-multi:(display-kgi:/dev/head1):display-x\0"
                              ^                       ^
                              |__str                  |__result
 
               target: "display-kgi:/dev/head1"
***************************************************************************
*/

char *ggParseTarget(char *str, char *target, int max)
{
	int bracketized = 0;
	int bracket_count = 0;

	while (*str && isspace((int) *str)) {
		str++;
	}

	if (*str == 0) {
		*target = 0;
		return str;
	}

	if (*str == '(') {
		bracketized=1;
		bracket_count++;
		str++;
	}

	while (*str) {
		if (*str == '(') {
			bracket_count++;
		} else if (*str == ')') {
		
			if (bracket_count == 0) {
				fprintf(stderr, "libgg: Bad target "
					"descriptor : unexpected ')'\n");
				*target = 0;
				return NULL;
			}

			bracket_count--;

			if (bracketized && (bracket_count == 0)) {
				str++;
				break;
			}
		}

		if (max <= 2) {
			fprintf(stderr, "libgg: target descriptor "
				"too long\n");
			*target = 0;
			return NULL;
		}
		
		*target++ = *str++; max--;
	}

	*target = 0;

	if (bracket_count > 0) {
		fprintf(stderr, "libgg: Bad target descriptor : "
			"missing ')'\n");
		return NULL;
	}

	return str;
}

/*
***************************************************************************
   char *
   ggParseOptions(char *str, gg_option *optlist, int count);

 	Parses the target options out of the argument string.  Each
 	option begins with a '-' characters, and can be given a value
 	using the '=' characters.  Some examples "-nodb", "-parent=8".
 	Options are separated by ':' characters or whitespace.  
 
 	The recognizable options are stored in the array of gg_options:
 	`optlist'.  Matching is case sensitive, but options can be
 	abbreviated right down to a single letter.  Returns NULL if an
 	error occurred.
 	
       For example:
 
               "trueemu:-parent=8 -d=4:display-fbdev"
                        ^              ^
                        |__str         |__result
 	
 		optlist[0].name: "parent"     (in)
 		optlist11].name: "dither"     (in)
 
 		optlist[0].result: "8"        (out)
 		optlist11].result: "4"        (out)
***************************************************************************
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define TERMINATOR(c)  (((c) == 0) || isspace((int)c) || ((c) == ':'))

char *ggParseOptions(char *str, gg_option *optlist, int count)
{
	char name[MAX_OPTION_NAME]; int len;

	gg_option *cur; int i;


	for (;;) {

		while (*str && isspace((int)*str)) 
			str++;

		if (*str != '-')
			break;

		/* parse option's name */

		len = 0;

		for (str++; ! TERMINATOR(*str) && (*str != '='); str++) {
			if (len < MAX_OPTION_NAME-1) {
				name[len++] = *str;
			}
		}

		if (len == 0) {
			fprintf(stderr, "libgg: Bad target options : "
				"missing option name\n");
			return NULL;
		}

		name[len] = 0;

		/* find option */

		cur = NULL;

		for (i=0; i < count; i++) {
			if (strncmp(optlist[i].name, name, len) == 0) {
				/* found it */
				cur = optlist + i;
			}
		}

		if (cur == NULL) {
			fprintf(stderr, "libgg: Unknown target option "
				"'%s'\n", name);
			return NULL;
		}

		/* store value */

		if (*str != '=') {
			strcpy(cur->result, "y");
			
		} else {
			len = 0;

			for (str++; ! TERMINATOR(*str); str++) {
				if (len < MAX_OPTION_RESULT-1) {
					cur->result[len++] = *str;
				}
			}
		}

		cur->result[len] = 0;

		if (*str == ':')
			str++;
	}

	return str;
}

#undef TERMINATOR
