/* 
 * tkUnix3d.c --
 *
 *	This file contains the platform specific routines for
 *	drawing 3d borders in the Motif style.
 *
 * Copyright (c) 1996 by Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkUnix3d.c 1.3 96/11/20 14:24:38
 */
/*
 * TkSTEP modifications Copyright (c) Alfredo K. Kojima
 * - surgery performed and implanted into Tk8.0 by Steve Murray
 */


#include <tk3d.h>

/*
 * This structure is used to keep track of the extra colors used
 * by Unix 3d borders.
 */

typedef struct {
    TkBorder info;
    GC solidGC;		/* Used to draw solid relief. */
} UnixBorder;

/*
 *----------------------------------------------------------------------
 *
 * TkpGetBorder --
 *
 *	This function allocates a new TkBorder structure.
 *
 * Results:
 *	Returns a newly allocated TkBorder.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

TkBorder *
TkpGetBorder()
{
    UnixBorder *borderPtr = (UnixBorder *) ckalloc(sizeof(UnixBorder));
    borderPtr->solidGC = None;
    return (TkBorder *) borderPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * TkpFreeBorder --
 *
 *	This function frees any colors allocated by the platform
 *	specific part of this module.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	May deallocate some colors.
 *
 *----------------------------------------------------------------------
 */

void
TkpFreeBorder(borderPtr)
    TkBorder *borderPtr;
{
    UnixBorder *unixBorderPtr = (UnixBorder *) borderPtr;
    Display *display = DisplayOfScreen(borderPtr->screen);

    if (unixBorderPtr->solidGC != None) {
	Tk_FreeGC(display, unixBorderPtr->solidGC);
    }
}
/*
 *--------------------------------------------------------------
 *
 * Tk_3DVerticalBevel --
 *
 *	This procedure draws a vertical bevel along one side of
 *	an object.  The bevel is always rectangular in shape:
 *			|||
 *			|||
 *			|||
 *			|||
 *			|||
 *			|||
 *	An appropriate shadow color is chosen for the bevel based
 *	on the leftBevel and relief arguments.  Normally this
 *	procedure is called first, then Tk_3DHorizontalBevel is
 *	called next to draw neat corners.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Graphics are drawn in drawable.
 *
 *--------------------------------------------------------------
 */

void
Tk_3DVerticalBevel(tkwin, drawable, border, x, y, width, height,
	leftBevel, relief)
    Tk_Window tkwin;		/* Window for which border was allocated. */
    Drawable drawable;		/* X window or pixmap in which to draw. */
    Tk_3DBorder border;		/* Token for border to draw. */
    int x, y, width, height;	/* Area of vertical bevel. */
    int leftBevel;		/* Non-zero means this bevel forms the
				 * left side of the object;  0 means it
				 * forms the right side. */
    int relief;			/* Kind of bevel to draw.  For example,
				 * TK_RELIEF_RAISED means interior of
				 * object should appear higher than
				 * exterior. */
{
    TkBorder *borderPtr = (TkBorder *) border;
    GC left, right;
    Display *display = Tk_Display(tkwin);
    int half;


    if ((borderPtr->lightGC == None) && (relief != TK_RELIEF_FLAT)) {
	TkpGetShadows(borderPtr, tkwin);
    }

    if (relief == TK_RELIEF_RAISED) {
	if (width > 1) {
	    if (leftBevel) {
	        left = borderPtr->lightGC;
	        right = borderPtr->bgGC;
	    } else {
	        left = borderPtr->darkGC;
	        right = borderPtr->dark2GC;
	    }
	    half = width/2;
	    XFillRectangle(display, drawable, left,
	 	x, y, (unsigned) half, (unsigned) height);
	    XFillRectangle(display, drawable, right,
		x+half, y, (unsigned) (width - half), (unsigned) height);
	} else {
	    left = (leftBevel) ? borderPtr->lightGC : borderPtr->dark2GC;
	    XFillRectangle(display, drawable, left,
	 	x, y, (unsigned) width, (unsigned) height);	
 	}
    } else if (relief == TK_RELIEF_SUNKEN) {
	if (width > 1) {
	    if (leftBevel) {
	    	left = borderPtr->darkGC;
	    	right = borderPtr->dark2GC;	    
	    } else {
	    	left = borderPtr->bgGC;
	   	right = borderPtr->lightGC;
	    }
	    half = width/2;
	    XFillRectangle(display, drawable, left, 
		x, y, (unsigned) half, (unsigned) height);
	    XFillRectangle(display, drawable, right, 
		x + half, y, (unsigned) (width - half), (unsigned) height);
        } else {
	    left = (leftBevel) ? borderPtr->dark2GC : borderPtr->lightGC;
	    XFillRectangle(display, drawable, left,
	 	x, y, (unsigned) width, (unsigned) height);
    	}
    } else if (relief == TK_RELIEF_RIDGE) {
	left = borderPtr->lightGC;
	right = borderPtr->dark2GC;
	ridgeGroove:
	half = width/2;
	if (!leftBevel && (width & 1)) {
	    half++;
	}
	XFillRectangle(display, drawable, left, x, y, (unsigned) half,
		(unsigned) height);
	XFillRectangle(display, drawable, right, x+half, y,
		(unsigned) (width-half), (unsigned) height);
    } else if (relief == TK_RELIEF_GROOVE) {
	left = borderPtr->dark2GC;
	right = borderPtr->lightGC;
	goto ridgeGroove;
    } else if (relief == TK_RELIEF_FLAT) {
	XFillRectangle(display, drawable, borderPtr->bgGC, x, y,
		(unsigned) width, (unsigned) height);
    } else if (relief == TK_RELIEF_SOLID) {
	UnixBorder *unixBorderPtr = (UnixBorder *) borderPtr;
	if (unixBorderPtr->solidGC == None) {
	    XGCValues gcValues;

	    gcValues.foreground = BlackPixelOfScreen(borderPtr->screen);
	    unixBorderPtr->solidGC = Tk_GetGC(tkwin, GCForeground, &gcValues);
	}
	XFillRectangle(display, drawable, unixBorderPtr->solidGC, x, y,
		(unsigned) width, (unsigned) height);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tk_3DHorizontalBevel --
 *
 *	This procedure draws a horizontal bevel along one side of
 *	an object.  The bevel has mitered corners (depending on
 *	leftIn and rightIn arguments).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

void
Tk_3DHorizontalBevel(tkwin, drawable, border, x, y, width, height,
	leftIn, rightIn, topBevel, relief)
    Tk_Window tkwin;		/* Window for which border was allocated. */
    Drawable drawable;		/* X window or pixmap in which to draw. */
    Tk_3DBorder border;		/* Token for border to draw. */
    int x, y, width, height;	/* Bounding box of area of bevel.  Height
				 * gives width of border. */
    int leftIn, rightIn;	/* Describes whether the left and right
				 * edges of the bevel angle in or out as
				 * they go down.  For example, if "leftIn"
				 * is true, the left side of the bevel
				 * looks like this:
				 *	___________
				 *	 __________
				 *	  _________
				 *	   ________
				 */
    int topBevel;		/* Non-zero means this bevel forms the
				 * top side of the object;  0 means it
				 * forms the bottom side. */
    int relief;			/* Kind of bevel to draw.  For example,
				 * TK_RELIEF_RAISED means interior of
				 * object should appear higher than
				 * exterior. */
{
    TkBorder *borderPtr = (TkBorder *) border;
    Display *display = Tk_Display(tkwin);
    int bottom, halfway, x1, x2, x1Delta, x2Delta;
    UnixBorder *unixBorderPtr = (UnixBorder *) borderPtr;
    GC topGC = None, bottomGC = None;
				/* Initializations needed only to prevent
				 * compiler warnings. */

    if ((borderPtr->lightGC == None) && (relief != TK_RELIEF_FLAT) &&
	    (relief != TK_RELIEF_SOLID)) {
	TkpGetShadows(borderPtr, tkwin);
    }

    /*
     * Compute a GC for the top half of the bevel and a GC for the
     * bottom half (they're the same in many cases).
     */

    switch (relief) {
	case TK_RELIEF_FLAT:
	    topGC = bottomGC = borderPtr->bgGC;
	    break;
	case TK_RELIEF_GROOVE:
	    topGC = borderPtr->dark2GC;
	    bottomGC = borderPtr->lightGC;
	    break;
	case TK_RELIEF_RAISED:
            if (height == 1) {
                topGC = (topBevel) ? borderPtr->lightGC : borderPtr->dark2GC;
            } else {
                topGC = (topBevel) ? borderPtr->lightGC : borderPtr->darkGC;
                bottomGC = (topBevel) ? borderPtr->bgGC : borderPtr->dark2GC;
            }
	    break;
	case TK_RELIEF_RIDGE:
	    topGC = borderPtr->lightGC;
	    bottomGC = borderPtr->dark2GC;
	    break;
	case TK_RELIEF_SOLID:
	    if (unixBorderPtr->solidGC == None) {
		XGCValues gcValues;

		gcValues.foreground = BlackPixelOfScreen(borderPtr->screen);
		unixBorderPtr->solidGC = Tk_GetGC(tkwin, GCForeground,
			&gcValues);
	    }
	    XFillRectangle(display, drawable, unixBorderPtr->solidGC, x, y,
		(unsigned) width, (unsigned) height);
	    return;
	case TK_RELIEF_SUNKEN:
            if (height == 1) {
                topGC = (topBevel) ? borderPtr->dark2GC : borderPtr->lightGC;
            } else {
                topGC = (topBevel) ? borderPtr->darkGC : borderPtr->bgGC;
                bottomGC = (topBevel) ? borderPtr->dark2GC : borderPtr->lightGC;
            }
	    break;
    }

    /*
     * Compute various other geometry-related stuff.
     */

    x1 = x;
    if (!leftIn) {
	x1 += height-1;
    }
    x2 = x+width;
    if (!rightIn) {
	x2 -= height-1;
    }
    x1Delta = (leftIn) ? 1 : -1;
    x2Delta = (rightIn) ? -1 : 1;
    halfway = y + height/2;
    if (!topBevel && (height & 1)) {
	halfway++;
    }
    bottom = y + height;

    /*
     * if borderwidth is 1, then just draw the line and get out !
     */
    if (height == 1) {
        if (x1 < x2) {
            XFillRectangle(display, drawable,
                topGC, x1, y, (unsigned) (x2-x1-topBevel), (unsigned) 1);
        }
    } else {
        /*
         * Draw one line for each y-coordinate covered by the bevel.
         */
        for ( ; y < bottom; y++) {
	    /*
	     * In some weird cases (such as large border widths for skinny
	     * rectangles) x1 can be >= x2.  Don't draw the lines
	     * in these cases.
	     */

	    if (x1 < x2) {
	        XFillRectangle(display, drawable,
		    (y < halfway) ? topGC : bottomGC, x1, y,
		    (unsigned) (x2-x1-topBevel), (unsigned) 1);
	    }
	    x1 += x1Delta;
	    x2 += x2Delta;
        }
    }
}

/*
 *----------------------------------------------------------------------
 *
 * TkpGetShadows --
 *
 *	This procedure computes the shadow colors for a 3-D border
 *	and fills in the corresponding fields of the Border structure.
 *	It's called lazily, so that the colors aren't allocated until
 *	something is actually drawn with them.  That way, if a border
 *	is only used for flat backgrounds the shadow colors will
 *	never be allocated.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The lightGC and darkGC fields in borderPtr get filled in,
 *	if they weren't already.
 *
 *----------------------------------------------------------------------
 */

void
TkpGetShadows(borderPtr, tkwin)
    TkBorder *borderPtr;		/* Information about border. */
    Tk_Window tkwin;		/* Window where border will be used for
				 * drawing. */
{
    XColor lightColor, darkColor;
    int stressed, tmp1, tmp2;
    XGCValues gcValues;

    if (borderPtr->lightGC != None) {
	return;
    }
    stressed = TkpCmapStressed(tkwin, borderPtr->colormap);

    /*
     * First, handle the case of a color display with lots of colors.
     * The shadow colors get computed using whichever formula results
     * in the greatest change in color:
     * 1. Lighter shadow is half-way to white, darker shadow is half
     *    way to dark.
     * 2. Lighter shadow is 40% brighter than background, darker shadow
     *    is 40% darker than background.
     */

    if (!stressed && (Tk_Depth(tkwin) >= 6)) {
	/*
	 * This is a color display with lots of colors.  For the dark
	 * shadow, cut 40% from each of the background color components.
	 * For the light shadow, boost each component by 40% or half-way
	 * to white, whichever is greater (the first approach works
	 * better for unsaturated colors, the second for saturated ones).
	 */

	darkColor.red = (60 * (int) borderPtr->bgColorPtr->red)/100;
	darkColor.green = (60 * (int) borderPtr->bgColorPtr->green)/100;
	darkColor.blue = (60 * (int) borderPtr->bgColorPtr->blue)/100;
	borderPtr->darkColorPtr = Tk_GetColorByValue(tkwin, &darkColor);
	gcValues.foreground = borderPtr->darkColorPtr->pixel;
	borderPtr->darkGC = Tk_GetGC(tkwin, GCForeground, &gcValues);

        /*
         * dark2 colous is always black
         */
        gcValues.foreground = BlackPixelOfScreen(borderPtr->screen);
        borderPtr->dark2GC = Tk_GetGC(tkwin, GCForeground, &gcValues);

        /*
         * light colour is always white
         */
	gcValues.foreground = WhitePixelOfScreen(borderPtr->screen);
	borderPtr->lightGC = Tk_GetGC(tkwin, GCForeground, &gcValues);
	return;
    }

    if (borderPtr->shadow == None) {
	borderPtr->shadow = Tk_GetBitmap((Tcl_Interp *) NULL, tkwin,
		Tk_GetUid("gray50"));
	if (borderPtr->shadow == None) {
	    panic("TkpGetShadows couldn't allocate bitmap for border");
	}
    }
    if (borderPtr->visual->map_entries > 2) {
	/*
	 * This isn't a monochrome display, but the colormap either
	 * ran out of entries or didn't have very many to begin with.
	 * Generate the light shadows with a white stipple and the
	 * dark shadows with a black stipple.
	 */

	gcValues.foreground = borderPtr->bgColorPtr->pixel;
	gcValues.background = BlackPixelOfScreen(borderPtr->screen);
	gcValues.stipple = borderPtr->shadow;
	gcValues.fill_style = FillOpaqueStippled;
	borderPtr->darkGC = Tk_GetGC(tkwin,
		GCForeground|GCBackground|GCStipple|GCFillStyle, &gcValues);
	gcValues.background = WhitePixelOfScreen(borderPtr->screen);
	borderPtr->lightGC = Tk_GetGC(tkwin,
		GCForeground|GCBackground|GCStipple|GCFillStyle, &gcValues);
        gcValues.foreground = gcValues.background;
        borderPtr->dark2GC = Tk_GetGC(tkwin,
                GCForeground|GCBackground, &gcValues);
	return;
    }

    /*
     * This is just a measly monochrome display, hardly even worth its
     * existence on this earth.  Make one shadow a 50% stipple and the
     * other the opposite of the background.
     */

    gcValues.foreground = WhitePixelOfScreen(borderPtr->screen);
    gcValues.background = BlackPixelOfScreen(borderPtr->screen);
    gcValues.stipple = borderPtr->shadow;
    gcValues.fill_style = FillOpaqueStippled;
    borderPtr->lightGC = Tk_GetGC(tkwin,
	    GCForeground|GCBackground|GCStipple|GCFillStyle, &gcValues);
    if (borderPtr->bgColorPtr->pixel
	    == WhitePixelOfScreen(borderPtr->screen)) {
	gcValues.foreground = BlackPixelOfScreen(borderPtr->screen);
	borderPtr->darkGC = Tk_GetGC(tkwin, GCForeground, &gcValues);
    } else {
	borderPtr->darkGC = borderPtr->lightGC;
	borderPtr->lightGC = Tk_GetGC(tkwin, GCForeground, &gcValues);
    }
    gcValues.foreground = gcValues.background;
    borderPtr->dark2GC = Tk_GetGC(tkwin,
        GCForeground|GCBackground, &gcValues);
}


/*
 *---------------------------------------------------------------------- 
 * 
 * Draw3DCircle --
 * 	Draws a beveled circle
 * 
 * TODO: Fix circle code
 *---------------------------------------------------------------------- 
 */
void Tk_Draw3DCircle(display, tkwin, d, x, y, wid, rad, relief, border)
    Display *display;
    Tk_Window tkwin;
    Drawable d;
    int x, y;          			/* position in d */
    int wid;				/* border width */
    int rad;
    int relief;
    Tk_3DBorder border;
{
    GC lightGC, darkGC, dark2GC, bgGC;
    int half, i;
    
    if (relief==TK_RELIEF_SUNKEN) {	
	lightGC = Tk_3DBorderGC(tkwin, border, TK_3D_LIGHT_GC);
	darkGC = Tk_3DBorderGC(tkwin, border, TK_3D_DARK_GC);
	dark2GC = Tk_3DBorderGC(tkwin, border, TK_3D_DARK2_GC);
	bgGC = Tk_3DBorderGC(tkwin, border, TK_3D_FLAT_GC);
    } else {
	darkGC = Tk_3DBorderGC(tkwin, border, TK_3D_LIGHT_GC);
	lightGC = Tk_3DBorderGC(tkwin, border, TK_3D_DARK_GC);
	bgGC = Tk_3DBorderGC(tkwin, border, TK_3D_DARK2_GC);
	dark2GC = Tk_3DBorderGC(tkwin, border, TK_3D_FLAT_GC);
    }    
    half = wid/2;
    if (half>1) {
	XSetLineAttributes(display, darkGC, half, LineSolid, CapButt,
		JoinMiter);
	XSetLineAttributes(display, lightGC, half, LineSolid, CapButt,
		JoinMiter);
	XSetLineAttributes(display, dark2GC, half, LineSolid, CapButt,
		JoinMiter);
	XSetLineAttributes(display, bgGC, half, LineSolid, CapButt, 
		JoinMiter);
    } 
    /*
     * These coords are based on trial and error
     */
    XDrawArc(display, d, dark2GC, x, y, rad*2-half, rad*2-half, 
	60*64, 150*64);
    XDrawArc(display, d, darkGC, x, y, rad*2, rad*2, 55*64, 160*64);
    XDrawArc(display, d, lightGC, x, y, rad*2, rad*2, 35*64, -160*64);
    XDrawArc(display, d, dark2GC, x+half, y+half, rad*2-half, 
	rad*2-half, 60*64, 150*64);    
    XDrawArc(display, d, bgGC, x, y, rad*2-half, rad*2-half, 
	30*64, -150*64);
    if (half>1) {
	XSetLineAttributes(display, dark2GC, 0, LineSolid, CapButt,
		JoinMiter);
	XSetLineAttributes(display, lightGC, 0, LineSolid, CapButt,
		JoinMiter);
	XSetLineAttributes(display, darkGC, 0, LineSolid, CapButt,
		JoinMiter);
	XSetLineAttributes(display, bgGC, 0, LineSolid, CapButt,
		JoinMiter);
    }
}



/*
 *----------------------------------------------------------------------
 * 
 * DrawCheckMark --
 *     
 *     Draws a 3D check mark on the drawable
 * 
 *----------------------------------------------------------------------
 */
void Tk_DrawCheckMark(display, tkwin, d, x, y, border)
    Display *display;
    Tk_Window tkwin;
    Drawable d;
    int x, y;          			/* position in d */
    Tk_3DBorder border;
{
    GC lightGC, darkGC, dark2GC;
    
    lightGC = Tk_3DBorderGC(tkwin,border,TK_3D_LIGHT_GC);
    darkGC = Tk_3DBorderGC(tkwin,border,TK_3D_DARK_GC);
    dark2GC = Tk_3DBorderGC(tkwin,border,TK_3D_DARK2_GC);
    XDrawLine(display, d, dark2GC, x+1, y+3, x+1, y+6); 
    XDrawLine(display, d, lightGC, x, y+3, x, y+8);     
    XDrawLine(display, d, lightGC, x, y+8, x+8, y);
    XDrawLine(display, d, dark2GC, x, y+9, x+8, y+1);
    XDrawLine(display, d, darkGC, x+1, y+9, x+8, y+2);
}

