/*
 * Copyright (c) 1997, 1998 Akira Yoshiyama <yosshy@debian.or.jp>
 * Copyright (c) 1998 Fumitoshi UKAI <ukai@debian.or.jp>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public Licenses as
 * published by the Free Software Foundation; either version 2 of 
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * wcsnrtombs_locale ()
 * for conversion from UCS-4 to multibyte character
 * programed
 *   by A. Yoshiyama (yosshy@jedi.seg.kobe-u.ac.jp)
 *   on Dec.17,1996
 */

#include <stdio.h>
#include <wchar.h>
#include <errno.h>
#include <wcsmbs_locale.h>
#include "wcs2mbs.h"

#if DEBUG
#include <stdio.h>
#include <stdlib.h>
#endif

#ifndef EILSEQ
#define EILSEQ EINVAL
#endif

size_t
dll_wcsnrtombs_locale(dst, src, srclim, dstlim, ps)
     char *dst;
     const wchar_t **src;
     size_t srclim;
     size_t dstlim;
     mbstate_t *ps;
{
  static mbstate_t internal = {0, (wint_t)0};
  extern table_wcs2mbs_t table_wcs2mbs[];
  extern int escpolicy;
  int written = 0;
  const wchar_t *run;
  table_wcs2mbs_t *lasttable;
#if DEBUG
  int i;
#endif

#if DEBUG
  fprintf(stderr,"%s:dll_wcsnrtombs_locale(dst, src, srclim, dstlim, ps)\n",
	  getenv(MONENV));
  fprintf(stderr,"dst=%p, src=%p, srclim=%d, dstlim=%d, ps=%p)\n",
	  dst, src, srclim, dstlim, ps);
  for (i=0; *((*src)+i); i++)
    fprintf(stderr, "%08x ", *((*src)+i));
  fputc('\n',stderr);
#endif

  if (ps == NULL)
    ps = &internal;

  if (src == NULL || *src == NULL)
    {
      /* Reset the current shift state. */
      ps->count = ps->value = 0;
#if DEBUG
      fprintf(stderr,"dll_wcsnrtombs_locale:result = %d\n", 0);
#endif
      return (size_t)0;
    }
  else
    run = *src;

  if (ps->count < 0 || ps->count >= table_wcs2mbs_maxentry)
    {
      /* The PS->COUNT is out of range. */
#if DEBUG
      fprintf(stderr,"dll_wcsnrtombs_locale:result = EINVAL\n");
#endif
      __set_errno (EINVAL);
      return (size_t)-1;
    }
  else
    /* The PS->COUNT parameter means the last shift state (table no.) */
    lasttable = &table_wcs2mbs[(int)(ps->count)];

  while ((written < dstlim || dst == NULL) && srclim-- > 0)
    {
      table_wcs2mbs_t *curtable;
      const upper_table_t *curupper;
      wchar_t wc = *run++;
      int i, j;
      
      if (wc < 0 || wc > 0x7fffffff)
        {
          /* This is no correct ISO 10646 character. */
          __set_errno(EILSEQ);
#if DEBUG
	  fprintf(stderr,"dll_wcsnrtombs_locale:result = EILSEQ\n");
#endif
          return (size_t)-1;
        }

      /* Check whether 'run' reachs the last of src */
      if (wc == L'\0')
        {
          /* It reaches the end. */
          if (escpolicy == HAS_STATE && lasttable != table_wcs2mbs)
            {
              if (dst != NULL)
		/* Add the escape sequence of the default code set if dst isn't NULL. */
		if (written + curtable->esnum <= dstlim)
		  for (i = 0; i < curtable->esnum; i++)
		    *dst++ = curtable->escseq[i];
		else
		  goto exit_loop;
	      
              /* Add escape sequence number to written */
              written += curtable->esnum;
	    }

	  /* Add a string terminator. */
	  if (dst != NULL && written < dstlim)
	    *dst = '\0';

	  /* set NULL to 'src' */
	  *src = NULL;

	  /* Reset shift state. */
	  ps->count = 0;

	  /* Return from this routine. */
#if DEBUG
	  if (*src != NULL)
	    fprintf(stderr,"*src=%p:%s\n", *src, *src);
	  fprintf(stderr,"dll_wcsnrtombs_locale:result = %d\n", written);
#endif
	  return written;

	}
      /* Search if WC isn't NULL. */
      for (j = 0; j < table_wcs2mbs_maxentry; j++)
	{
	  /* Get the size of one cell of lower table */
	  int mbcnum;
	  char ch;
	  curtable = &table_wcs2mbs[j];
	  mbcnum = curtable->mbcnum + 1;

	  /* Get the upper byte of UCS2 */
	  ch = (char)((wc & 0xFF00) >> 8);

	  /* Search it in the current upper table. */
	  for (curupper =  curtable->uppertable;
	       curupper == curtable->uppertable || curupper->key != (char)0;
	       curupper++)
	    /* Check whether the current entry is it. */
	    if (curupper->key == ch)
	      {
		const char *p;

		/* Get the lower byte of UCS2. */
		ch = (char)(wc & 0xFF);

		/* Search it in the current lower table. */
		for (p = curtable->lowertable + curupper->pointer * mbcnum;
		     p < curtable->lowertable + (curupper+1)->pointer * mbcnum;
		     p += mbcnum)
		  /* Check whether the current entry is it. */
		  if (*p == ch)
		    {
		      if (escpolicy == HAS_STATE && curtable == lasttable)
			{
			  /* Check whether WRITTEN is lager than DSTLIM. */
			  if (dst != NULL 
			      && written + curtable->mbcnum > dstlim)
			    goto exit_loop;
			}
		      else
			{
			  /* Check whether 'written' is lager than 'dstlim' */
			  if (dst != NULL
			      && written + curtable->mbcnum + curtable->esnum > dstlim)
			    goto exit_loop;

			  /* Copy the escape sequence if 'dst' isn't NULL */
			  if (dst != NULL)
			    for (i = 0; i < curtable->esnum; i++)
			      *dst++ = curtable->escseq[i];

			  /* Add the number of escape sequence to written. */
			  written += curtable->esnum;
			}

		      /* Copy the multibyte characters if 'dst' isn't NULL. */
		      if (dst != NULL)
			for (i = 0; i < curtable->mbcnum; i++)
			  *dst++ = p[i+1];

		      /* Add the number of multibyte character. */
		      written += curtable->mbcnum;
		      
		      /* Set the current table to LASTTABLE. */
		      lasttable = curtable;
		      
		      /* Exit this seek loop. */
		      goto exit_seek;

		    } /* end of "if (*p == ch)" */

		/* No in this table. Search next. */
		break;
	      } /* end of if (curupper->key == ch) */

	} /* end of "for (curtable = *table...) */

      /* This is an illegal encoding. */
      __set_errno(EILSEQ);

#if DEBUG
      fprintf(stderr,"dll_wcsnrtombs_locale:result = %d\n", -1);
#endif

      return (size_t)-1;
      
    exit_seek:
    }
 exit_loop:

  /* Store the current position of src */
  *src = run;

  /* Store the current shift state. */
  ps->count = (int)(lasttable - table_wcs2mbs);

  /* Return to main routine. */

#if DEBUG
  if (*src != NULL)
    fprintf(stderr,"*src=%p:%s\n", *src, *src);
  fprintf(stderr,"dll_wcsnrtombs_locale:result = %d\n", written);
#endif

  return written;
}
