/*
 * Grace - Graphics for Exploratory Data Analysis
 * 
 * Home page: http://plasma-gate.weizmann.ac.il/Grace/
 * 
 * Copyright (c) 1991-95 Paul J Turner, Portland, OR
 * Copyright (c) 1996-98 GRACE Development Team
 * 
 * Maintained by Evgeny Stambulchik <fnevgeny@plasma-gate.weizmann.ac.il>
 * 
 * 
 *                           All Rights Reserved
 * 
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 * 
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 * 
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>

/* for globals.h */
#define MAIN

#include "globals.h"
#include "patchlevel.h"

#include "utils.h"

#include "device.h"
#include "psdrv.h"
#include "mfdrv.h"
#ifdef NONE_GUI
#  include "dummydrv.h"
#else
#  include "x11drv.h"
#endif
#ifdef HAVE_LIBPDF         
#  include "pdfdrv.h"
#endif
#ifdef HAVE_LIBGD         
#  include "rstdrv.h"
#endif
#include "t1fonts.h"
#include "protos.h"

#ifdef NONE_GUI
char gui_version[] = "None";
#else
#  ifdef HAVE_LESSTIF
char gui_version[] = "Lesstif";
#  else
char gui_version[] = "Motif";
#  endif
#endif

extern char print_file[];
extern int install_cmap;

static void usage(FILE *stream, char *progname);

int inpipe = FALSE;		/* if xmgrace is to participate in a pipe */

#ifndef NONE_GUI
Device_entry dev_x11 = {DEVICE_TERM,
          "X11",
          xlibinitgraphics,
          NULL,
          "",
          FALSE,
          TRUE,
          {DEFAULT_PAGE_WIDTH, DEFAULT_PAGE_HEIGHT, 72.0, 72.0}
         };
#else
Device_entry dev_dummy = {DEVICE_TERM,
          "Dummy",
          dummyinitgraphics,
          NULL,
          "",
          TRUE,
          FALSE,
          {DEFAULT_PAGE_WIDTH, DEFAULT_PAGE_HEIGHT, 72.0, 72.0}
         };
#endif

Device_entry dev_ps = {DEVICE_PRINT,
          "PostScript",
          psprintinitgraphics,
          ps_gui_setup,
          "ps",
          TRUE,
          FALSE,
          {3300, 2550, 300.0, 300.0}
         };

Device_entry dev_eps = {DEVICE_FILE,
          "EPS",
          epsinitgraphics,
          NULL,
          "eps",
          TRUE,
          FALSE,
          {2500, 2500, 300.0, 300.0}
         };

Device_entry dev_mf = {DEVICE_FILE,
          "Metafile",
          mfinitgraphics,
          NULL,
          "gmf",
          FALSE,
          TRUE,
          {DEFAULT_PAGE_WIDTH, DEFAULT_PAGE_HEIGHT, 72.0, 72.0}
         };

#ifdef HAVE_LIBPDF         
Device_entry dev_pdf = {DEVICE_FILE,
          "PDF",
          pdfinitgraphics,
          pdf_gui_setup,
          "pdf",
          TRUE,
          FALSE,
          {612, 792, 72.0, 72.0}
         };
#endif

#ifdef HAVE_LIBGD         
Device_entry dev_gd = {DEVICE_FILE,
          "GD",
          gdinitgraphics,
          NULL,
          "gd",
          FALSE,
          TRUE,
          {DEFAULT_PAGE_WIDTH, DEFAULT_PAGE_HEIGHT, 72.0, 72.0}
         };
Device_entry dev_gif = {DEVICE_FILE,
          "GIF",
          gifinitgraphics,
          gif_gui_setup,
          "gif",
          FALSE,
          TRUE,
          {DEFAULT_PAGE_WIDTH, DEFAULT_PAGE_HEIGHT, 72.0, 72.0}
         };
Device_entry dev_pnm = {DEVICE_FILE,
          "PNM",
          pnminitgraphics,
          pnm_gui_setup,
          "pnm",
          FALSE,
          TRUE,
          {DEFAULT_PAGE_WIDTH, DEFAULT_PAGE_HEIGHT, 72.0, 72.0}
         };
#endif

int main(int argc, char *argv[])
{
    FILE *fp;
    int i, j;
    int gno;
    int cur_graph = cg = 0;	/* default graph is graph 0 */
    int loadlegend = 0;		/* legend on and load file names */
    int gcols = 1, grows = 1;
    int gracebat;		/* if executed as 'gracebat' then TRUE */
    int remove = FALSE;		/* remove file after read */
    int noprint = FALSE;	/* if gracebat, then don't print if true */
    extern char batchfile[];	/* name of file to execute */
    extern int density_flag;	/* temp, for density plots, defined in
				 * plotone.c */
#if defined(DEBUG)    
    extern int yydebug;
#endif

    char gracerc_file[GR_MAXPATHLEN], *s;
    
    Page_geometry pg;
    
    getcwd(currentdir, GR_MAXPATHLEN);	/* get the starting directory */

#ifndef GRACE_HOME
#  define GRACE_HOME "/usr/local/grace"
#endif

    /*
     * grace home directory
     */
    if ((s = getenv("GRACE_HOME")) != NULL) {
	strcpy(grace_home, s);
    } else {
        strcpy(grace_home, GRACE_HOME);
    }
    

#ifndef GRACE_HELPVIEWER
#  define GRACE_HELPVIEWER "netscape -noraise -remote openURL\\(%s,newwindow\\) >>/dev/null 2>&1 || netscape %s"
#endif

    /*
     * check for changed help file viewer command
     */
    if ((s = getenv("GRACE_HELPVIEWER")) != NULL) {
	strcpy(help_viewer, s);
    }   else {
        strcpy(help_viewer, GRACE_HELPVIEWER);
    }  


    /* initialize plots, strings, graphs */
    set_program_defaults();

    /* initialize the nonl-fit parameters */
    initialize_nonl();

    /* initialize the parser symbol table */
    init_symtab();
    
    /* initialize the rng */
    srand48(100L);

    /* initialize device, here tdevice is always 0 = Xlib */
    device = 0;
    
    plfile[0] = 0;		/* parameter file name */

    set_graph_active(cur_graph);

#ifndef NONE_GUI
    tdevice = register_device(dev_x11);
#else
    tdevice = register_device(dev_dummy);
#endif
    select_device(tdevice);

    hdevice = register_device(dev_ps);
    register_device(dev_eps);

    register_device(dev_mf);

#ifdef HAVE_LIBPDF
    register_device(dev_pdf);
#endif

#ifdef HAVE_LIBGD
    register_device(dev_gd);
    register_device(dev_gif);
    register_device(dev_pnm);
#endif

/* initialize the t1lib */
    if (init_t1() != GRACE_EXIT_SUCCESS) {
        errmsg("Initialization of t1-library failed!");
	exit (1);
    }

    /*
     * if program name is gracebat then don't initialize the X toolkit
     */
    s = mybasename(argv[0]);

    if (strcmp("gracebat", s) && strcmp("gracebat.exe", s)) {
	gracebat = FALSE;
#ifndef NONE_GUI    
        xlibprocess_args(&argc, argv);
#endif
    } else {
	gracebat = TRUE;
    }

    /* check for system wide startup file */
    if ((fp = fopen(strcat(strcpy(gracerc_file, grace_home),"/gracerc"), "r")) != NULL) {
	fclose(fp);
	getparms(gracerc_file);
    }
 
    /* check for startup file in local directory */
    if ((fp = fopen(".gracerc", "r")) != NULL) {
        fclose(fp);
        getparms(".gracerc");
    } else {
        /* check for startup file in effective users home dir */
        if ((s = getenv("HOME")) != NULL) {
            strcpy(gracerc_file, s);
            strcat(gracerc_file, "/.gracerc");
            if ((fp = fopen(gracerc_file, "r")) != NULL) {
        	fclose(fp);
        	getparms(gracerc_file);
            }
        } 
    }

    if (argc >= 2) {
	for (i = 1; i < argc; i++) {
	    if (argv[i][0] == '-') {
		if (argmatch(argv[i], "-version", 5)) {
	            fprintf(stdout, "Grace-%d.%d.%d %s, GUI: %s\n",  
		       MAJOR_REV, MINOR_REV, PATCHLEVEL, BETA_VER, gui_version);
	            fprintf(stdout, "(C) Copyright 1991-1995 Paul J Turner\n");
	            fprintf(stdout, "(C) Copyright 1996-1998 Grace Development Team\n");
	            fprintf(stdout, "All Rights Reserved\n");
	
		    exit(0);
		}
#if defined(DEBUG)
		if (argmatch(argv[i], "-debug", 6)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for debug flag\n");
			usage(stderr, argv[0]);
		    } else {
		    	debuglevel = atoi(argv[i]);
		    	if (debuglevel == 4) { 
			    /* turn on debugging in pars.y */
			    yydebug = TRUE;
		    	}
		    }
		} else
#endif
		       if (argmatch(argv[i], "-nosigcatch", 6)) {
		    sigcatch = 0;
		} else if (argmatch(argv[i], "-autoscale", 2)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for autoscale flag, should be one of 'x', 'y', 'xy'\n");
			usage(stderr, argv[0]);
		    } else {
			if (!strcmp("x", argv[i])) {
			    g[cur_graph].autoscale = 1;
			} else if (!strcmp("y", argv[i])) {
			    g[cur_graph].autoscale = 2;
			} else if (!strcmp("xy", argv[i])) {
			    g[cur_graph].autoscale = 3;
			} else {
			    fprintf(stderr, "%s: Improper argument for -a flag should be one of 'x', 'y', 'xy'\n", argv[0]);
			    usage(stderr, argv[0]);
			}
		    }
		} else if (argmatch(argv[i], "-noauto", 7)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for no autoscale flag, should be one of 'x', 'y', 'xy'\n");
			usage(stderr, argv[0]);
		    } else {
			if (!strcmp("x", argv[i])) {
			    g[cur_graph].noautoscale = 1;
			} else if (!strcmp("y", argv[i])) {
			    g[cur_graph].noautoscale = 2;
			} else if (!strcmp("xy", argv[i])) {
			    g[cur_graph].noautoscale = 3;
			} else {
			    fprintf(stderr, "%s: Improper argument for -noauto flag should be one of 'x', 'y', 'xy'\n", argv[0]);
			    usage(stderr, argv[0]);
			}
		    }
		} else if (argmatch(argv[i], "-batch", 2)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for batch file\n");
			usage(stderr, argv[0]);
		    } else {
			strcpy(batchfile, argv[i]);
		    }
		} else if (argmatch(argv[i], "-pipe", 5)) {
		    inpipe = TRUE;
/*
		    int flags = 0;
		    fcntl(0, F_SETFL, flags | O_NONBLOCK);
*/
		} else if (argmatch(argv[i], "-noprint", 8)) {
		    noprint = TRUE;
		} else if (argmatch(argv[i], "-logwindow", 5)) {
		    logwindow = TRUE;
		} else if (argmatch(argv[i], "-nologwindow", 7)) {
		    logwindow = FALSE;
		} else if (argmatch(argv[i], "-npipe", 6)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for named pipe\n");
			usage(stderr, argv[0]);
		    } else {
			strcpy(pipe_name, argv[i]);
			named_pipe = 1;
		    }
#if defined(HAVE_NETCDF) || defined(HAVE_MFHDF)
		} else if (argmatch(argv[i], "-netcdf", 7) || argmatch(argv[i], "-hdf", 4)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for netcdf file\n");
			usage(stderr, argv[0]);
		    } else {
			strcpy(netcdf_name, argv[i]);
			readcdf = 1;
		    }
		} else if (argmatch(argv[i], "-netcdfxy", 9) || argmatch(argv[i], "-hdfxy", 6)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for netcdf X variable name\n");
			usage(stderr, argv[0]);
		    } else {
			strcpy(xvar_name, argv[i]);
		    }
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for netcdf Y variable name\n");
			usage(stderr, argv[0]);
		    } else {
			strcpy(yvar_name, argv[i]);
		    }
		    if (strcmp(xvar_name, "null")) {
			readnetcdf(cg, -1, netcdf_name, xvar_name, yvar_name, -1, -1, 1);
		    } else {
			readnetcdf(cg, -1, netcdf_name, NULL, yvar_name, -1, -1, 1);
		    }
#endif				/* HAVE_NETCDF */
		} else if (argmatch(argv[i], "-timer", 6)) {
		    extern int timer_delay;	/* TODO move to globals.h */

		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for time delay\n");
			usage(stderr, argv[0]);
		    } else {
			timer_delay = atoi(argv[i]);
		    }
		} else if (argmatch(argv[i], "-maxsets", 8)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for max number of sets\n");
			usage(stderr, argv[0]);
		    } else {
			maxplot = atoi(argv[i]);
			realloc_plots(maxplot);
		    }
		} else if (argmatch(argv[i], "-maxblock", 9)) {
		    int itmp;			
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for max size of block data\n");
			usage(stderr, argv[0]);
		    } else {
			itmp = atoi(argv[i]);
			if (itmp < maxplot) {
			    itmp = maxplot;
			}
			alloc_blockdata(itmp);
		    }
		} else if (argmatch(argv[i], "-graphsets", 10)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for max number of sets for he current graph\n");
			usage(stderr, argv[0]);
		    } else {
			realloc_graph_plots(cur_graph, atoi(argv[i]));
		    }
		} else if (argmatch(argv[i], "-maxgraph", 8)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for max number of graphs\n");
			usage(stderr, argv[0]);
		    } else {
			maxgraph = atoi(argv[i]);
			realloc_graphs();
		    }
		} else if (argmatch(argv[i], "-maxlines", 8)) {
		    int itmp;
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for max number of lines\n");
			usage(stderr, argv[0]);
		    } else {
			itmp = atoi(argv[i]);
			realloc_lines(itmp);
		    }
		} else if (argmatch(argv[i], "-maxboxes", 8)) {
		    int itmp;
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for max number of boxes\n");
			usage(stderr, argv[0]);
		    } else {
			itmp = atoi(argv[i]);
			realloc_boxes(itmp);
		    }
		} else if (argmatch(argv[i], "-maxellipses", 12)) {
		    int itmp;
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for max number of ellipses\n");
			usage(stderr, argv[0]);
		    } else {
			itmp = atoi(argv[i]);
			realloc_ellipses(itmp);
		    }
		} else if (argmatch(argv[i], "-maxstr", 7)) {
		    int itmp;
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for max number of strings\n");
			usage(stderr, argv[0]);
		    } else {
			itmp = atoi(argv[i]);
			realloc_strings(itmp);
		    }
#ifndef NONE_GUI
		} else if (argmatch(argv[i], "-install", 7)) {
		    install_cmap = CMAP_INSTALL_ALWAYS;
		} else if (argmatch(argv[i], "-noinstall", 9)) {
		    install_cmap = CMAP_INSTALL_NEVER;
#endif
		} else if (argmatch(argv[i], "-timestamp", 10)) {
		    timestamp.active = TRUE;
		} else if (argmatch(argv[i], "-remove", 7)) {
		    remove = TRUE;
		} else if (argmatch(argv[i], "-fixed", 5)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for fixed canvas width\n");
			usage(stderr, argv[0]);
		    } else {
		        if (i == argc - 1) {
			    fprintf(stderr, "Missing argument for fixed canvas height\n");
			    usage(stderr, argv[0]);
		        } else {
		            pg = get_page_geometry();
                            pg.width = atoi(argv[i]);
		    	    i++;
		    	    pg.height = atoi(argv[i]);
                            set_page_geometry(pg);
		    	    page_layout = PAGE_FIXED;
			}
		    }
		} else if (argmatch(argv[i], "-free", 5)) {
		    page_layout = PAGE_FREE;
		} else if (argmatch(argv[i], "-noask", 5)) {
		    noask = TRUE;
/*
 * 		} else if (argmatch(argv[i], "-font_aa", 8)) {
 * 		    pg.fontaa = TRUE;
 * 		} else if (argmatch(argv[i], "-nofont_aa", 10)) {
 * 		    pg.fontaa = FALSE;
 */
		} else if (argmatch(argv[i], "-mono", 5)) {
		    monomode = TRUE;
		} else if (argmatch(argv[i], "-dc", 3)) {
		    allow_dc = 1;
		} else if (argmatch(argv[i], "-nodc", 5)) {
		    allow_dc = 0;
		} else if (argmatch(argv[i], "-redraw", 7)) {
		    auto_redraw = 1;
		} else if (argmatch(argv[i], "-noredraw", 9)) {
		    auto_redraw = 0;
		} else if (argmatch(argv[i], "-statusredraw", 7)) {
		    status_auto_redraw = 1;
		} else if (argmatch(argv[i], "-nostatusredraw", 9)) {
		    status_auto_redraw = 0;
		} else if (argmatch(argv[i], "-GXxor", 6)) {
		    invert = 0;
		} else if (argmatch(argv[i], "-GXinvert", 6)) {
		    invert = 1;
		} else if (argmatch(argv[i], "-hdevice", 5)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for hardcopy device select flag\n");
			usage(stderr, argv[0]);
		    } else {
		        if (set_printer_by_name(argv[i]) != GRACE_EXIT_SUCCESS) {
                            errmsg("Unknown or unsupported device");
                            exit(1);
                        }
                    }
		} else if (argmatch(argv[i], "-log", 2)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for log plots flag\n");
			usage(stderr, argv[0]);
		    }
		    if (!strcmp("x", argv[i])) {
			g[cur_graph].xscale = SCALE_LOG;
		    } else if (!strcmp("y", argv[i])) {
			g[cur_graph].yscale = SCALE_LOG;
		    } else if (!strcmp("xy", argv[i])) {
			g[cur_graph].xscale = SCALE_LOG;
			g[cur_graph].yscale = SCALE_LOG;
		    } else {
			fprintf(stderr, "%s: Improper argument for -l flag; should be one of 'x', 'y', 'xy'\n", argv[0]);
		    }
		} else if (argmatch(argv[i], "-printfile", 6)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing file name for printing\n");
			usage(stderr, argv[0]);
		    } else {
			ptofile = TRUE;
                        strcpy(print_file, argv[i]);
		    }
		} else if (argmatch(argv[i], "-hardcopy", 6)) {
		    gracebat = TRUE;
		} else if (argmatch(argv[i], "-pexec", 6)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for exec\n");
			usage(stderr, argv[0]);
		    } else {
			char pstring[MAX_STRING_LENGTH];
			int ilen;

			strcpy(pstring, argv[i]);
			ilen = strlen(pstring);
			pstring[ilen] = '\n';
			pstring[ilen + 1] = 0;
			read_param(pstring);
		    }
		} else if (argmatch(argv[i], "-graph", 6)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing parameter for graph select\n");
			usage(stderr, argv[0]);
		    } else {
			sscanf(argv[i], "%d", &gno);
			if (gno >= 0 && gno < maxgraph) {
			    cg = cur_graph = gno;
			    set_graph_active(gno);
			} else {
			    fprintf(stderr, "Graph number must be between 0 and %d\n", maxgraph - 1);
			}
		    }
		} else if (argmatch(argv[i], "-block", 6)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing parameter for block data\n");
			usage(stderr, argv[0]);
		    } else {
			if (getdata(cur_graph, argv[i], cursource, SET_BLOCK)) {
			}
		    }
		} else if (argmatch(argv[i], "-bxy", 4)) {
		    char blocksetcols[32];
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing parameter for block data set creation\n");
		    }
		    strcpy(blocksetcols, argv[i]);
		    create_set_fromblock(cg, curtype, blocksetcols);
		} else if (argmatch(argv[i], "-xy", 3)) {
		    curtype = SET_XY;
		} else if (argmatch(argv[i], "-poly", 5)) {
		    curtype = SET_POLY;
		} else if (argmatch(argv[i], "-xydx", 5)) {
		    curtype = SET_XYDX;
		} else if (argmatch(argv[i], "-xydy", 5)) {
		    curtype = SET_XYDY;
		} else if (argmatch(argv[i], "-xydxdx", 7)) {
		    curtype = SET_XYDXDX;
		} else if (argmatch(argv[i], "-xydydy", 7)) {
		    curtype = SET_XYDYDY;
		} else if (argmatch(argv[i], "-xydxdy", 7)) {
		    curtype = SET_XYDXDY;
		} else if (argmatch(argv[i], "-xyz", 4)) {
		    curtype = SET_XYZ;
		} else if (argmatch(argv[i], "-xyd", 4)) {
		    curtype = SET_XYZ;
		    density_flag = TRUE;
		} else if (argmatch(argv[i], "-xyrt", 4)) {
		    curtype = SET_XYRT;
		} else if (argmatch(argv[i], "-nxy", 4)) {
		    curtype = SET_NXY;
		} else if (argmatch(argv[i], "-hilo", 5)) {
		    curtype = SET_XYHILO;
		} else if (argmatch(argv[i], "-xystring", 9)) {
		    curtype = SET_XYSTRING;
		} else if (argmatch(argv[i], "-type", 2)) {
		    /* other file types here */
		    i++;
                    if (argmatch(argv[i], "poly", 4)) {
			curtype = SET_POLY;
		    } else if (argmatch(argv[i], "xydx", 4)) {
			curtype = SET_XYDX;
		    } else if (argmatch(argv[i], "xydy", 4)) {
			curtype = SET_XYDY;
		    } else if (argmatch(argv[i], "xydxdx", 6)) {
			curtype = SET_XYDXDX;
		    } else if (argmatch(argv[i], "xydydy", 6)) {
			curtype = SET_XYDYDY;
		    } else if (argmatch(argv[i], "xydxdy", 6)) {
			curtype = SET_XYDXDY;
		    } else if (argmatch(argv[i], "xyz", 3)) {
			curtype = SET_XYZ;
		    } else if (argmatch(argv[i], "xyr", 3)) {
			curtype = SET_XYRT;
		    } else if (argmatch(argv[i], "hilo", 4)) {
			curtype = SET_XYHILO;
		    } else {
			fprintf(stderr, "%s: Unknown file type '%s' assuming XY\n", argv[0], argv[i]);
			curtype = SET_XY;
		    }
		} else if (argmatch(argv[i], "-graphtype", 7)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for graph type\n");
		    } else {
			if (!strcmp("xy", argv[i])) {
			    g[cur_graph].type = GRAPH_XY;
			} else if (!strcmp("polar", argv[i])) {
			    g[cur_graph].type = GRAPH_POLAR;
			} else if (!strcmp("bar", argv[i])) {
			    g[cur_graph].type = GRAPH_CHART;
			} else if (!strcmp("smith", argv[i])) {
			    g[cur_graph].type = GRAPH_SMITH;
			    g[cur_graph].w.xg1 = -1.0;
			    g[cur_graph].w.yg1 = -1.0;
			    g[cur_graph].w.xg2 = 1.0;
			    g[cur_graph].w.yg2 = 1.0;
			} else {
			    fprintf(stderr, "%s: Improper argument for -graphtype\n", argv[0]);
			    usage(stderr, argv[0]);
			}
		    }
		} else if (argmatch(argv[i], "-arrange", 7)) {
		    if (i + 2 >= argc) {
			fprintf(stderr, "Missing argument for graph arrangement\n");
			usage(stderr, argv[0]);
		    } else {
			i++;
			grows = atoi(argv[i]);
			i++;
			gcols = atoi(argv[i]);
		    }
		} else if (argmatch(argv[i], "-cols", 5)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for graph column arrangement\n");
			usage(stderr, argv[0]);
		    } else {
			gcols = atoi(argv[i]);
		    }
		} else if (argmatch(argv[i], "-rows", 5)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for graph row arrangement\n");
			usage(stderr, argv[0]);
		    } else {
			grows = atoi(argv[i]);
		    }
		} else if (argmatch(argv[i], "-legend", 4)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for -legend\n");
			usage(stderr, argv[0]);
		    } else {
			if (!strcmp("load", argv[i])) {
			    loadlegend = TRUE;
			    g[cur_graph].l.active = TRUE;
			} else {
			    fprintf(stderr, "Improper argument for -legend\n");
			    usage(stderr, argv[0]);
			}
		    }
		} else if (argmatch(argv[i], "-rvideo", 7)) {
		    revflag = TRUE;
		} else if (argmatch(argv[i], "-param", 2)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing parameter file name\n");
			usage(stderr, argv[0]);
		    } else {
			strcpy(plfile, argv[i]);
			if (!getparms(plfile)) {
			    g[cur_graph].parmsread = FALSE;
			    fprintf(stderr, "Unable to read parameter file %s\n", plfile);
			} else {
			    g[cur_graph].parmsread = TRUE;
			}
		    }
		} else if (argmatch(argv[i], "-results", 2)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing results file name\n");
			usage(stderr, argv[0]);
		    } else {
			strcpy(resfile, argv[i]);
/*
*  open resfile if -results option given
*/
			if (!fexists(resfile)) {
			    if ((resfp = filter_write(resfile)) == NULL) {
				fprintf(stderr, "Unable to open file %s", resfile);
				exit(1);
			    }
			}
		    }
		} else if (argmatch(argv[i], "-saveall", 8)) {
		    char savefile[GR_MAXPATHLEN];
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing save file name\n");
			usage(stderr, argv[0]);
		    } else {
			strcpy(savefile, argv[i]);
			do_writesets(maxgraph, -1, 1, savefile, sformat);
		    }
		} else if (argmatch(argv[i], "-wd", 3)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing parameters for working directory\n");
			usage(stderr, argv[0]);
		    } else {
			if (isdir(argv[i])) {
			    strcpy(buf, argv[i]);
			    if (chdir(buf) == 0) {
					getcwd(workingdir, 512);
					if( workingdir[strlen(workingdir)-1] != '/' )
						strcat( workingdir, "/" );
			    } else {
					fprintf(stderr, "Can't change to directory %s, fatal error",
																buf);
					exit(1);
			    }
			} else {
			    fprintf(stderr, "%s is not a directory, fatal error\n",argv[i]);
			    exit(1);
			}
		    }
		} else if (argmatch(argv[i], "-source", 2)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing argument for data source parameter\n");
			usage(stderr, argv[0]);
		    }
		    if (argmatch(argv[i], "pipe", 4)) {
			cursource = SOURCE_PIPE;
		    } else if (argmatch(argv[i], "disk", 4)) {
			cursource = SOURCE_DISK;
		    } else if (argmatch(argv[i], "stdin", 5)) {
			cursource = SOURCE_STDIN;
		    }
		    /* we are in a pipe */
		    if (cursource == SOURCE_STDIN) {
			if (getdata(cur_graph, "STDIN", SOURCE_STDIN, curtype)) {
			}
		    }
		} else if (argmatch(argv[i], "-viewport", 2)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing parameters for viewport setting\n");
			usage(stderr, argv[0]);
		    } else {
			g[cur_graph].v.xv1 = atof(argv[i++]);
			g[cur_graph].v.yv1 = atof(argv[i++]);
			g[cur_graph].v.xv2 = atof(argv[i++]);
			g[cur_graph].v.yv2 = atof(argv[i]);
		    }
		} else if (argmatch(argv[i], "-world", 2)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing parameters for world setting\n");
			usage(stderr, argv[0]);
		    } else {
			g[cur_graph].w.xg1 = atof(argv[i++]);
			g[cur_graph].w.yg1 = atof(argv[i++]);
			g[cur_graph].w.xg2 = atof(argv[i++]);
			g[cur_graph].w.yg2 = atof(argv[i]);
		    }
		} else if (argmatch(argv[i], "-seed", 5)) {
		    i++;
		    if (i == argc) {
			fprintf(stderr, "Missing seed for srand48()\n");
			usage(stderr, argv[0]);
		    } else {
			srand48(atol(argv[i]));	/* note atol() */
		    }
		} else if (argmatch(argv[i], "-symcolorbug", 12)) {
		    symcolorbug = 1;
		} else if (argmatch(argv[i], "-help", 2)) {
		    usage(stdout, argv[0]);
		} else if (argmatch(argv[i], "-usage", 5)) {
		    usage(stdout, argv[0]);
		} else {
		    fprintf(stderr, "No option %s\n", argv[i]);
		    usage(stderr, argv[0]);
		}
	    } else {
		if (i != argc) {
		    lock_dirtystate();
		    if (getdata(cur_graph, argv[i], cursource, curtype)) {
			strcpy(docname, argv[i]);
			if (remove) {
			    unlink(argv[i]);
			}
			clear_dirtystate();
		    } else {
		        clear_dirtystate();
			set_dirtystate();
		    }
		}		/* end if */
	    }			/* end else */
	}			/* end for */
    }				/* end if */
    
    
    for (i = 0; i < maxgraph; i++) {
	if (isactive_graph(i) && (activeset(i))) {
	    if (g[i].parmsread == FALSE) {
		switch (g[i].noautoscale) {
		case 0:
		    autoscale_set(i, ALL_SETS, ALL_AXES);
                case 1:
		    autoscale_set(i, ALL_SETS, ALL_X_AXES);
		    break;
		case 2:
		    autoscale_set(i, ALL_SETS, ALL_Y_AXES);
                    break;
		case 3:
		    break;
		}
	    }
/*
 * if auto scaling type set with -a option, then scale appropriate axis
 */
	    else {
	        switch (g[i].autoscale) {
	        case 1:
	            autoscale_set(i, ALL_SETS, ALL_X_AXES);
	            break;
	        case 2:
	            autoscale_set(i, ALL_SETS, ALL_Y_AXES);
	            break;
	        case 3:
	            autoscale_set(i, ALL_SETS, ALL_AXES);
	            break;
	        }
	    }
	}
    }				/* end for */

    /*
     * Process events.
     */
    if (sigcatch) {
        installSignal();
    }

/*
 * load legend
 */
    if (loadlegend) {
	for (i = 0; i < maxgraph; i++) {
	    if (isactive_graph(i) && (activeset(i))) {
		for (j = 0; j < maxplot; j++) {
		    if (isactive_set(i, j)) {
			strcpy(g[i].p[j].lstr, g[i].p[j].comments);
		    }
		}
	    }
	}
    }
/*
 * straighten our cursource if a pipe was used
 */
    if (cursource == 2) {
	cursource = SOURCE_DISK;
    }
/*
 * arrange graphs if grows & gcols set
 */
    arrange_graphs(grows, gcols);

/* 
 * initialize colormap data
 */
    initialize_cmap();
        
/*
 * initialize the world and viewport
 */
    select_graph(cg);

/*
 *     if (monomode == TRUE) {
 *         if (get_fontaa() ==  TRUE) {
 * 	    errmsg("Font antialiasing not available in the mono mode");
 * 	    set_fontaa(FALSE);
 * 	}
 *     }
 */
    

/*
 * if -h on command line or executed as gracebat just plot the graph and quit
 */
    if (gracebat == TRUE) {
	if (hdevice == 0) {
	    fprintf(stderr,
		    "%s: terminal device can't be used for batch plotting\n", argv[0]);
	    exit(1);
	}
	if (inpipe) {
	    getdata(cg, "STDIN", SOURCE_STDIN, SET_XY);
	    inpipe = 0;
	}
	if (batchfile[0]) {
	    getparms(batchfile);
	}
	if (!noprint) {
	    do_hardcopy();
	}
	if (resfp) {
	    fclose(resfp);
	}
	exit(0);
    }
    
/*
 * go main loop
 */
#ifndef NONE_GUI
    if (gracebat == FALSE) {
        initialize_screen();
    }
#endif
    
    do_main_loop();
    return 0;
}

static void usage(FILE *stream, char *progname)
{
/* We use alphabetial order */

    fprintf(stream, "Usage of %s command line arguments: \n", progname);

    fprintf(stream, "-arrange   [rows] [cols]              Arrange the graphs in a grid rows by cols\n");
    fprintf(stream, "-autoscale [x|y|xy]                   Override any parameter file settings\n");
    fprintf(stream, "-batch     [batch_file]               Execute batch_file on start up\n");
    fprintf(stream, "-block     [block_data]               Assume data file is block data\n");
    fprintf(stream, "-boxplot   [boxplot_file]             Assume data is in X MEDIAN Y1 Y2 Y3 Y4\n");
    fprintf(stream, "                                        format\n");
    fprintf(stream, "-bxy       [x:y:etc.]                 Form a set from the current block data set\n");
    fprintf(stream, "                                        using the current set type from columns\n");
    fprintf(stream, "                                        given in the argument\n");
    fprintf(stream, "-cols      [cols]\n");
    fprintf(stream, "-dc                                   Allow double click operations on the\n");
    fprintf(stream, "                                        canvas\n");
#if defined(DEBUG)
    fprintf(stream, "-debug     [debug_level]              Set debugging options\n");
#endif
    fprintf(stream, "-hdevice   [hardcopy_device_name]     Set default hardcopy device\n");
    fprintf(stream, "-fixed     [width] [height]           Set canvas size fixed to width*height\n");
    fprintf(stream, "-free                                 Use free page layout\n");
    fprintf(stream, "-graph     [graph_number]             Set the current graph number\n");
    fprintf(stream, "-graphsets [number_of_sets]           Set the number of data sets for the\n");
    fprintf(stream, "                                        current graph\n");
    fprintf(stream, "-graphtype [graph_type]               Set the type of the current graph\n");
    fprintf(stream, "-GXxor                                Use xor to draw rubberband lines and graph\n");
    fprintf(stream, "                                        focus markers\n");
    fprintf(stream, "-GXinvert                             Use invert to draw rubberband lines and\n");
    fprintf(stream, "                                        graph focus markers\n");
    fprintf(stream, "-hardcopy                             No interactive session, just print and\n");
    fprintf(stream, "                                        quit\n");
    fprintf(stream, "-hilo      [hilo_file]                Assume data is in X HI LO OPEN CLOSE\n");
    fprintf(stream, "                                        format\n");
    fprintf(stream, "-install                              Install private colormap\n");
    fprintf(stream, "-legend    [load]                     Turn the graph legend on\n");
    fprintf(stream, "-log       [x|y|xy]                   Set the graph type to logarithmic\n");
    fprintf(stream, "-logwindow                            Open the log window\n");
    fprintf(stream, "-maxblock  [number_of_columns]        Set the number of columns for block data\n");
    fprintf(stream, "                                        (default is %d)\n", MAXPLOT);
    fprintf(stream, "-maxboxes  [number_of_boxes]          Set the number of boxes\n");
    fprintf(stream, "-maxellipses  [number_of_ellipses]    Set the number of ellipses\n");
    fprintf(stream, "-maxgraph  [number_of_graphs]         Set the number of graphs for this session\n");
    fprintf(stream, "                                        (minimum is 10)\n");
    fprintf(stream, "-maxlines  [number_of_lines]          Set the number of lines\n");
    fprintf(stream, "-maxsets   [number_of_sets]           Set the number of data sets per graph\n");
    fprintf(stream, "                                        (minimum is 30)\n");
    fprintf(stream, "-maxstr    [number_of_strings]        Set the number of strings \n");
    fprintf(stream, "-mono                                 Run %s in monochrome mode (affects\n", progname);
    fprintf(stream, "                                        the display only)\n");
#if defined(HAVE_NETCDF) || defined(HAVE_MFHDF)
    fprintf(stream, "-netcdf    [netcdf file]              Assume data file is bnetCDF format\n");
    fprintf(stream, "-netcdfxy  [X var name] [Y var name]  If -netcdf was used previously, read from\n");
    fprintf(stream, "                                        the netCDF file, 'X var name' and 'Y\n");
    fprintf(stream, "                                        var name' and create a set. If 'X var\n");
    fprintf(stream, "                                        name' equals \"null\" then load the\n");
    fprintf(stream, "                                        index of Y to X\n");
#endif
    fprintf(stream, "-noask                                Assume the answer is yes to all requests -\n");
    fprintf(stream, "                                        if the operation would overwrite a file,\n");
    fprintf(stream, "                                        grace will do so without prompting\n");
    fprintf(stream, "-noauto    [x|y|xy]                   Supress autoscaling for the specified axis\n");
    fprintf(stream, "-nodc                                 Ignore double clicks on the canvas\n");
    fprintf(stream, "-noinstall                            Don't use private colormap\n");
    fprintf(stream, "-nologwindow                          No log window, overrides resource setting\n");
    fprintf(stream, "-noprint                              In batch mode, do not print\n");
    fprintf(stream, "-noredraw                             Don't do a redraw for refreshing the\n");
    fprintf(stream, "                                        canvas when the server doesn't do\n");
    fprintf(stream, "                                        backing store\n");
    fprintf(stream, "-nosigcatch                           Don't catch signals\n");
    fprintf(stream, "-nostatusredraw                       Don't do a redraw after each action in the\n");
    fprintf(stream, "                                        status window\n");
    fprintf(stream, "-npipe     [file]                     Read data from named pipe on startup\n");
    fprintf(stream, "-nxy       [nxy_file]                 Assume data file is in X Y1 Y2 Y3 ...\n");
    fprintf(stream, "                                        format\n");
    fprintf(stream, "-param     [parameter_file]           Load parameters from parameter_file to the\n");
    fprintf(stream, "                                        current graph\n");
    fprintf(stream, "-pexec     [parameter_string]         Interpret string as a parameter setting\n");
    fprintf(stream, "-pipe                                 Read data from stdin on startup\n");
    fprintf(stream, "-printfile [file for hardcopy output] Save print output to file \n");
    fprintf(stream, "-redraw                               Do a redraw for refreshing the canvas when\n");
    fprintf(stream, "                                        the server doesn't do backing store\n");
    fprintf(stream, "-remove                               Remove data file after read\n");
    fprintf(stream, "-results   [results_file]             Write the results from regression to\n");
    fprintf(stream, "                                        results_file\n");
    fprintf(stream, "-rows      [grows]                    Arrange graphs in grows rows \n");
    fprintf(stream, "-rvideo                               Exchange the color indices for black and\n");
    fprintf(stream, "                                        white\n");
    fprintf(stream, "-saveall   [save_file]                Save all graphs to save_file\n");
    fprintf(stream, "-seed      [seed_value]               Integer seed for random number generator\n");
    fprintf(stream, "-source    [disk|pipe|stdin]          Source of next data file\n");
    fprintf(stream, "-statusredraw                         Do a redraw after each action in the\n");
    fprintf(stream, "                                        status window\n");
    fprintf(stream, "-symcolorbug                          Compatibility flag for xmgr 3.x projects\n");
    fprintf(stream, "-timer     [delay]                    Set timer for named pipes to delay m \n");
    fprintf(stream, "-timestamp \n");
    fprintf(stream, "-type      [xy|xydx|xydy|xydxdx|xydydy|hilo] Set the type of the next data file\n");
    fprintf(stream, "-version                              Show the program version\n");
    fprintf(stream, "-viewport  [xmin ymin xmax ymax]      Set the viewport for the current graph\n");
    fprintf(stream, "-wd        [directory]                Set the working directory\n");
    fprintf(stream, "-world     [xmin ymin xmax ymax]      Set the user coordinate system for the\n");
    fprintf(stream, "                                        current graph\n");
    fprintf(stream, "-xy        [xy_file]                  Assume data file is in X Y format - sets\n");
    fprintf(stream, "                                        are separated by lines containing non-\n");
    fprintf(stream, "                                        numeric data\n");
    fprintf(stream, "-xyd       [xyd_file]                 Assume data file is in X Y density format\n");
    fprintf(stream, "-xydx      [xydx_file]                Assume data file is in X Y DX format\n");
    fprintf(stream, "-xydxdx    [xydxdx_file]              Assume data file is in X Y DX1 DX2 format\n");
    fprintf(stream, "-xydxdy    [xydxdy_file]              Assume data file is in X Y DX DY format\n");
    fprintf(stream, "-xydy      [xydy_file]                Assume data file is in X Y DY format\n");
    fprintf(stream, "-xydydy    [xydydy_file]              Assume data file is in X Y DY1 DY2 format\n");
    fprintf(stream, "-xyr       [xyr_file]                 Assume data file is in X Y RADIUS format\n");
    fprintf(stream, "-xyz       [xyz_file]                 Assume data file is in X Y Z format\n");

    fprintf(stream, "-usage|-help                          This message\n");
    fprintf(stream, "\n");
    fprintf(stream, " ** If it scrolls too fast, run `%s -help | more\' **\n", progname);
    exit(0);
}

/*
 * command loop
 */
void do_main_loop(void)
{
#ifdef NONE_GUI
    char pstring[MAX_STRING_LENGTH];
    int ilen;
    int i = 1;
    
    while (TRUE) {
        printf("grace:%d> ", i);
        fgets(pstring, MAX_STRING_LENGTH - 1, stdin);
        ilen = strlen(pstring);
        if (ilen < 2) {
            continue;
        }
        read_param(pstring);
        i++;
    }
#else
    do_main_winloop();
#endif
}
