/* ncftpbatch.c
 * 
 * Copyright (c) 1992-1998 by Mike Gleason, NCEMRSoft.
 * All rights reserved.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 * 
 */

#include "syshdrs.h"

#ifdef HAVE_LONG_FILE_NAMES

#define kSpoolDir "spool"
#define kSpoolLogName "batchlog"
#define kFirewallPrefFileName "firewall"

#include "../ncftp/util.h"

int gGotSig = 0;
FTPLibraryInfo gLib;
FTPConnectionInfo gConn;
int gIsTTY;
int gSpoolSerial = 0;
int gSpooled = 0;
char gSpoolDir[256];
int gFirewallType;
char gFirewallHost[64];
char gFirewallUser[32];
char gFirewallPass[32];
char gFirewallExceptionList[256];
unsigned int gFirewallPort;
int gItemInUse = 0;
char gItemPath[256];
char gMyItemPath[256];
int gOperation;
char gHost[64];
char gHostIP[32];
unsigned int gPort;
char gRUser[32];
char gRPass[128];
int gXtype;
int gRecursive;
int gDelete;
int gPassive;
char gRDir[256];
char gLDir[256];
char gRFile[256];
char gLFile[256];

/* Writes logging data to a ~/.ncftp/batchlog file.
 * This is nice for me when I need to diagnose problems.
 */
time_t gLogTime;
FILE *gLogFile = NULL;
char gLogLBuf[256];
int gMyPID;
const char *gLogOpenMode = "a";
int gUnused;
int gMayCancelJmp = 0;

#ifdef HAVE_SIGSETJMP
sigjmp_buf gCancelJmp;
#else	/* HAVE_SIGSETJMP */
jmp_buf gCancelJmp;
#endif	/* HAVE_SIGSETJMP */

extern char gOurDirectoryPath[256];
extern int FTPRebuildConnectionInfo(const FTPLIPtr lip, const FTPCIPtr cip);
extern char *optarg;
extern int optind;
extern char gFirewallExceptionList[256];
extern char gOS[], gVersion[];



/*VARARGS*/
static void
Log(char *fmt, ...)
{
	va_list ap;
	struct tm *ltp;

	if (gLogFile != NULL) {
		(void) time(&gLogTime);
		ltp = localtime(&gLogTime);
		if (ltp != NULL) {
			(void) fprintf(gLogFile , "%06d %04d-%02d-%02d %02d:%02d:%02d | ",
				gMyPID,
				ltp->tm_year + 1900,
				ltp->tm_mon + 1,
				ltp->tm_mday,
				ltp->tm_hour,
				ltp->tm_min,
				ltp->tm_sec
			);
		}
		va_start(ap, fmt);
		(void) vfprintf(gLogFile, fmt, ap);
		va_end(ap);
	}
	if (gIsTTY != 0) {
		va_start(ap, fmt);
		(void) vfprintf(stdout, fmt, ap);
		va_end(ap);
	}
}	/* Log */



/*VARARGS*/
static void
LogPerror(char *fmt, ...)
{
	va_list ap;
	struct tm *ltp;
	int oerrno;
	
	oerrno = errno;
	if (gLogFile != NULL) {
		(void) time(&gLogTime);
		ltp = localtime(&gLogTime);
		if (ltp != NULL) {
			(void) fprintf(gLogFile , "%06d %04d-%02d-%02d %02d:%02d:%02d | ",
				gMyPID,
				ltp->tm_year + 1900,
				ltp->tm_mon + 1,
				ltp->tm_mday,
				ltp->tm_hour,
				ltp->tm_min,
				ltp->tm_sec
			);
		}
		va_start(ap, fmt);
		(void) vfprintf(gLogFile, fmt, ap);
		va_end(ap);
#ifdef HAVE_STRERROR
		(void) fprintf(gLogFile, ": %s\n", strerror(oerrno));
#else
		(void) fprintf(gLogFile, ": errno=%d\n", (oerrno));
#endif
	}
	if (gIsTTY != 0) {
		va_start(ap, fmt);
		(void) vfprintf(stdout, fmt, ap);
		va_end(ap);
#ifdef HAVE_STRERROR
		(void) fprintf(stdout, ": %s\n", strerror(oerrno));
#else
		(void) fprintf(stdout, ": errno=%d\n", (oerrno));
#endif
	}
}	/* LogPerror */




static void
DebugHook(const FTPCIPtr cipUnused, char *msg)
{
	gUnused = (int) cipUnused;			/* shut up gcc */
	Log("%s", msg);
}	/* DebugHook */




static void
CloseLog(void)
{
	if (gLogFile != NULL) {
		(void) fclose(gLogFile);
		gLogFile = NULL;
	}
}	/* CloseLog */




static void
OpenLog(void)
{
	FILE *fp;
	char pathName[256];

	CloseLog();
	(void) OurDirectoryPath(pathName, sizeof(pathName), kSpoolLogName);

	fp = fopen(pathName, gLogOpenMode);
	if (fp != NULL) {
#ifdef HAVE_SETVBUF
		(void) setvbuf(fp, gLogLBuf, _IOLBF, sizeof(gLogLBuf));
#endif	/* HAVE_SETVBUF */
		(void) time(&gLogTime);
		gLogFile = fp;
		gMyPID = (int) getpid();
	}
}	/* OpenLog */




static void
ExitStuff(void)
{
	if (gItemInUse > 0) {
		gItemInUse = 0;
		(void) rename(gMyItemPath, gItemPath);
	}
}	/* ExitStuff */




static void
SigAlrm(int sigNum)
{
	if (gMayCancelJmp != 0) {
		gUnused = sigNum;
		if (gItemInUse > 0) {
			gItemInUse = 0;
			(void) rename(gMyItemPath, gItemPath);
		}
#ifdef HAVE_SIGSETJMP
		siglongjmp(gCancelJmp, 1);
#else	/* HAVE_SIGSETJMP */
		longjmp(gCancelJmp, 1);
#endif	/* HAVE_SIGSETJMP */
	}
}	/* SigExit */




static void
SigExit(int sigNum)
{
	ExitStuff();
	Log("-----caught signal %d, exiting-----\n", sigNum);
	exit(0);
}	/* SigExit */




/* Load those options specific to the firewall/proxy settings.  These are
 * kept in a different file so that other programs can read it and not
 * have to worry about the other junk in the prefs file.
 */
static void
LoadFirewallPrefs(void)
{
	FILE *fp;
	char pathName[256];
	char line[256];
	char *tok1, *tok2;
	int n;
	char *cp;

	(void) OurDirectoryPath(pathName, sizeof(pathName), kFirewallPrefFileName);

	/* Set default values. */
	gFirewallType = kFirewallNotInUse;
	gFirewallPort = 0;
	gFirewallHost[0] = '\0';
	gFirewallUser[0] = '\0';
	gFirewallPass[0] = '\0';
	gFirewallExceptionList[0] = '\0';

	fp = fopen(pathName, "r");
	if (fp != NULL) {
		/* Opened the firewall preferences file. */
		line[sizeof(line) - 1] = '\0';
		while (fgets(line, sizeof(line) - 1, fp) != NULL) {
			tok1 = strtok(line, " =\t\r\n");
			if ((tok1 == NULL) || (tok1[0] == '#'))
				continue;
			tok2 = strtok(NULL, "\r\n");
			if (tok2 == NULL)
				continue;
			if (ISTREQ(tok1, "firewall-type")) {
				n = atoi(tok2);
				if ((n > 0) && (n <= kFirewallLastType))
					gFirewallType = n;
			} else if (ISTREQ(tok1, "firewall-host")) {
				(void) STRNCPY(gFirewallHost, tok2);
			} else if (ISTREQ(tok1, "firewall-port")) {
				n = atoi(tok2);
				if (n > 0)
					gFirewallPort = (unsigned int) n;
			} else if (ISTREQ(tok1, "firewall-user")) {
				(void) STRNCPY(gFirewallUser, tok2);
			} else if (ISTREQ(tok1, "firewall-pass")) {
				(void) STRNCPY(gFirewallPass, tok2);
			} else if (ISTREQ(tok1, "firewall-password")) {
				(void) STRNCPY(gFirewallPass, tok2);
			} else if (ISTREQ(tok1, "firewall-exception-list")) {
				(void) STRNCPY(gFirewallExceptionList, tok2);
			}
		}
		(void) fclose(fp);
	}

	if (gFirewallExceptionList[0] == '\0') {
		FTPInitializeOurHostName(&gLib);
		cp = strchr(gLib.ourHostName, '.');

		if (cp != NULL) {
			(void) STRNCPY(gFirewallExceptionList, cp);
			(void) STRNCAT(gFirewallExceptionList, ",localdomain");
		}
	}
}	/* LoadFirewallPrefs */




static void
FTPInit(void)
{
	int result;

	result = FTPInitLibrary(&gLib);
	if (result < 0) {
		(void) fprintf(stderr, "ncftpbatch: init library error %d (%s).\n", result, FTPStrError(result));
		exit(1);
	}

	result = FTPInitConnectionInfo(&gLib, &gConn, kDefaultFTPBufSize);
	if (result < 0) {
		(void) fprintf(stderr, "ncftpbatch: init connection info error %d (%s).\n", result, FTPStrError(result));
		exit(1);
	}
}	/* FTPInit */




/* These things are done first, before we even parse the command-line
 * options.
 */
static void
PreInit(void)
{
	gIsTTY = ((isatty(2) != 0) && (getppid() > 1)) ? 1 : 0;
	InitUserInfo();

	FTPInit();
	LoadFirewallPrefs();
	srand((unsigned int) getpid());

	(void) OurDirectoryPath(gSpoolDir, sizeof(gSpoolDir), kSpoolDir);
	(void) signal(SIGINT, SigExit);
	(void) signal(SIGTERM, SigExit);
}	/* PreInit */




/* These things are just before the program exits. */
static void
PostShell(void)
{
	CloseLog();
}	/* PostShell */




static int
PrePrintItem(void)
{
	FILE *fp;
	char line[256];
	char *tok1, *tok2;
	
	fp = fopen(gMyItemPath, "r");
	if (fp == NULL) {
		/* Could have been renamed already. */
		return (-1);
	}

	gOperation = '?';
	gHost[0] = '\0';
	gHostIP[0] = '\0';
	gRUser[0] = '\0';
	gRPass[0] = '\0';
	gXtype = 'I';
	gRecursive = 0;
	gDelete = 0;
	gPassive = 2;
	gRDir[0] = '\0';
	gLDir[0] = '\0';
	gRFile[0] = '\0';
	gLFile[0] = '\0';

	line[sizeof(line) - 1] = '\0';
	while (fgets(line, sizeof(line) - 1, fp) != NULL) {
		tok1 = strtok(line, " =\t\r\n");
		if ((tok1 == NULL) || (tok1[0] == '#'))
			continue;
		tok2 = strtok(NULL, "\r\n");
		if (tok2 == NULL)
			continue;

		if (strcmp(tok1, "op") == 0) {
			gOperation = tok2[0];
		} else if (strcmp(tok1, "hostname") == 0) {
			(void) STRNCPY(gHost, tok2);
		} else if (strcmp(tok1, "host-ip") == 0) {
			/* Don't really use this anymore, it is
			 * only used if the host is not set.
			 */
			(void) STRNCPY(gHostIP, tok2);
		} else if (strcmp(tok1, "user") == 0) {
			(void) STRNCPY(gRUser, tok2);
		} else if (strcmp(tok1, "pass") == 0) {
			(void) STRNCPY(gRPass, tok2);
		} else if (strcmp(tok1, "xtype") == 0) {
			gXtype = tok2[0];
		} else if (strcmp(tok1, "recursive") == 0) {
			gRecursive = StrToBool(tok2);
		} else if (strcmp(tok1, "delete") == 0) {
			gDelete = StrToBool(tok2);
		} else if (strcmp(tok1, "passive") == 0) {
			if (isdigit((int) tok2[0]))
				gPassive = atoi(tok2);
			else
				gPassive = StrToBool(tok2);
		} else if (strcmp(tok1, "remote-dir") == 0) {
			(void) STRNCPY(gRDir, tok2);
		} else if (strcmp(tok1, "local-dir") == 0) {
			(void) STRNCPY(gLDir, tok2);
		} else if (strcmp(tok1, "remote-file") == 0) {
			(void) STRNCPY(gRFile, tok2);
		} else if (strcmp(tok1, "local-file") == 0) {
			(void) STRNCPY(gLFile, tok2);
		}	/* else, unknown command. */
	}
	(void) fclose(fp);

	if (islower(gOperation))
		gOperation = toupper(gOperation);

	if (gHost[0] == '\0') {
		if (gHostIP[0] != '\0') {
			(void) STRNCPY(gHost, gHostIP);
		} else {
			return (-1);
		}
	}

	if (gOperation == 'G') {
		if (gRecursive != 0) {
			if (gRFile[0] == '\0') {
				return (-1);
			}
			if (gLDir[0] == '\0') {
				return (-1);
			}
		} else {
			if (gRFile[0] == '\0') {
				return (-1);
			}
			if (gLFile[0] == '\0') {
				return (-1);
			}
		}
	} else if (gOperation == 'P') {
		if (gRecursive != 0) {
			if (gLFile[0] == '\0') {
				return (-1);
			}
			if (gRDir[0] == '\0') {
				return (-1);
			}
		} else {
			if (gLFile[0] == '\0') {
				return (-1);
			}
			if (gRFile[0] == '\0') {
				return (-1);
			}
		}
	} else {
		return (-1);
	}

	if (gRUser[0] == '\0')
		(void) STRNCPY(gRUser, "anonymous");

	return (0);
}	/* PrePrintItem */




static int
DoItem(int iType)
{
	FILE *fp;
	char line[256];
	char *tok1, *tok2;
	int needOpen;
	int result;
	int n;
	FTPSigProc osigalrm;
	
	fp = fopen(gMyItemPath, "r");
	if (fp == NULL) {
		LogPerror(gMyItemPath);
		return (-1);
	}

	gOperation = iType;
	gHost[0] = '\0';
	gHostIP[0] = '\0';
	gPort = kDefaultFTPPort;
	gRUser[0] = '\0';
	gRPass[0] = '\0';
	gXtype = 'I';
	gRecursive = 0;
	gDelete = 0;
	gPassive = 2;
	gRDir[0] = '\0';
	gLDir[0] = '\0';
	gRFile[0] = '\0';
	gLFile[0] = '\0';

	line[sizeof(line) - 1] = '\0';
	while (fgets(line, sizeof(line) - 1, fp) != NULL) {
		tok1 = strtok(line, " =\t\r\n");
		if ((tok1 == NULL) || (tok1[0] == '#'))
			continue;
		tok2 = strtok(NULL, "\r\n");
		if (tok2 == NULL)
			continue;

		if (strcmp(tok1, "op") == 0) {
			gOperation = tok2[0];
		} else if (strcmp(tok1, "hostname") == 0) {
			(void) STRNCPY(gHost, tok2);
		} else if (strcmp(tok1, "host-ip") == 0) {
			(void) STRNCPY(gHostIP, tok2);
		} else if (strcmp(tok1, "port") == 0) {
			n = atoi(tok2);
			if (n > 0)
				gPort = (unsigned int) n;
		} else if (strcmp(tok1, "user") == 0) {
			(void) STRNCPY(gRUser, tok2);
		} else if (strcmp(tok1, "pass") == 0) {
			(void) STRNCPY(gRPass, tok2);
		} else if (strcmp(tok1, "xtype") == 0) {
			gXtype = tok2[0];
		} else if (strcmp(tok1, "recursive") == 0) {
			gRecursive = StrToBool(tok2);
		} else if (strcmp(tok1, "delete") == 0) {
			gDelete = StrToBool(tok2);
		} else if (strcmp(tok1, "passive") == 0) {
			if (isdigit((int) tok2[0]))
				gPassive = atoi(tok2);
			else
				gPassive = StrToBool(tok2);
		} else if (strcmp(tok1, "remote-dir") == 0) {
			(void) STRNCPY(gRDir, tok2);
		} else if (strcmp(tok1, "local-dir") == 0) {
			(void) STRNCPY(gLDir, tok2);
		} else if (strcmp(tok1, "remote-file") == 0) {
			(void) STRNCPY(gRFile, tok2);
		} else if (strcmp(tok1, "local-file") == 0) {
			(void) STRNCPY(gLFile, tok2);
		}	/* else, unknown command. */
	}
	(void) fclose(fp);

	if (islower(gOperation))
		gOperation = toupper(gOperation);

	/* First, check if the parameters from the batch file
	 * are valid.  If not, return successfully so the file
	 * gets removed.
	 */
	if (gHost[0] == '\0') {
		if (gHostIP[0] != '\0') {
			(void) STRNCPY(gHost, gHostIP);
		} else {
			Log("batch file parameter missing: %s.\n", "host");
			return (0);
		}
	}

	if (gOperation == 'G') {
		if (gRecursive != 0) {
			if (gRFile[0] == '\0') {
				Log("batch file parameter missing: %s.\n", "remote-file");
				return (0);
			}
			if (gLDir[0] == '\0') {
				Log("batch file parameter missing: %s.\n", "local-dir");
				return (0);
			}
		} else {
			if (gRFile[0] == '\0') {
				Log("batch file parameter missing: %s.\n", "remote-file");
				return (0);
			}
			if (gLFile[0] == '\0') {
				Log("batch file parameter missing: %s.\n", "local-file");
				return (0);
			}
		}
	} else if (gOperation == 'P') {
		if (gRecursive != 0) {
			if (gLFile[0] == '\0') {
				Log("batch file parameter missing: %s.\n", "local-file");
				return (0);
			}
			if (gRDir[0] == '\0') {
				Log("batch file parameter missing: %s.\n", "remote-dir");
				return (0);
			}
		} else {
			if (gLFile[0] == '\0') {
				Log("batch file parameter missing: %s.\n", "local-file");
				return (0);
			}
			if (gRFile[0] == '\0') {
				Log("batch file parameter missing: %s.\n", "remote-file");
				return (0);
			}
		}
	} else {
		Log("Invalid batch operation: %c.\n", gOperation);
		return (0);
	}

	if (gLDir[0] != '\0') {
		if (chdir(gLDir) < 0) {
			LogPerror("Could not cd to local-dir=%s", gLDir);
			return (0);
		} else if ((gOperation == 'G') && (access(gLDir, W_OK) < 0)) {
			LogPerror("Could not write to local-dir=%s", gLDir);
			return (0);
		}
	}

	if (gRUser[0] == '\0')
		(void) STRNCPY(gRUser, "anonymous");

	/* Decode password, if it was base-64 encoded. */
	if (strncmp(gRPass, kPasswordMagic, kPasswordMagicLen) == 0) {
		FromBase64(line, gRPass + kPasswordMagicLen, strlen(gRPass + kPasswordMagicLen), 1);
		(void) STRNCPY(gRPass, line);
	}

	/* Now see if we need to open a new host.  We try to leave the
	 * host connected, so if they batch multiple files using the
	 * same remote host we don't need to re-open the remote host.
	 */
	needOpen = 0;
	if (gConn.connected == 0) {
		/* Not connected at all. */
		Log("Was not connected originally.\n");
		needOpen = 1;
	} else if (ISTRCMP(gHost, gConn.host) != 0) {
		/* Host is different. */
		needOpen = 1;
		Log("New host (%s), old host was (%s).\n", gHost, gConn.host);
	} else if (strcmp(gRUser, gConn.user) != 0) {
		/* Same host, but new user. */
		needOpen = 1;
		Log("New user (%s), old user was (%s).\n", gRUser, gConn.user);
	}

	if (needOpen != 0) {
		(void) FTPCloseHost(&gConn);
		if (FTPInitConnectionInfo(&gLib, &gConn, kDefaultFTPBufSize) < 0) {
			/* Highly unlikely... */
			Log("init connection info failed!\n");
			ExitStuff();
			exit(1);
		}

		gConn.debugLogProc = DebugHook;
		gConn.debugLog = NULL;
		gConn.errLogProc = NULL;
		gConn.errLog = NULL;
		(void) STRNCPY(gConn.host, gHost);
		gConn.port = gPort;
		(void) STRNCPY(gConn.user, gRUser);
		(void) STRNCPY(gConn.pass, gRPass);
		gConn.maxDials = 1;
		gConn.dataPortMode = gPassive;

		if (MayUseFirewall(gConn.host, gFirewallType, gFirewallExceptionList) != 0) {
			gConn.firewallType = gFirewallType; 
			(void) STRNCPY(gConn.firewallHost, gFirewallHost);
			(void) STRNCPY(gConn.firewallUser, gFirewallUser);
			(void) STRNCPY(gConn.firewallPass, gFirewallPass);
			gConn.firewallPort = gFirewallPort;
		}
		
		gConn.connTimeout = 30;
		gConn.ctrlTimeout = 135;
		gConn.xferTimeout = 300;
		Log("Opening %s:%u as user %s...\n", gHost, gPort, gRUser);
		osigalrm = NcSignal(SIGALRM, (FTPSigProc) SIG_IGN);
		result = FTPOpenHost(&gConn);
		(void) NcSignal(SIGALRM, osigalrm);
		if (result < 0) {
			Log("Couldn't open %s, will try again next time.\n", gHost);
			(void) FTPCloseHost(&gConn);
			return (-1);	/* Try again next time. */
		}
		(void) FTPCmd(&gConn, "CLNT NcFTPBatch %.5s %s", gVersion + 11, gOS);
	}

	if (FTPChdir(&gConn, gRDir) < 0) {
		Log("Could not remote cd to %s.\n", gRDir);

		/* Leave open, but unspool. */
		return (0);
	}

	if (gOperation == 'G') {
		if (gRecursive != 0) {
			result = FTPGetFiles3(&gConn, gRFile, gLDir, gRecursive, kGlobNo, gXtype, kResumeYes, kAppendNo, gDelete, kTarNo);
		} else {
			result = FTPGetOneFile3(&gConn, gRFile, gLFile, gXtype, (-1), kResumeYes, kAppendNo, gDelete, 0);
		}
		if (result == kErrCouldNotStartDataTransfer) {
			Log("Remote item %s is no longer retrievable.n", gRFile);
			result = 0;	/* file no longer on host */
		}
	} else /* if (gOperation == 'P') */ {
		if (gRecursive != 0) {
			result = FTPPutFiles3(&gConn, gLFile, gRDir, gRecursive, kGlobNo, gXtype, kAppendNo, NULL, NULL, kResumeYes, gDelete, 0);
		} else {
			result = FTPPutOneFile3(&gConn, gLFile, gRFile, gXtype, (-1), kAppendNo, NULL, NULL, kResumeYes, gDelete, 0);
		}
		if (result == kErrCouldNotStartDataTransfer) {
			Log("Remote item %s is no longer retrievable.n", gRFile);
			result = 0;	/* file no longer on host */
		}
	}
	return (result);
}	/* DoItem */




static int
DecodeName(const char *const src, int *yyyymmdd, int *hhmmss)
{
	char itemName[64];
	char *tok, *ps;
	int t;
	int valid = -1;

	(void) STRNCPY(itemName, src);
	for (t = 0, ps = itemName; ((tok = strtok(ps, "-")) != NULL); ps = NULL) {
		t++;
		switch (t) {
			case 4:
				*yyyymmdd = atoi(tok);
				break;
			case 5:
				*hhmmss = atoi(tok);
				valid = 0;
				break;
			case 6:
				valid = -1;
				break;
		}
	}
	if (valid < 0) {
		*yyyymmdd = 0;
		*hhmmss = 0;
	}
	return (valid);
}	/* DecodeName */




static void
Now(int *yyyymmdd, int *hhmmss)
{
	struct tm *ltp;
	time_t now;

	(void) time(&now);
	ltp = localtime(&now);
	if (ltp == NULL) {
		*yyyymmdd = 0;
		*hhmmss = 0;
	} else {
		*yyyymmdd = ((ltp->tm_year + 1900) * 10000)
			+ ((ltp->tm_mon + 1) * 100)
			+ (ltp->tm_mday);
		*hhmmss = (ltp->tm_hour * 10000)
			+ (ltp->tm_min * 100)
			+ (ltp->tm_sec);
	}
}	/* Now */




static void
EventShell(volatile int sleepval)
{
	volatile int passes;
	int nItems;
	struct dirent *direntp;
	struct stat st;
	DIR *volatile DIRp;
	char *cp;
	int iType;
	int iyyyymmdd, ihhmmss, nyyyymmdd, nhhmmss;
	int sj;

	DIRp = NULL;
	OpenLog();
	Log("-----started-----\n");

#ifdef HAVE_SIGSETJMP
	sj = sigsetjmp(gCancelJmp, 1);
#else	/* HAVE_SIGSETJMP */
	sj = setjmp(gCancelJmp);
#endif	/* HAVE_SIGSETJMP */

	if (sj != 0) {
		gMayCancelJmp = 0;
		if (DIRp != NULL) {
			(void) closedir(DIRp);
			DIRp = NULL;
		}
		FTPShutdownHost(&gConn);
		Log("Timed-out, starting over.\n");
	}
	gMayCancelJmp = 1;

	for (passes = 0; ; ) {
		passes++;
		if ((passes > 1) || ((passes == 1) && (sleepval > 0))) {
			if (sleepval == 0)
				sleepval = 5;
			else
				sleepval = (int) (((0.1 * (rand() % 15)) + 1.2) * sleepval); 

			/* Re-open it, in case they deleted the log
			 * while this process was running.
			 */
			OpenLog();
			Log("Sleeping %d seconds before starting pass %d.\n", sleepval, passes);
			(void) sleep(sleepval);
		}

		(void) signal(SIGALRM, SigAlrm);
		if ((DIRp = opendir(gSpoolDir)) == NULL) {
			perror(gSpoolDir);
			exit(1);
		}
		Log("Starting pass %d.\n", passes);
		for (nItems = 0; ; ) {
			direntp = readdir(DIRp);
			if (direntp == NULL)
				break;

			(void) STRNCPY(gItemPath, gSpoolDir);
			(void) STRNCAT(gItemPath, "/");
			(void) STRNCAT(gItemPath, direntp->d_name);
			if ((stat(gItemPath, &st) < 0) || (S_ISREG(st.st_mode) == 0)) {
				/* Item may have been
				 * deleted by another
				 * process.
				 */
				continue;
			}

			if (DecodeName(direntp->d_name, &iyyyymmdd, &ihhmmss) < 0) {
				/* Junk file in the spool directory. */
				continue;
			}

			cp = strrchr(gItemPath, '/');
			if (cp == NULL) {
				/* Impossible */
				continue;
			}
			cp++;

			iType = (int) *cp;
			if ((iType != 'g') && (iType != 'p')) {
				/* No more items waiting for processing. */
				continue;
			}

			/* Count items waiting for processing. */
			nItems++;

			Now(&nyyyymmdd, &nhhmmss);
			if ((nyyyymmdd < iyyyymmdd) || ((nyyyymmdd == iyyyymmdd) && (nhhmmss < ihhmmss))) {
				/* Process only if the specified start
				 * time has passed.
				 */
				continue;
			}

			(void) STRNCPY(gMyItemPath, gItemPath);
			gMyItemPath[(int) (cp - gItemPath)] = 'x';

			/* Race condition between other ncftpbatches,
			 * but only one of them will rename it
			 * successfully.
			 */
			if (rename(gItemPath, gMyItemPath) == 0) {
				Log("Processing path: %s\n", gMyItemPath);
				gItemInUse = 1;
				if (DoItem(iType) < 0) {
					/* rename it back, so it will
					 * get reprocessed.
					 */
					(void) rename(gMyItemPath, gItemPath);
					Log("Re-queueing %s.\n", gItemPath);
				} else {
					(void) unlink(gMyItemPath);
					Log("Done with %s.\n", gItemPath);
				}
				(void) chdir("/");
			}
		}
		(void) closedir(DIRp);
		if (nItems == 0) {
			/* Spool directory is empty, done. */
			Log("The spool directory is now empty.\n");
			break;
		}
	}
	(void) FTPCloseHost(&gConn);
	gMayCancelJmp = 0;
	(void) NcSignal(SIGALRM, (FTPSigProc) SIG_DFL);
	Log("-----done-----\n");
}	/* EventShell */





static void
ListQueue(void)
{
	int nItems;
	struct dirent *direntp;
	struct stat st;
	DIR *DIRp;
	char *cp;
	int iyyyymmdd, ihhmmss;
	char dstr[64];
	char yyyy[8], mm[4], dd[4];
	char HH[4], MM[4];

	if ((DIRp = opendir(gSpoolDir)) == NULL) {
		perror(gSpoolDir);
		(void) fprintf(stderr, "This directory is created automatically the first time you do a background\noperation from NcFTP.\n");
		exit(1);
	}
	for (nItems = 0; ; ) {
		direntp = readdir(DIRp);
		if (direntp == NULL)
			break;

		(void) STRNCPY(gItemPath, gSpoolDir);
		(void) STRNCAT(gItemPath, "/");
		(void) STRNCAT(gItemPath, direntp->d_name);
		if ((stat(gItemPath, &st) < 0) || (S_ISREG(st.st_mode) == 0)) {
			/* Item may have been
			 * deleted by another
			 * process.
			 */
			continue;
		}

		if (DecodeName(direntp->d_name, &iyyyymmdd, &ihhmmss) < 0) {
			/* Junk file in the spool directory. */
			continue;
		}

		cp = strrchr(gItemPath, '/');
		if (cp == NULL) {
			/* Impossible */
			continue;
		}
		cp++;

		(void) STRNCPY(gMyItemPath, gItemPath);
		if (PrePrintItem() == 0) {
			nItems++;
			if (nItems == 1) {
				(void) printf("---Scheduled-For-----Host----------------------------Command--------------------\n");
			}
			(void) sprintf(dstr, "%08d", iyyyymmdd);
			(void) memcpy(yyyy, dstr, 4); yyyy[4] = '\0';
			(void) memcpy(mm, dstr + 4, 2); mm[2] = '\0';
			(void) memcpy(dd, dstr + 6, 2); dd[2] = '\0';
			(void) sprintf(dstr, "%06d", ihhmmss);
			(void) memcpy(HH, dstr + 0, 2); HH[2] = '\0';
			(void) memcpy(MM, dstr + 2, 2); MM[2] = '\0';
			(void) printf("%c  %s-%s-%s %s:%s  %-30s  ",
				(gItemPath[0] == 'x') ? '*' : ' ',
				yyyy, mm, dd, HH, MM,
				gHost
			);
			if (gOperation != 'P') {
				(void) printf("GET");
				if (gRecursive != 0) {
					(void) printf(" -R %s", gRFile);
				} else {
					(void) printf(" %s", gRFile);
				}
			} else {
				(void) printf("PUT");
				if (gRecursive != 0) {
					(void) printf(" -R %s", gLFile);
				} else {
					(void) printf(" %s", gLFile);
				}
			}
			(void) printf("\n");
		}
	}
	(void) closedir(DIRp);
	if (nItems == 0) {
		/* Spool directory is empty, done. */
		(void) printf("Your \"%s\" directory is empty.\n", gSpoolDir);
	}
}	/* ListQueue */





static int
PRead(int sfd, char *const buf0, size_t size, int retry)
{
	int nread;
	int nleft;
	char *buf = buf0;

	errno = 0;
	nleft = (int) size;
	while (1) {
		nread = read(sfd, buf, nleft);
		if (nread <= 0) {
			if (nread == 0) {
				/* EOF */
				nread = size - nleft;
				return (nread);
			} else if (errno != EINTR) {
				nread = size - nleft;
				if (nread == 0)
					nread = -1;
				return (nread);
			} else {
				errno = 0;
				nread = 0;
				/* Try again. */
			}
		}
		nleft -= nread;
		if ((nleft <= 0) || (retry == 0))
			break;
		buf += nread;
	}
	nread = size - nleft;
	return (nread);
}	/* PRead */




static void
ReadCore(int fd)
{
	FTPLibraryInfo tLib;
	FTPConnectionInfo tConn;
	int rc;

	if ((PRead(fd, (char *) &tLib, sizeof(tLib), 1) == sizeof(tLib))
		&& (PRead(fd, (char *) &tConn, sizeof(tConn), 1) == sizeof(tConn))
		&& (strncmp(tConn.magic, gConn.magic, sizeof(tConn.magic)) == 0)
	) {
		(void) memcpy(&gConn, &tConn, sizeof(gConn));
		(void) memcpy(&gLib, &tLib, sizeof(gLib));
		rc = FTPRebuildConnectionInfo(&gLib, &gConn);
		if (rc < 0) {
			FTPInit();
		} else {
			gConn.debugLogProc = DebugHook;
		}
	}
}	/* ReadCore */




static void
Daemon(void)
{
	int i, fd;
	int devnull;
	int pid;

	/* Redirect standard in, out, and err, if they were terminals. */
	devnull = open("/dev/null", O_RDWR, 00666);

	for (i=0; i<3; i++) {
		if (gConn.ctrlSocketR == i)
			continue;
		if (gConn.ctrlSocketW == i)
			continue;
		if (isatty(1) > 0) {
			/* Close tty files and replace
			 * with /dev/null.
			 */
			(void) close(i);
			if (devnull >= 0)
				(void) dup2(devnull, i);
		}
	}

	if (devnull >= 0)
		(void) close(devnull);

	/* Close all unneeded descriptors. */
	for (fd = 3; fd < 256; fd++) {
		if (gConn.ctrlSocketR == fd)
			continue;
		if (gConn.ctrlSocketW == fd)
			continue;
		(void) close(fd);
	}

	pid = fork();
	if (pid < 0)
		exit(1);
	else if (pid > 0)
		exit(0);	/* parent. */

#ifdef HAVE_SETSID
	/* Become session leader for this "group." */
	(void) setsid();
#endif

	/* Run as "nohup."  Don't want to get hangup signals. */
	(void) NcSignal(SIGHUP, (FTPSigProc) SIG_IGN);

	/* Turn off TTY control signals, just to be sure. */
	(void) NcSignal(SIGINT, (FTPSigProc) SIG_IGN);
	(void) NcSignal(SIGQUIT, (FTPSigProc) SIG_IGN);
#ifdef SIGTSTP
	(void) NcSignal(SIGTSTP, (FTPSigProc) SIG_IGN);
#endif
	
	/* Become our own process group. */
#ifdef HAVE_SETPGID
	(void) setpgid(0, 0);
#elif defined(HAVE_SETPGRP) && defined(SETPGRP_VOID)
	(void) setpgrp();
#elif defined(HAVE_SETPGRP) && !defined(SETPGRP_VOID)
	(void) setpgrp(0, getpid());
#endif

#ifdef TIOCNOTTY
	/* Detach from controlling terminal, so this
	 * process is not associated with any particular
	 * tty.
	 */
	fd = open("/dev/tty", O_RDWR, 0);
	if (fd >= 0) {
		(void) ioctl(fd, TIOCNOTTY, 0);
		(void) close(fd);
	}
#endif

	/* Change to root directory so filesystems
	 * can be unmounted.
	 */
	(void) chdir("/");
}	/* Daemon */




static void
Usage(void)
{
	(void) fprintf(stderr, "Usages:\n");
	(void) fprintf(stderr, "\tncftpbatch -d | -D\t\t\t(start NcFTP batch processing)\n");
	(void) fprintf(stderr, "\tncftpbatch -l\t\t\t\t(list spooled jobs)\n");
	(void) fprintf(stderr, "\nLibrary version: %s.\n", gLibNcFTPVersion + 5);
	(void) fprintf(stderr, "This is a freeware program by Mike Gleason (mgleason@probe.net).\n");
	exit(2);
}	/* Usage */




int
main(int argc, const char **const argv)
{
	int c;
	int daemon = -1;
	int sleepval = 0;
	int listonly = -1;
	int readcore = -1;

	PreInit();
	while ((c = getopt(argc, (char **) argv, "|:XDdls:w")) > 0) switch(c) {
		case 'd':
			daemon = 1;
			break;
		case 'D':
			daemon = 0;
			break;
		case 'l':
			listonly = 1;
			break;
		case 's':
			sleepval = atoi(optarg);
			break;
		case 'w':
			gLogOpenMode = "w";
			break;
		case '|':
			readcore = atoi(optarg);
			break;
		case 'X':
			/* Yes, I do exist. */
			exit(0);
		default:
			Usage();
	}

	if ((listonly < 0) && (daemon < 0)) {
		/* Must specify either -l or -d/-D */
		Usage();
	}

	if (listonly > 0) {
		ListQueue();
	} else {
		if (readcore >= 0) {
			/* Inherit current live FTP session
			 * from ncftp!
			 */
			ReadCore(readcore);
		}
		if (daemon > 0) {
			Daemon();
			gIsTTY = 0;
		}

		EventShell(sleepval);
		PostShell();
	}
	exit(0);
}	/* main */

#else	/* HAVE_LONG_FILE_NAMES */
main()
{
	fprintf(stderr, "this program needs long filenames, sorry.\n");
	exit(1);
}	/* main */
#endif	/* HAVE_LONG_FILE_NAMES */
