/*
  $Id: error.c,v 1.3 1996/08/10 14:21:13 luik Exp $

  error.c - error message routines for omirrd.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: error.c,v 1.3 1996/08/10 14:21:13 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#include <stdio.h>
#if STDC_HEADERS
#include <stdarg.h>
#define VA_START(ap, f)	va_start(ap, f)
#define VA_END(ap)	va_end(ap)
#else
#include <varargs.h>
#define VA_START(ap, f)	va_start(ap)
#define VA_END(ap)	va_end(ap)
#endif
#define NOVAPROTO		/* suppress varargs prototypes in "error.h" */
#include "error.h"
#undef NOVAPROTO
#include "omirrd.h"


/* Bit values for `log_mask'. They specify the different destinations
   for log messages.  */
#define DO_STDOUT   (1L << 0)	/* log to stdout */
#define DO_STDERR   (1L << 1)	/* log to stderr */
#define DO_FILE	    (1L << 2)	/* log to file `logf' */
#define DO_SYSLOG   (1L << 3)	/* log to syslog(3) */

static unsigned long log_mask = DO_STDERR; /* current log mask */
static FILE *logf;		/* current log file for DO_FILE */


/* errlog_stdout - set or clear DO_STDOUT flag in `log_mask'. Returns
   old value of flag.  */
int errlog_stdout(int flag)
{
    int old = (log_mask & DO_STDOUT) != 0;
    if (flag)
	log_mask |= DO_STDOUT;
    else
	log_mask &= ~DO_STDOUT;
    return (old);
}

/* errlog_stderr - set or clear DO_STDERR flag in `log_mask'. Returns
   old value of flag.  */
int errlog_stderr(int flag)
{
    int old = (log_mask & DO_STDERR) != 0;
    if (flag)
	log_mask |= DO_STDERR;
    else
	log_mask &= ~DO_STDERR;
    return (old);
}

/* errlog_file - set or clear DO_FILE flag in `log_mask'. Also closes
   old log file and opens `filename' as new log file.  Returns old
   value of flag or -1 to flag an error condition.  */
int errlog_file(int flag, char *filename)
{
    int old = (log_mask & DO_FILE) != 0;

    if (logf) {
	fclose(logf);		/* close old log file */
	logf = NULL;
    }

    if (flag) {
	if ((logf = fopen(filename, "w")))
	    log_mask |= DO_FILE;
	else old = -1;		/* error condition returns -1 */
    }
    else {
	log_mask &= ~DO_FILE;
    }
    return (old);
}

/* errlog_syslog - set or clear DO_SYSLOG flag in `log_mask'. Returns
   old value of flag.  */
int errlog_syslog(int flag)
{
    int old = (log_mask & DO_SYSLOG) != 0;
    if (flag)
	log_mask |= DO_SYSLOG;
    else
	log_mask &= ~DO_SYSLOG;
    return (old);
}



/* errlognameprefix - print a log message prefix to file `f'. The
   prefix either consists of the name and current line number of the
   file currently processed (from the global variables
   `current_file_name' and `current_line') or of the name of the
   current program (from `program_name').  */
static void errlognameprefix(FILE *f)
{
    if (current_file_name) {
	fprintf(f, "%s:", current_file_name);
	if (current_line != -1)
	    fprintf(f, "%d:", current_line);
	putc(' ', f);
    }
    else if (program_name)
	fprintf(f, "%s: ", program_name);
}


#if HAVE_VPRINTF

/*ARGSUSED*/
static void vferrlog(FILE *f, int priority, const char *fmt, va_list ap)
{
    errlognameprefix(f);
    vfprintf(f, fmt, ap);
}


void verrlog(int priority, const char *fmt, va_list ap)
{
    if (log_mask & DO_STDOUT)
	vferrlog(stdout, priority, fmt, ap);
    if (log_mask & DO_STDERR)
	vferrlog(stderr, priority, fmt, ap);
    if (log_mask & DO_FILE)
	vferrlog(logf, priority, fmt, ap);
    if (log_mask & DO_SYSLOG) {
#if HAVE_VSYSLOG
	vsyslog(priority, fmt, ap);
#else /* !HAVE_VSYSLOG */
	char buf[512];
	vsprintf(buf, fmt, ap);
	syslog(priority, buf);
#endif /* !HAVE_VSYSLOG */
    }
}


#if STDC_HEADERS
void errlog(int priority, const char *fmt, ...)
#else
void errlog(priority, fmt, va_alist)
int priority;
const char *fmt;
va_dcl
#endif
{
    va_list ap;

    VA_START(ap, fmt);
    verrlog(priority, fmt, ap);
    VA_END(ap);
}


#if STDC_HEADERS
void fatal(const char *fmt, ...)
#else
void fatal(fmt, va_alist)
const char *fmt;
va_dcl
#endif
{
    va_list ap;

    VA_START(ap, fmt);
    verrlog(LOG_CRIT, fmt, ap);
    VA_END(ap);
}

 
#if STDC_HEADERS
void error(const char *fmt, ...)
#else
void error(fmt, va_alist)
const char *fmt;
va_dcl
#endif
{
    va_list ap;

    VA_START(ap, fmt);
    verrlog(LOG_ERR, fmt, ap);
    VA_END(ap);
}


#if STDC_HEADERS
void warning(const char *fmt, ...)
#else
void warning(fmt, va_alist)
const char *fmt;
va_dcl
#endif
{
    va_list ap;

    VA_START(ap, fmt);
    verrlog(LOG_WARNING, fmt, ap);
    VA_END(ap);
}

#else /* !HAVE_VPRINTF */

void errlog(int priority, const char *fmt,
	    const char *a1, const char *a2, const char *a3, const char *a4,
	    const char *a5, const char *a6, const char *a7, const char *a8)
{
    if (log_mask & DO_STDOUT) {
	errlognameprefix(stdout);
	printf(fmt, a1, a2, a3, a4, a5, a6, a7, a8);
    }
    if (log_mask & DO_STDERR) {
	errlognameprefix(stderr);
	fprintf(stderr, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
    }
    if (log_mask & DO_FILE) {
	errlognameprefix(logf);
	fprintf(logf, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
    }
    if (log_mask & DO_SYSLOG) {
	char buf[512];
	sprintf(buf, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
	syslog(priority, buf);
    }
}

void fatal(const char *fmt,
	   const char *a1, const char *a2, const char *a3, const char *a4,
	   const char *a5, const char *a6, const char *a7, const char *a8)
{
    errlog(LOG_CRIT, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
}

void error(const char *fmt,
	   const char *a1, const char *a2, const char *a3, const char *a4,
	   const char *a5, const char *a6, const char *a7, const char *a8)
{
    errlog(LOG_ERR, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
}

void warning(const char *fmt,
	     const char *a1, const char *a2, const char *a3, const char *a4,
	     const char *a5, const char *a6, const char *a7, const char *a8)
{
    errlog(LOG_WARNING, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
}

#endif /* !HAVE_VPRINTF */


