/*
  $Id: omirrk.c,v 1.21 1997/01/16 15:04:18 luik Exp $

  omirrk.c - online mirror daemon kernel interface file.
  Copyright (C) 1996-1997, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: omirrk.c,v 1.21 1997/01/16 15:04:18 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#ifdef ENABLE_OMIRRK

#include <stdio.h>
#include <stdlib.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#if HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#include <linux/omirr.h>
#include "libomirr.h"
#include "debug.h"
#include "error.h"
#include "mp.h"
#include "bio.h"
#include "lio.h"
#include "request.h"
#include "iname.h"
#include "omirrd.h"
#include "omirrk.h"


#ifndef WIFEXITED
#define	WIFEXITED(S)	((int)((S) & 0xFF) == 0)
#endif
#ifndef	WIFSIGNALED
#define	WIFSIGNALED(S)	((int)((S) & 0xFF) > 0 && (int)((S) & 0xFF00) == 0)
#endif
#ifndef	WIFSTOPPED
#define	WIFSTOPPED(S)	((int)((S) & 0xFF) == 0177 && (int)((S) & 0xFF00) != 0)
#endif
#ifndef	WEXITSTATUS
#define	WEXITSTATUS(S)	((int)(((S) >> 8) & 0xFF))
#endif
#ifndef	WTERMSIG
#define	WTERMSIG(S)	((int)((S) & 0x7F))
#endif
#ifndef	WSTOPSIG
#define	WSTOPSIG(S)	((int)(((S) >> 8) & 0xFF))
#endif


/* Data passed to lio callback `process_kernel_message' from main
   routine `kernel_interface_main'.  */
typedef struct CallbackData_ {
    int fd;			/* output file descriptor (pipe to omirrd) */
    ReqQueue req_queue;		/* request queue used to save requests */
} *CallbackData, CallbackDataRec;


/* Static function declarations.  */
static void process_request_queue(ReqQueue req_queue);



/**********************************************************************
 * Utility functions.
 **********************************************************************/

/* cleanup - cleanup before calling exit.  */
static void cleanup(int sig)
{
    /* Nothing to do currently */
}



/**********************************************************************
 * Signal handlers for omirrk.
 **********************************************************************/

static RETSIGTYPE handle_sigpipe(int sig)
{
    debuglog(DEBUG_DAEMON, ("handle_sigpipe: processing SIGPIPE\n"));
    /* Nothing to do because write errors to pipe are handled by mp
       error callbacks, e.g. `close_omirrd_connection'.  */
}

static RETSIGTYPE handle_sigterm(int sig)
{
    debuglog(DEBUG_DAEMON, ("handle_sigterm: processing SIGTERM\n"));
    cleanup(sig);
    _exit(0);
}

static RETSIGTYPE handle_sigchld(int sig)
{
    pid_t pid;
    int wait_status;
    char *msg;
    int msg_arg;

    while (1) {			/* repeat wait to catch all exited children */
#ifdef HAVE_WAITPID
	pid = waitpid((pid_t) -1, &wait_status, WNOHANG);
#else
	pid = wait3(&wait_status, WNOHANG, NULL);
#endif
	if (pid == -1 && errno == EINTR)
	    continue;		/* try again */

	if (pid == 0 || (pid == -1 && errno == ECHILD))
	    break;		/* no exited child available */

	if (WIFEXITED(wait_status)) {
	    msg = "exited with status";
	    msg_arg = WEXITSTATUS(wait_status);
	}
	else if (WIFSIGNALED(wait_status)) {
	    msg = "terminated due to signal";
	    msg_arg = WTERMSIG(wait_status);
	}
	else if (WIFSTOPPED(wait_status)) {
	    msg = "stopped due to signal";
	    msg_arg = WSTOPSIG(wait_status);
	}
	else {
	    msg = "returned unknown status";
	    msg_arg = wait_status;
	}
	debuglog(DEBUG_DAEMON, ("handle_sigchld: processing SIGCHLD, pid %ld %s %d\n",
				(long) pid, msg, msg_arg));
    }
}


/*ARGSUSED*/
static RETSIGTYPE process_signal(int sig, void *data)
{
    switch (sig) {
      case SIGPIPE:
	handle_sigpipe(sig);
	break;

      case SIGTERM:
	handle_sigterm(sig);
	break;

      case SIGCHLD:
	handle_sigchld(sig);
	break;

      default:
	error("internal: process_signal: unknown signal %d\n", sig);
	break;
    }
}



/**********************************************************************
 * Message processing.
 *********************************************************************/

static void process_request_queue(ReqQueue req_queue)
{
    ReqRequest r;
    ReqState state;
    int incomplete;		/* count incomplete requests until max */
    char *line;

    r = reqGetFirstRequest(req_queue);
    incomplete = 0;
    while (r && incomplete < MAX_INCOMPLETE_REQUESTS) {
	state = reqFillRequest(r); /* determine missing parts in request */
	
	if (state == REQ_COMPLETE && incomplete == 0) {
	    if (reqProcessKernelRequest(r)) {
		if ((line = reqRequestToMessage(r))) {
		    bioWrite(bio_context, r->outfd, line, strlen(line) + 1);
		    free(line);
		}
	    }
	    r = reqGetNextRequest(r);
	    reqDropFirstRequest(req_queue);
	}
	else if (state == REQ_ERROR && incomplete == 0) {
	    if (r->path.ompath && r->path.path && *r->path.path == '\0')
		syslog(LOG_ERR, "couldn't resolve `%s' - ignoring request\n",
		       r->path.ompath);
	    if (r->newpath.ompath && r->newpath.path && *r->newpath.path == '\0')
		syslog(LOG_ERR, "couldn't resolve `%s' - ignoring request\n",
		       r->newpath.ompath);
	    r = reqGetNextRequest(r);
	    reqDropFirstRequest(req_queue);
	}
	else if (state == REQ_PROCESSING) {
	    incomplete++;
	    r = reqGetNextRequest(r);
	}
	else {
	    r = reqGetNextRequest(r);
	}
    }
}


static void process_kernel_message(char *line, void *data)
{
    CallbackData lio_callback_data = data;
    int outfd = lio_callback_data->fd; /* pipe output file descriptor */
    ReqQueue req_queue = lio_callback_data->req_queue;
    ReqRequest r;

    debuglog(DEBUG_KPROT, ("%s\n", line));

    /* add message to request queue */
    if ((r = reqParseMessage(req_queue, line)))
	r->outfd = outfd;

    /* now try to process the requests */
    process_request_queue(req_queue);
}


/* close_omirrd_connection - error callback for pipe file descriptor
   to omirrd process. Called if a write to the pipe connection returns
   an error. Since this means that we can't talk to omirrd any more,
   we terminate with a fatal error.  */
/*ARGSUSED*/
static int close_omirrd_connection(int fd, void *data)
{
    error("lost connection to omirrd: %s - terminating\n", xstrerror(errno));
    _exit(1);
    /*NOTREACHED*/
}



/**********************************************************************
 * iname callbacks
 *********************************************************************/


/* process_lookup_done - called from iname module (via
   inamAddLookupDoneCallback) if a delayed lookup (with find
   subprocess) completes.  This function simply calls
   `process_request_queue', so that the new iname-cache entry is
   immediately used to perhaps resolve the waiting requests in the
   queue.  */
/*ARGSUSED*/
static void process_lookup_done(DevIno dip, void *data)
{
    ReqQueue req_queue = (ReqQueue) data;
    
    process_request_queue(req_queue);
}



/**********************************************************************
 * omirrk (kernel interface) main program.
 *********************************************************************/

void kernel_interface_main(int outfd)
{
    int fd;			/* file descriptor to /proc/omirr (PATH_OMIRR) */
    ReqQueue req_queue;
    CallbackDataRec lio_callback_data;

    program_name = "omirrk";	/* change program name for error messages */
    setproctitle("<omirrk>");

#ifdef LOG_DAEMON		/* new openlog(3) with three args */
    openlog("omirrk", 0, LOG_OMIRRD);
#else
    openlog("omirrk", 0);
#endif

    if ((fd = open(PATH_OMIRR, O_RDONLY)) == -1) {
	error("open %s failed: %s - terminating\n",
	      PATH_OMIRR, xstrerror(errno));
	_exit(2);
    }

    if (ioctl(fd, OMIRRSETDONTLOGMASK, OMIRR_DONTLOG_PGRP) == -1)
	warning("omirrk: failed to set dont_log_mask: %s\n", xstrerror(errno));
    
    if (clearbuf_flag)
	if (ioctl(fd, OMIRRCLEARBUF, 0) == -1)
	    warning("omirrk: failed to clear ring buffer: %s\n", xstrerror(errno));

    if (block_mode_flag)
	if (ioctl(fd, OMIRRSETBLOCKMODE, block_mode_flag) == -1)	
	    warning("omirrk: failed to set blocking mode: %s\n", xstrerror(errno));

    /* Allocate request queue */
    req_queue = reqAllocQueue();

    /* Add callback for inam module which is called if a delayed
       inode-name lookup has completed.  The `process_lookup_done'
       function then tries to process additional requests by calling
       process_request_queue.  */
    inamAddLookupDoneCallback(process_lookup_done, req_queue);

    /* Allocate buffered I/O, line I/O and multiplex contexts.  */
    bio_context = bioAlloc();
    lio_context = lioAlloc();
    mp_context = mpAlloc();

    /* mp_context: add signal handlers.  */
    mpSetSignalCallback(mp_context, SIGHUP, MP_SIG_IGN, NULL, NULL);
    mpSetSignalCallback(mp_context, SIGPIPE, process_signal, NULL, NULL);
    mpSetSignalCallback(mp_context, SIGTERM, process_signal, NULL, NULL);
    mpSetSignalCallback(mp_context, SIGCHLD, process_signal, NULL, NULL);

    /* lio_context: add `process_kernel_message' as read
       callback. This is the function processing the messages from the
       kernel and writing them to the daemon main process using outfd.  */
    lio_callback_data.fd = outfd;
    lio_callback_data.req_queue = req_queue;
    lioAddReadCallback(lio_context, fd,
		       process_kernel_message, &lio_callback_data);

    /* mp_context: add `lioMpReadFunc' as read callback, which will
       split the input stream into lines and call the callback
       function of the lio context.  */
    mpAddReadCallback(mp_context, fd,
		      lioMpReadFunc, (void *) lio_context);

    /* mp_context: add `close_omirrd_connection' as error callback to
       close pipe filedescriptor in case of an error.  */
    mpAddErrorCallback(mp_context, outfd, close_omirrd_connection, NULL);

    /* bio_context: allocate buffer for buffered, non-blocking I/O on
       `outfd'. Write operations are multiplexed via `mp_context'.  */
    bioAddWriteBuffer(bio_context, outfd, mp_context);

    mpMainLoop(mp_context);
    error("internal: mpMainLoop returned - terminating\n");

    cleanup(SIGTERM);
    _exit(1);
}

#endif /* defined(ENABLE_OMIRRK) */

