/*
  $Id: sig.c,v 1.2 1996/08/12 13:30:10 luik Exp $

  sig.c - functions for signal handling for omirrd.
  Copyright (C) 1996, Andreas Luik, <luik@pharao.s.bawue.de>.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "common.h"

#if defined(RCSID) && !defined(lint)
static char rcsid[] UNUSED__ = "$Id: sig.c,v 1.2 1996/08/12 13:30:10 luik Exp $";
#endif /* defined(RCSID) && !defined(lint) */

#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <signal.h>
#include "sig.h"

/*
 * These functions implement an interface to the signal handling
 * functions of the operating system. The following conditions are
 * fulfilled:
 *
 * 1. Automatic restart of system calls is not required, i.e. slow
 * system calls can return -1 with errno == EINTR.
 *
 * 2. The select() system call is never automatically restarted.
 *
 * 3. The signal handler remains installed (therefore this module is
 * not usable on very old Unixes like V7 or SVR2).
 */

#ifdef _POSIX_VERSION		/* POSIX signals */

/* sigHandler - install `disp' as signal handler for signal
   `sig'. `disp' may be either a pointer to a signal handler function,
   SIG_IGN or SIG_DFL.  Returns the previous signal handler.  */
RETSIGTYPE (*sigHandler(int sig, RETSIGTYPE (*disp)(int)))(int)
{
    struct sigaction act;
    struct sigaction oldact;

    act.sa_handler = disp;
    sigemptyset(&act.sa_mask);
    act.sa_flags = 0;
    if (sigaction(sig, &act, &oldact) < 0)
	return(SIG_ERR);
    return(oldact.sa_handler);
}

/* sigBlock - blocks the specified signal `sig' from delivery. If
   successful, it returns 1 if the signal was already blocked before
   the call, or 0 if the signal was not yet blocked. On failure, it
   returns -1.  */
int sigBlock(int sig)
{
    int result;
    sigset_t newset;
    sigset_t oldset;

    sigemptyset(&newset);
    sigaddset(&newset, sig);
    if ((result = sigprocmask(SIG_BLOCK, &newset, &oldset)) != -1)
	result = sigismember(&oldset, sig);
    return result;
}

/* sigBlock - unblocks the specified signal `sig' from delivery. On
   failure, sigBlock returns -1, 0 otherwise.  */
int sigUnblock(int sig)
{
    sigset_t newset;

    sigemptyset(&newset);
    sigaddset(&newset, sig);
    return sigprocmask(SIG_UNBLOCK, &newset, NULL);
}

#else /* !defined(_POSIX_VERSION) */

#if HAVE_SIGVEC			/* 4.2BSD, 4.3BSD signals */

/* sigHandler - see above for description.  */
RETSIGTYPE (*sigHandler(int sig, RETSIGTYPE (*disp)(int)))(int)
{
    struct sigvec act;
    struct sigvec oldact;

    act.sv_handler = disp;
    act.sv_mask = 0;
    act.sv_flags = 0;
    if (sigvec(sig, &act, &oldact) < 0)
	return(SIG_ERR);
    return(oldact.sv_handler);
}

/* sigBlock - see above for description.  */
int sigBlock(int sig)
{
    int mask;

    mask = sigblock(sigmask(sig));
    return (mask & sigmask(sig)) != 0;
}

/* sigUnblock - see above for description.  */
int sigUnblock(int sig)
{
    int mask;

    mask = sigblock(0);
    mask &= ~sigmask(sig);
    sigsetmask(mask);
    return 0;
}

#else /* !HAVE_SIGVEC */

#if HAVE_SIGSET			/* SVR3 */

/* sigHandler - see above for description.  */
RETSIGTYPE (*sigHandler(int sig, RETSIGTYPE (*disp)(int)))(int)
{
    return sigset(sig, disp);
}

/* sigBlock - blocks the specified signal `sig' from delivery. If
   successful, it returns 0. This differs from the above function
   which return the previous boolean value of the signal block
   flag. Unfortunately, it is not possible to determine this value on
   SVR3. On failure, it returns -1.  */
/* sigBlock - see above for description.  */
int sigBlock(int sig)
{
    return sighold(sig);
}

/* sigUnblock - see above for description.  */
int sigUnblock(int sig)
{
    return sigrelse(sig);
}

#else /* !HAVE_SIGSET */

 #error No useful signal handling functions found

#endif /* !HAVE_SIGSET */

#endif /* !HAVE_SIGVEC */

#endif /* !defined(_POSIX_VERSION) */

