/*
 * Copyright (c) 1982, 1986, 1988, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	@(#)ip_icmp.c	8.2 (Berkeley) 1/4/94
 * ip_icmp.c,v 1.7 1995/05/30 08:09:42 rgrimes Exp
 */

#include "slirp.h"
#include "ip_icmp.h"

struct	icmpstat icmpstat;

/* The message sent when emulating PING */
/* Be nice and tell them it's just a psuedo-ping packet */
char icmp_ping_msg[] = "This is a psuedo-PING packet used by Slirp to emulate ICMP ECHO-REQUEST packets.\n";

extern char icmp_ping_msg[];

/*
 * Process a received ICMP message.
 */
void
icmp_input(m, hlen)
     struct mbuf *m;
     int hlen;
{
  register struct icmp *icp;
  register struct ip *ip = mtod(m, struct ip *);
  int icmplen = ip->ip_len;
  register int i;
  int code;
  struct socket *so;
  struct sockaddr_in addr;
	
  DEBUG_CALL("icmp_input");
  DEBUG_ARG("m = %lx", (long)m);
  DEBUG_ARG("m_len = %d", m->m_len);

  icmpstat.icps_received++;
	
  /*
   * Locate icmp structure in mbuf, and check
   * that not corrupted and of at least minimum length.
   */
  if (icmplen < ICMP_MINLEN) {
    icmpstat.icps_tooshort++;
    goto freeit;
  }
  i = hlen + min(icmplen, ICMP_ADVLENMIN);
  if (m->m_len < i)  {
    icmpstat.icps_tooshort++;
    return;
  }
  ip = mtod(m, struct ip *);
  m->m_len -= hlen;
  m->m_data += hlen;
  icp = mtod(m, struct icmp *);
  if (cksum(m, icmplen)) {
    icmpstat.icps_checksum++;
    goto freeit;
  }
  m->m_len += hlen;
  m->m_data -= hlen;
  
  if (icp->icmp_type > ICMP_MAXTYPE)
    goto freeit;
  /*	icmpstat.icps_inhist[icp->icmp_type]++; */
  code = icp->icmp_code;
  switch (icp->icmp_type) {
    
  case ICMP_ECHO:
    icp->icmp_type = ICMP_ECHOREPLY;
    ip->ip_len += hlen;	/* since ip_input deducts this */
    if (ip->ip_dst.s_addr == our_addr.s_addr) {
      icmp_reflect(m);
    } else {
      if ((so = socreate()) == NULL)
	goto freeit;
      udp_attach(so);
      so->so_m = m;
      so->so_faddr = ip->ip_dst;
      so->so_fport = htons(7);
      so->so_laddr = ip->ip_src;
      so->so_lport = htons(9);
      so->so_iptos = ip->ip_tos;
      so->so_type = IPPROTO_ICMP;
      so->so_state = SS_ISFCONNECTED;
      
      /* Send the packet */
      addr.sin_family = AF_INET;
      if ((so->so_faddr.s_addr & htonl(0xffffff00)) == special_addr.s_addr) {
	/* It's an alias */
	switch(ntohl(so->so_faddr.s_addr) & 0xff) {
	case CTL_DNS:
	  addr.sin_addr = dns_addr;
	  break;
	case CTL_ALIAS:
	default:
	  addr.sin_addr = loopback_addr;
	  break;
	}
      } else {
	addr.sin_addr = so->so_faddr;
      }
      addr.sin_port = htons(7);
      if(sendto(so->s, icmp_ping_msg, strlen(icmp_ping_msg), 0,
		(struct sockaddr *)&addr, sizeof(addr)) == -1) {
	DEBUG_MISC((dfd,"udp icmp tx errno = %d-%s\n",errno,strerror(errno)));
	icmp_error(m, ICMP_UNREACH,ICMP_UNREACH_NET, 0,strerror(errno)); 
	udp_detach(so);
      }
    } /* if ip->ip_dst.s_addr == our_addr.s_addr */
    break;
  case ICMP_UNREACH:
    /* XXX? report error? */
  case ICMP_TIMXCEED:
  case ICMP_PARAMPROB:
  case ICMP_SOURCEQUENCH:
  case ICMP_TSTAMP:
  case ICMP_MASKREQ:
  case ICMP_REDIRECT:
    icmpstat.icps_notsupp++;
    m_freem(m);
    break;
    
  default:
    icmpstat.icps_badtype++;
  freeit:
    m_freem(m);
  } /* swith */
  /* m is m_free()'d, put in a socket, or given to ip_send */
}


/*
 * Send ICMP_UNREACH back to the source
 * mbuf *msrc is used as a template, but is NOT m_free()'d
 * mbuf *msrc is reported as the bad ip packet.  The header should
 * be correct and in host byte order.
 */
#define ICMP_MAXDATALEN 556
void
icmp_error(msrc, type, code, minsize, message)
     struct mbuf *msrc;
     u_char type;
     u_char code;
     int minsize;
     char *message;
{
  int optlen, hlen, shlen;
  unsigned s_ip_len;
  register struct ip *ip;
  register struct icmp *icp;
  register struct mbuf *m;

  if(type != ICMP_UNREACH && type != ICMP_TIMXCEED) goto end_error;

  if(!(m=m_get())) goto end_error;               /* get mbuf */
  { int new_m_size;
    new_m_size=sizeof(struct ip )+ICMP_MINLEN+msrc->m_len+ICMP_MAXDATALEN;
    if(new_m_size>m->m_size) m_inc(m, new_m_size);
  }
  memcpy(m->m_data, msrc->m_data, msrc->m_len);
  m->m_len = msrc->m_len;                        /* copy msrc to m */

  /* make the header of the reply packet */
  ip = mtod(m, struct ip *);         
  shlen=hlen=(ip->ip_hl << 2);
  optlen=hlen-sizeof(struct ip);
  s_ip_len=ip->ip_len;

  if (optlen > 0) {
    /*
     * Strip out original options by copying rest of first
     * mbuf's data back, and adjust the IP length.
     */
    ip->ip_len -= optlen;
    hlen=sizeof(struct ip);
    ip->ip_hl = sizeof(struct ip) >> 2;
    m->m_len -= optlen;
    optlen += sizeof(struct ip );
    memmove((caddr_t)(ip + 1), (caddr_t)ip + optlen,
	    (unsigned)(m->m_len - sizeof(struct ip)));
  }
  
  /* fill in icmp */
  m->m_data += hlen;                  
  m->m_len -= hlen;

  icp = mtod(m, struct icmp *);

  if(minsize) s_ip_len=shlen;                   /* return header only */
  else if(s_ip_len>ICMP_MAXDATALEN) /* maximum size */
    s_ip_len=ICMP_MAXDATALEN;


  m->m_len=ICMP_MINLEN+s_ip_len;        /* 8 bytes ICMP header */
                                     
  if(m->m_len<ICMP_ADVLENMIN)           /* minsize = 8+sizeof(struct ip)+8 */
    m->m_len = ICMP_ADVLENMIN;          /* minimum length icmp header */

  icp->icmp_type = type;
  icp->icmp_code = code;
  icp->icmp_id = 0;
  icp->icmp_seq = 0;

  memcpy(&icp->icmp_ip, msrc->m_data, s_ip_len);   /* report the ip packet */
  HTONS(icp->icmp_ip.ip_len);
  HTONS(icp->icmp_ip.ip_id);
  HTONS(icp->icmp_ip.ip_off);

#if DEBUG
  if(message) {           /* DEBUG : append message to ICMP packet */
    int message_len;
    char *cpnt;
    message_len=strlen(message);
    if(message_len>=ICMP_MAXDATALEN) message_len=ICMP_MAXDATALEN-1;
    cpnt=(char *)m->m_data+m->m_len;
    *cpnt++=0;
    memcpy(cpnt, message, message_len);
    m->m_len+=message_len+1;
  }
#endif

  icp->icmp_cksum = 0;
  icp->icmp_cksum = cksum(m, m->m_len);

  m->m_data -= hlen;
  m->m_len += hlen;

  /* fill in ip */
  ip->ip_len = m->m_len;
  
  ip->ip_tos=((ip->ip_tos & 0x1E) | 0xC0);

  ip->ip_p = IPPROTO_ICMP;
  ip->ip_dst = ip->ip_src;    /* ip adresses */
  ip->ip_src = our_addr;
  ip->ip_ttl = MAXTTL;

  (void) ip_output((struct socket *)NULL, m);
  
  icmpstat.icps_reflect++;

end_error:
  return;
}
#undef ICMP_MAXDATALEN

/*
 * Reflect the ip packet back to the source
 */
void
icmp_reflect(m)
	struct mbuf *m;
{
	register struct ip *ip = mtod(m, struct ip *);
	int optlen = (ip->ip_hl << 2) - sizeof(struct ip);
	struct in_addr icmp_dst;
	
	icmp_dst = ip->ip_dst;
	ip->ip_dst = ip->ip_src;
	ip->ip_src = icmp_dst;
	ip->ip_ttl = MAXTTL;

	if (optlen > 0) {
		/*
		 * Strip out original options by copying rest of first
		 * mbuf's data back, and adjust the IP length.
		 */
		ip->ip_len -= optlen;
		ip->ip_hl = sizeof(struct ip) >> 2;
		m->m_len -= optlen;
		optlen += sizeof(struct ip);
		memmove((caddr_t)(ip + 1), (caddr_t)ip + optlen,
			 (unsigned)(m->m_len - sizeof(struct ip)));
	}
	icmpstat.icps_reflect++;
	icmp_send(m);
}

/*
 * Send an icmp packet back to the ip level,
 * after supplying a checksum.
 */
void
icmp_send(m)
	register struct mbuf *m;
{
	register struct ip *ip = mtod(m, struct ip *);
	register int hlen;
	register struct icmp *icp;

	hlen = ip->ip_hl << 2;
	m->m_data += hlen;
	m->m_len -= hlen;
	icp = mtod(m, struct icmp *);
	icp->icmp_cksum = 0;
	icp->icmp_cksum = cksum(m, ip->ip_len - hlen);
	m->m_data -= hlen;
	m->m_len += hlen;
	(void) ip_output((struct socket *)NULL, m);
}
