/*

 X-Chat

Copyright (c) 1998 Peter Zelezny.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "xchat.h"
#include "draw.h"
#include "dcc.h"
#include "style.h"
#include <time.h>

extern struct DCC *find_dcc(char *nick, char *file, int type);
extern void dcc_close(struct DCC *dcc, int stat, int destroy);
extern struct DCC *new_dcc(void);
extern void PrintText(struct session *sess, char *text);
extern void dcc_connect(struct session *sess, struct DCC *dcc);
extern int dcc_listen_init(struct DCC *dcc);


void dcc_close_draw_window(struct DCC *dcc)
{
   gtk_widget_destroy(dcc->dccdraw->window);
   free(dcc->dccdraw);
   dcc->dccdraw = 0;
}

void kill_dccdraw(GtkWidget *win, struct DCC *dcc)
{
   dcc_close(dcc, 0, TRUE);
}

static gint configure_event (GtkWidget *widget, GdkEventConfigure *event, struct DCC *dcc)
{
   if(dcc->dccdraw->pixmap) gdk_pixmap_unref(dcc->dccdraw->pixmap);

   dcc->dccdraw->pixmap = gdk_pixmap_new(widget->window,
	       	                     widget->allocation.width,
	                             widget->allocation.height,
	                             -1);
   gdk_draw_rectangle (dcc->dccdraw->pixmap,
		       widget->style->white_gc,
                       TRUE,
		       0, 0,
		       widget->allocation.width,
		       widget->allocation.height);
   return TRUE;
}

         /* Redraw the screen from the backing pixmap */
static gint expose_event (GtkWidget *widget, GdkEventExpose *event, struct DCC *dcc)
{
   gdk_draw_pixmap(widget->window,
	           widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
	           dcc->dccdraw->pixmap,
                   event->area.x, event->area.y,
                   event->area.x, event->area.y,
                   event->area.width, event->area.height);
   return FALSE;
}

void dcc_draw_pixel(struct DCC *dcc, int x, int y)
{
   char buf[sizeof(struct xchatpixel)+2];
   struct xchatpixel pix;
   pix.x = htons(x);
   pix.y = htons(y);
   buf[0] = 1;  // packet type: 1 = draw pixel
   buf[1] = 4;  // packet size: 4 bytes
   memcpy((char *)&buf[2], (char *)&pix, sizeof(struct xchatpixel));
   send(dcc->sok, buf, sizeof(struct xchatpixel) + 2, 0);
}

        /* Draw a rectangle on the screen */
static void draw_brush (GtkWidget *widget, gdouble x, gdouble y, struct DCC *dcc, int send)
{
   GdkRectangle update_rect;

   update_rect.x = x - 1;
   update_rect.y = y - 1;
   update_rect.width = 2;
   update_rect.height = 2;
   gdk_draw_rectangle(dcc->dccdraw->pixmap,
	               widget->style->black_gc,
	               TRUE,
		       update_rect.x, update_rect.y,
	               update_rect.width, update_rect.height);
   gtk_widget_draw(widget, &update_rect);
   if(send) dcc_draw_pixel(dcc, x, y);
}

static gint button_press_event (GtkWidget *widget, GdkEventButton *event, struct DCC *dcc)
{
   if (event->button == 1 && dcc->dccdraw->pixmap != NULL)
                 draw_brush (widget, event->x, event->y, dcc, TRUE);
   return TRUE;
}

static gint motion_notify_event (GtkWidget *widget, GdkEventMotion *event, struct DCC *dcc)
{
   int x, y;
   GdkModifierType state;

   if (event->is_hint)
      gdk_window_get_pointer (event->window, &x, &y, &state);
   else {
      x = event->x;
      y = event->y;
      state = event->state;
   }

   if (state & GDK_BUTTON1_MASK && dcc->dccdraw->pixmap != NULL)
      draw_brush (widget, x, y, dcc, TRUE);

   return TRUE;
}

void dcc_open_draw_window(struct DCC *dcc)
{
   GtkWidget *vbox;
   char tbuf[128];

   dcc->dccdraw = malloc(sizeof(struct dccdraw));
   memset(dcc->dccdraw, 0, sizeof(struct dccdraw));
   dcc->dccdraw->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

   sprintf(tbuf, "DCC-Draw: %s", dcc->nick);
   gtk_window_set_title(GTK_WINDOW(dcc->dccdraw->window), tbuf);
   gtk_signal_connect(GTK_OBJECT(dcc->dccdraw->window), "destroy",
                      GTK_SIGNAL_FUNC(kill_dccdraw), dcc);
   gtk_widget_set_usize(dcc->dccdraw->window, dcc->pos+4, dcc->size+4);

   vbox = gtk_vbox_new (FALSE, 0);
   gtk_container_add (GTK_CONTAINER (dcc->dccdraw->window), vbox);
   gtk_widget_show (vbox);

   dcc->dccdraw->drawing_area = gtk_drawing_area_new ();
   gtk_drawing_area_size (GTK_DRAWING_AREA (dcc->dccdraw->drawing_area), dcc->pos, dcc->size);
   gtk_box_pack_start (GTK_BOX (vbox), dcc->dccdraw->drawing_area, TRUE, TRUE, 0); 
   gtk_widget_show (dcc->dccdraw->drawing_area);

   gtk_signal_connect(GTK_OBJECT(dcc->dccdraw->drawing_area), "expose_event",
		      (GtkSignalFunc) expose_event, dcc);
   gtk_signal_connect(GTK_OBJECT(dcc->dccdraw->drawing_area),"configure_event",
	              (GtkSignalFunc) configure_event, dcc);
   gtk_signal_connect(GTK_OBJECT(dcc->dccdraw->drawing_area), "motion_notify_event",
		      (GtkSignalFunc) motion_notify_event, dcc);
   gtk_signal_connect(GTK_OBJECT (dcc->dccdraw->drawing_area), "button_press_event",
                      (GtkSignalFunc) button_press_event, dcc);

   gtk_widget_set_events (dcc->dccdraw->drawing_area, GDK_EXPOSURE_MASK
			              | GDK_LEAVE_NOTIFY_MASK
		                      | GDK_BUTTON_PRESS_MASK
		                      | GDK_POINTER_MOTION_MASK
			              | GDK_POINTER_MOTION_HINT_MASK);
   gtk_widget_show(dcc->dccdraw->window);
}

void dcc_read_draw(struct DCC *dcc, gint sok)
{
   char buf[256];
   long len;
   unsigned char type[2];
   struct xchatpixel pix;

   len = recv(dcc->sok, type, 2, 0);
   if(len < 1)
   {
      dcc_close(dcc, 0, TRUE);
      return;
   }
   switch(type[0])
   {
    case 1:
      len = recv(dcc->sok, (char *)&pix, sizeof(struct xchatpixel), 0);
      if(len < 1)
      {
	 dcc_close(dcc, 0, TRUE);
	 return;
      }
      pix.x = ntohs(pix.x);
      pix.y = ntohs(pix.y);
      draw_brush(dcc->dccdraw->drawing_area, pix.x, pix.y, dcc, FALSE);
      break;
    default:
      recv(dcc->sok, buf, type[1], 0);
      printf("Unsupported DRAW packet: %d\n", type[0]);
   }
}

void dcc_draw(struct session *sess, char *nick)
{
   char outbuf[256];
   struct DCC *dcc;

   dcc = find_dcc(nick, "", TYPE_DRAWSEND);
   if(dcc)
   {
      switch(dcc->stat)
      {
       case STAT_ACTIVE:
       case STAT_QUEUED:
       case STAT_CONNECTING:
	 sprintf(outbuf, STARTON" Already offering DCC DRAW to %s\n", nick);
	 PrintText(sess, outbuf);
	 return;
       case STAT_ABORTED:
       case STAT_FAILED:
	 dcc_close(dcc, 0, TRUE);
      }
   }

   dcc = find_dcc(nick, "", TYPE_DRAWRECV);
   if(dcc)
   {
      switch(dcc->stat)
      {
	 case STAT_QUEUED:
	   dcc_connect(0, dcc);
	   break;
	 case STAT_FAILED:
	 case STAT_ABORTED:
	   dcc_close(dcc, 0, TRUE);
      }
      return;
   }

   // offer DCC DRAW

   dcc = new_dcc();
   if(!dcc) return;
   dcc->pos = 400;
   dcc->size = 300;
   dcc->sess = sess;
   dcc->stat = STAT_QUEUED;
   dcc->sok = -1;
   dcc->type = TYPE_DRAWSEND;
   time(&dcc->starttime);
   strcpy(dcc->nick, nick);
   if(dcc_listen_init(dcc))
   {
      sprintf(outbuf, "PRIVMSG %s :\001DCC DRAW 400 300 %lu %ld\001\r\n",
		 nick, dcc->addr, dcc->port);
      send(dcc->sess->server->sok, outbuf, strlen(outbuf), 0);
      sprintf(outbuf, STARTON" Offering DCC DRAW to %s\n", nick);
      PrintText(sess, outbuf);
   } else
      dcc_close(dcc, 0, TRUE); 
}
