/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>

#ifdef linux
#include <endian.h>
#elif defined (FreeBSD)
#include <machine/endian.h>
#elif defined (sgi)
#include <sys/endian.h>
#elif defined (sun)
#include <sys/byteorder.h>
#endif

#include <X11/Intrinsic.h>

#include "types.h"
#include "audio_file.h"
#include "riff.h"
#include "xwave.h"
#include "ccitt/g711.h"
#include "ccitt/g72x.h"
#include "adpcm2pcm/adpcm.h"
#include "endian.h"
 

extern Main_Data *MD;

static int riff_new(Audio_File *af);


bool is_riff(int fd)
{
   RiffHeader rh;

   if (read(fd,&rh,sizeof(RiffHeader))!=sizeof(RiffHeader)) return(False);
   if (lseek(fd,0,SEEK_SET)==-1) return(False);

#ifdef big_endian
   M_32_SWAP(rh.riff);
   M_32_SWAP(rh.wave);
#endif

   if (rh.riff!=RIFF_MAGIC) return(False);
   if (rh.wave!=WAVE_MAGIC) return(False);
   return(True);
}

int riff_open(Audio_File *af,int mode)
{
   RiffHeader rh;
   FmtHeader fh;
   long chunk;
   int headoffs,i;
   ulong length,slength;

   if (mode==AF_NEW) return(riff_new(af));
   
   if (read(af->fd,&rh,sizeof(RiffHeader))!=sizeof(RiffHeader)) 
     return(AF_ERROR);

#ifdef big_endian
   M_32_SWAP(rh.riff);
   M_32_SWAP(rh.length);
   M_32_SWAP(rh.wave);
#endif

   if (read(af->fd,&fh,sizeof(FmtHeader))!=sizeof(FmtHeader)) 
     return(AF_ERROR);

#ifdef big_endian
   M_32_SWAP(fh.fmt);
   M_32_SWAP(fh.chunk_length);
#endif

   if (fh.fmt!=FMT_MAGIC) return(AF_NOTSUPPORTED);
   
   /* seek to begin of data header */
   if ((headoffs=lseek(af->fd,fh.chunk_length-16,SEEK_CUR))==-1) 
     return(AF_ERROR);

   while (True) {
      if ((i=read (af->fd,&chunk,sizeof(chunk)))!=sizeof(chunk)) 
	return(AF_ERROR);
      headoffs+=i;
      if ((i=read (af->fd,&length,sizeof(ulong)))!=sizeof(ulong)) 
	return(AF_ERROR);
      headoffs+=i;
      
#ifdef big_endian
      M_32_SWAP(chunk);
      M_32_SWAP(length);
#endif
      
      if (chunk==DATA_MAGIC) break;
      if (lseek(af->fd,length,SEEK_CUR)==-1) return(AF_ERROR);
   }
   
#ifdef big_endian
   M_16_SWAP(fh.format);
   M_16_SWAP(fh.modus);
   M_32_SWAP(fh.freq);
   M_32_SWAP(fh.bpsec);
   M_16_SWAP(fh.bpspl);
   M_16_SWAP(fh.res);
#endif

   af->bps=fh.res;
   af->channels=fh.modus;
   af->type=AF_RIFF;
   af->freq=fh.freq;
   af->length=length;
   af->headoffs=headoffs;

   slength=lseek(af->fd,0,SEEK_END);
   if ((length-headoffs)>slength) af->length=slength;
   lseek(af->fd,headoffs,SEEK_SET);
   
   switch (fh.format) {
    case RIFF_PCM: 
      af->comp=AF_PCM;
      break;
    case RIFF_ALAW:
      af->length*=2;af->bps*=2;
      af->comp=AF_ALAW;
      break;
    case RIFF_MULAW: 
      af->length*=2;af->bps*=2;
      af->comp=AF_MULAW;
      break;
    case RIFF_IDVI_ADPCM:
      af->length*=4;af->bps*=4;
      af->comp=AF_ADPCM;
      break;
    default:
      return(AF_NOTSUPPORTED);
   }

   return(AF_RIFF);
}

int riff_new(Audio_File *af)
{
   RiffHeader rh;
   FmtHeader fh;
   long chunk;
   ulong length,count;

   length=af->length;

   switch (af->comp) {
    case AF_PCM:
      fh.format = RIFF_PCM;
      break;
    case AF_ALAW:
    case AF_MULAW:
      if (af->bps!=16) return(AF_NOTSUPPORTED);
      if (af->comp==AF_ALAW) fh.format = RIFF_ALAW;
      if (af->comp==AF_MULAW) fh.format = RIFF_MULAW;
      length/=2;
      break;
    case AF_ADPCM:
      if (af->bps!=16) return(AF_NOTSUPPORTED);
      fh.format=RIFF_IDVI_ADPCM;
      length/=4;
      break;
    default:
      return(AF_NOTSUPPORTED);
   }

   rh.riff=RIFF_MAGIC;
   rh.length=36+ length;
   rh.wave=WAVE_MAGIC;


#ifdef big_endian
   M_32_SWAP(rh.riff);
   M_32_SWAP(rh.length);
   M_32_SWAP(rh.wave);
#endif
   
   if ((af->headoffs=write (af->fd,&rh,sizeof(RiffHeader)))
       !=sizeof(RiffHeader)) return(AF_ERROR);

   fh.fmt=FMT_MAGIC;
   fh.chunk_length = 16;
   fh.res=af->bps;
   fh.modus = af->channels;
   fh.freq = af->freq;

   switch (af->comp) {
    case AF_ALAW:
    case AF_MULAW:
      fh.res/=2;
      break;
    case AF_ADPCM:
      fh.res/=4;
      break;
   }
   
   fh.bpspl=fh.modus*(fh.res / 8);
   fh.bpsec = fh.freq * fh.bpspl;

#ifdef big_endian
   M_32_SWAP(fh.fmt);
   M_32_SWAP(fh.chunk_length);
   M_16_SWAP(fh.format);
   M_16_SWAP(fh.modus);
   M_32_SWAP(fh.freq);
   M_32_SWAP(fh.bpsec);
   M_16_SWAP(fh.bpspl);
   M_16_SWAP(fh.res);
#endif
   
   if ((count=write (af->fd,&fh,sizeof(FmtHeader)))!=sizeof(FmtHeader))
     return(AF_ERROR);
   
   af->headoffs+=count;
   
   chunk=DATA_MAGIC;

#ifdef big_endian
   M_32_SWAP(chunk);
   M_32_SWAP(length);
#endif
   
   if ((count=write (af->fd,&chunk,sizeof(chunk)))!=sizeof(chunk)) 
     return(AF_ERROR);
   af->headoffs+=count;

   if ((count=write (af->fd,&length,sizeof(length)))!=sizeof(length)) 
     return(AF_ERROR);
   af->headoffs+=count;
   
   return(AF_RIFF);
}

int riff_read(Audio_File af,char *buffer,int size)
{
   switch (af.comp) {
    case AF_PCM:
      return(read(af.fd,buffer,size));
    case AF_ALAW:
    case AF_MULAW: {
       short *linear=(short *) buffer;
       byte *law;
       register int i;
       if ((law = (byte *) malloc(size/2)) == NULL) return(AF_ERROR);
       if ((size=read(af.fd,law,size/2))==-1) {
	  free(law);
	  return(AF_ERROR);
       }
       if (af.comp==AF_ALAW) for(i=0;i<size;i++) linear[i]=alaw2linear(law[i]);
       if (af.comp==AF_MULAW)for(i=0;i<size;i++) linear[i]=ulaw2linear(law[i]);
       free(law);
       return(size*2);
    }
    case AF_ADPCM: {
       struct adpcm_state decoder_state;
       short *linear=(short*) buffer;
       char *abuf;
       if ((abuf=malloc(size/4))==NULL) return(AF_ERROR);
       if ((size=read(af.fd,abuf,size/4))==-1) {
	  free(abuf);
	  return(AF_ERROR);
       }
       adpcm_decoder(abuf,linear, size*2, &decoder_state);
       free(abuf);
       return(size*4);
    }
   }
   return(AF_ERROR);
}

int riff_write(Audio_File af,char *buffer,int size)
{
   switch (af.comp) {
    case AF_PCM:
      return(write(af.fd,buffer,size));
    case AF_ALAW:
    case AF_MULAW: {
       short *linear=(short *) buffer;
       byte *law;
       register int i;
       if ((law = (byte *) malloc(size/2)) == NULL) return(AF_ERROR);
       if (af.comp==AF_ALAW) 
	 for(i=0;i<size/2;i++) law[i]=linear2alaw(linear[i]);
       if (af.comp==AF_MULAW)
	 for(i=0;i<size/2;i++) law[i]=linear2ulaw(linear[i]);
       if ((size=write(af.fd,law,size/2))==-1) {
	  free(law);
	  return(AF_ERROR);
       }
       free(law);
       return(size*2);
    }
    case AF_ADPCM: {
       struct adpcm_state coder_state;
       short *linear=(short*) buffer;
       char *abuf;
       if ((abuf=malloc(size/4))==NULL) return(AF_ERROR);
       adpcm_coder(linear, abuf, size/2, &coder_state);
       if ((size=write(af.fd,abuf,size/4))==-1) {
	  free(abuf);
	  return(AF_ERROR);
       }
       free(abuf);
       return(size*4);
    }
   }
   return(AF_ERROR);
}

int riff_seek(Audio_File af,int pos,int mode)
{
   switch (af.comp) {
    case AF_PCM:
      return(lseek(af.fd,pos,mode));
    case AF_ALAW:
    case AF_MULAW:
      return(lseek(af.fd,pos/2,mode));
    case AF_ADPCM:
      return(lseek(af.fd,pos/4,mode));
   }
   return(AF_ERROR);
}
   
int riff_close(Audio_File af)
{
   if (riff_seek(af,0,SEEK_SET)==AF_ERROR) return(AF_ERROR);
     
   if (riff_new (&af)==AF_ERROR) return(AF_ERROR);
   return(close(af.fd));
}

