/*
 *  keys.c:		Key dialog window	
 *
 *  Written by:		Ullrich Hafner
 *		
 *  Copyright (C) 1998 Ullrich Hafner <hafner@informatik.uni-wuerzburg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1998/10/08 22:03:20 $
 *  $Author: hafner $
 *  $Revision: 1.11 $
 *  $State: Exp $
 */

#include "config.h"

#if HAVE_STRING_H
#	include <string.h>
#else /* not HAVE_STRING_H */
#	include <strings.h>
#endif /* not HAVE_STRING_H */

#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <X11/keysym.h>
#include <proplist.h>

#include "keys.h"
#include "misc.h"
#include "error.h"

/*******************************************************************************

			     global variables
  
*******************************************************************************/

extern GtkTooltips	 *tooltips;
extern proplist_t	 *windowmaker;
extern bool_t		 changed;
extern proplist_t	 *plist_changed;
extern proplist_t	 *pl_yes;

/*******************************************************************************

			     local variables
  
*******************************************************************************/

typedef struct key_wigets
{
   GtkWidget  *control;
   GtkWidget  *meta;
   GtkWidget  *shift;
   GtkWidget  *keyname;
   proplist_t *key;
} key_wigets_t;

/*******************************************************************************

				prototypes
  
*******************************************************************************/

static GList *
keysym_list (void);
static int
compare (const void *string1, const void *string2);
static void
set_key (GtkWidget *widget, gpointer ptr);

/*******************************************************************************

				public code
  
*******************************************************************************/

GtkWidget *
key_dialog (proplist_t *key, proplist_t *info)
{
   GtkWidget	*vbox = gtk_vbox_new (FALSE, 0);
   bool_t	control, meta, shift;
   unsigned	number;
   GList	*list = keysym_list ();
   key_wigets_t	*data = Calloc (1, sizeof (key_wigets_t));

   data->key = key;
   
   /*
    *  Extract modifiers and keysym from attribute 'key'
    */
   {
      char *s = strdup (PLGetString (PLGetDictionaryEntry (windowmaker, key)));

      control = meta = shift = FALSE;
      number  = 0;

      if (!strcaseeq (s, "None"))
      {
	 char *next;
	 char *cur = s;
	 
	 while ((next = strchr (cur, '+')))
	 {
	    *next = 0;
	    if (strcaseeq (cur, "Control"))
	       control = TRUE;
	    else if (strcaseeq (cur, "Meta") || strcaseeq (cur, "Alt")
		      || strcaseeq (cur, "Mod1"))
	       meta = TRUE;
	    else if (strcaseeq (cur, "Shift"))
	       shift = TRUE;
	    cur = next + 1;
	 }
	 if (XStringToKeysym (cur) != NoSymbol)
	 {
	    GList *l = g_list_first (list);
	    
	    while (l)
	    {
	       if (strcaseeq ((char *) l->data, cur))
		  break;
	       number++;
	       l = l->next;
	    }
	    if (number > g_list_length (list))
	       number = 0;
	 }
      }
      Free (s);
   }
   
   /*
    *  Check buttons for control and meta key
    */
   {
      GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
      GtkWidget *button;
	    
      gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

      data->control = button = gtk_check_button_new_with_label ("Control");
      gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), control);
      gtk_tooltips_set_tip (tooltips, button, PLGetString (info), NULL);
      gtk_signal_connect (GTK_OBJECT (button), "toggled",
			  GTK_SIGNAL_FUNC (set_key), (gpointer) data);

      data->meta = button = gtk_check_button_new_with_label ("Meta");
      gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), meta);
      gtk_tooltips_set_tip (tooltips, button, PLGetString (info), NULL);
      gtk_signal_connect (GTK_OBJECT (button), "toggled",
			  GTK_SIGNAL_FUNC (set_key), (gpointer) data);

      data->shift = button = gtk_check_button_new_with_label ("Shift");
      gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), shift);
      gtk_tooltips_set_tip (tooltips, button, PLGetString (info), NULL);
      gtk_signal_connect (GTK_OBJECT (button), "toggled",
			  GTK_SIGNAL_FUNC (set_key), (gpointer) data);
   }
   /*
    *  Combo box for all available keysyms
    */
   {
      GtkWidget	*cb;

      cb = gtk_combo_new ();
      gtk_combo_set_popdown_strings (GTK_COMBO (cb), list);
      gtk_widget_set_usize (GTK_COMBO (cb)->entry, 120, 0);
      gtk_widget_set_usize (cb, 120, 0);
      gtk_box_pack_start (GTK_BOX (vbox), cb, FALSE, TRUE, 0);
      gtk_tooltips_set_tip (tooltips, GTK_COMBO (cb)->entry,
			    PLGetString (info), NULL);
      gtk_list_select_item (GTK_LIST (GTK_COMBO (cb)->list), number);
      gtk_signal_connect (GTK_OBJECT (GTK_COMBO (cb)->entry), "changed",
			  GTK_SIGNAL_FUNC (set_key), (gpointer) data);
      data->keyname = GTK_COMBO (cb)->entry;
   }

   if (!number)
   {
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->control), FALSE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->meta), FALSE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->shift), FALSE);
      gtk_widget_set_sensitive (data->control, FALSE);
      gtk_widget_set_sensitive (data->meta, FALSE);
      gtk_widget_set_sensitive (data->shift, FALSE);
   }
      
   gtk_widget_show_all (vbox);
   
   return vbox;
}

/*******************************************************************************

				private code
  
*******************************************************************************/

static GList *
keysym_list (void)
{
   int		m, M;			/* minimum and maximum keycode */
   int		kpk;			/* keys per keycode */
   int		i, k;			/* counter */
   KeySym	*keysyms;		/* all available keysyms */
   char		**keyname;		/* array of keynames */
   static GList	*list = NULL;		/* list of keynames */

   if (list)
      return list;			/* already computed */

   XDisplayKeycodes (GDK_DISPLAY(), &m, &M);
   keysyms = XGetKeyboardMapping (GDK_DISPLAY(), m, M - m + 1, &kpk);

   keyname = Calloc ((M - m + 1 + 1), sizeof (char *));
	 
   for (i = k = 0; i < (M - m + 1); i++)
      if (!IsModifierKey (keysyms [i * kpk]))
      {
	 char *name = XKeysymToString (keysyms [i * kpk]);

	 if (name)
	 {
	    keyname [k] = strdup (name);
	    if (isalpha (*keyname [k]))
	       *keyname [k] = toupper (*keyname [k]);
	    k++;
	 }
      }
   XFree (keysyms);

   qsort (keyname, k, sizeof (char *), compare);

   list = g_list_append (NULL, "None");
   for (i = 0; i < k; i++)
      list = g_list_append (list, keyname [i]);

   Free (keyname);

   return list;
}

static int
compare (const void *string1, const void *string2)
/*
 *  Compare two keynames. Sort by 1) string length and 2) alphabet.
 *
 *  Return value:
 *	<  0 : string1 < string2
 *	== 0 : string1 = string2
 *	>  0 : string1 > string2
 */
{
   const char	*s1 = *(char **) string1;
   const char	*s2 = *(char **) string2;
   unsigned	l1  = strlen (s1);
   unsigned	l2  = strlen (s2);

   /*
    *  KP_ entries should be grouped together
    */
   {
      char *t1 = strdup (s1);
      char *t2 = strdup (s2);

      if (l1 > 3)
	 t1 [3] = 0;
      if (l2 > 3)
	 t2 [3] = 0;

      if (strcaseeq (t1, "KP_") && !strcaseeq (t2, "KP_"))
      {
	 Free (t1);
	 Free (t2);
	 return +1;
      }
      if (strcaseeq (t2, "KP_") && !strcaseeq (t1, "KP_"))
      {
	 Free (t1);
	 Free (t2);
	 return -1;
      }
      Free (t1);
      Free (t2);
   }
   /*
    *  Fnum entries should be grouped together
    */
   if (s1 [0] != 'F' && l1 > 1)
      l1 +=2;
   if (s2 [0] != 'F' && l2 > 1)
      l2 +=2;

   if (l1 > 3 && l2 > 3)
      return strcmp (s1, s2);
   
   if (l1 != l2)
      return l1 - l2;

   if (l1 == 1)				/* letter or digit */
      return *s1 - *s2;

   return 0;
}

static void
set_key (GtkWidget *widget, gpointer ptr)
/*
 *  Change key attribute
 */
{
   key_wigets_t	*data = (key_wigets_t *) ptr;
   char		*text = gtk_entry_get_text (GTK_ENTRY (data->keyname));

   if (strcaseeq (text, "None"))
   {
      PLInsertDictionaryEntry (windowmaker, data->key, PLMakeString ("None"));
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->control), FALSE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->meta), FALSE);
      gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (data->shift), FALSE);
      gtk_widget_set_sensitive (data->control, FALSE);
      gtk_widget_set_sensitive (data->meta, FALSE);
      gtk_widget_set_sensitive (data->shift, FALSE);
   }
   else
   {
      char tmp [MAXSTRLEN];

      gtk_widget_set_sensitive (data->control, TRUE);
      gtk_widget_set_sensitive (data->meta, TRUE);
      gtk_widget_set_sensitive (data->shift, TRUE);

      tmp [0] = 0;
      if (GTK_TOGGLE_BUTTON (data->control)->active)
	 strcat (tmp, "Control+");
      if (GTK_TOGGLE_BUTTON (data->meta)->active)
	 strcat (tmp, "Meta+");
      if (GTK_TOGGLE_BUTTON (data->shift)->active)
	 strcat (tmp, "Shift+");
      strcat (tmp, text);
   
      PLInsertDictionaryEntry (windowmaker, data->key, PLMakeString (tmp));
   }
   
   PLInsertDictionaryEntry (plist_changed, data->key, pl_yes);
   changed = YES;
}
