/*
 *  tileofday.c:	Tile of the day support
 *
 *  Written by:		Ullrich Hafner
 *  
 *  Copyright (C) 1998 Ullrich Hafner <hafner@informatik.uni-wuerzburg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

/*
 *  $Date: 1998/08/22 21:13:13 $
 *  $Author: hafner $
 *  $Revision: 1.30 $
 *  $State: Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <proplist.h>
#include <stdio.h>

#include "misc.h"
#include "dialog.h"
#include "window.h"
#include "texture.h"
#include "tileofday.h"
#include "icons.h"

#include "error.h"

#ifdef TILE_OF_DAY

/*******************************************************************************

			global variables 
  
*******************************************************************************/

extern char		 *wmpath;	/* wmconfig.c */
extern bool_t		 changed;	/* window.c */
extern proplist_t	 *pixmap_path;	/* window.c */
extern GtkTooltips	 *tooltips;	/* window.c */
extern proplist_t	 *windowmaker;	/* window.c */
extern proplist_t	 *plist_changed;
extern proplist_t	 *pl_yes;

bool_t   cron_changed  = NO;

/*******************************************************************************

		          local variables
  
*******************************************************************************/

static char	**cron_table = NULL;
static char	*cron_minute = NULL;
static char	*cron_hour   = NULL;
static char	*cron_day    = NULL;
static bool_t	cron_set     = NO;
static char	*tile_string = "tile-of-the-day";

static GtkWidget *pixmap_tileofday [3];
static GtkWidget *combo [3];
static GtkWidget *toggle_button;

/*******************************************************************************

				prototypes
  
*******************************************************************************/

static void
get_tileofday (GtkWidget *widget, gpointer ptr);
static void
set_tileofday (GtkWidget *widget, gpointer ptr);
static void
set_combo (GtkWidget *widget, gpointer ptr);
static void
set_toggle (GtkWidget *widget, gpointer ptr);
static void
set_default_time (GtkWidget *widget, gpointer ptr);
static void
tile_toggle (GtkWidget *widget, gpointer ptr);

/*******************************************************************************

				public code
  
*******************************************************************************/

void
tileofday_dialog (GtkWidget *widget, gpointer ptr)
/*
 *  Dialog window for tile of the day configuration
 *
 *  No return value.
 */
{
   static GtkWidget *window  = NULL;
   char		    *tiles[] = {"tile-of-the-day", "random-tile",
			       "guest-tile", NULL};
   char		    **tile;
   
   for (tile = tiles; *tile; tile++)
   {
      char *path = get_pixmap_path (*tile);

      if (path)
      {
	 Free (path);
	 continue;
      }
      else
      {
	 unsigned n;
	 bool_t	  success = NO;
	 /*
	  *  Generate an empty pixmap file 'tile'
	  */
	 for (n = 0; n < PLGetNumberOfElements (pixmap_path) && !success; n++)
	 {
	    char *path = PLGetString (PLGetArrayElement (pixmap_path, n));
	    char *tmp;
	    FILE *dst;

	    path = expand_tilde (path);
	    tmp  = Calloc (strlen (path) + strlen (*tile) + 2, sizeof (char));

	    strcpy (tmp, path);
	    strcat (tmp, "/");
	    strcat (tmp, *tile);
	 
	    dst     = fopen (tmp, "w");
	    success = write_bullethole (dst);
	    Free (path);
	 }
	 if (!success)
	 {
	    char *text = "Can't generate tile template `%s'\n"
			 "in neither of your PixmapPath directories.\n"
			 "Please ensure that you have write access\n"
			 "for at least one directory.";
	    char *msg;

	    msg = Calloc (strlen (text) + strlen (*tile) + 1, sizeof (char));
	    sprintf (msg, text, *tile);
	    
	    dialog_popup (DIALOG_ERROR, msg, NULL, NULL);
	    Free (msg);
	    return;
	 }
      }
   }
   
   if (!window)
   {
      GtkWidget	 *box, *button;
      wtexture_t *wtexture = (wtexture_t *) ptr;
      GSList	 *group    = NULL;
      
      window = gtk_dialog_new ();
      gtk_window_set_title (GTK_WINDOW (window), "Tiles from themes.org");
      gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

      gtk_signal_connect (GTK_OBJECT (window), "delete_event",
			  GTK_SIGNAL_FUNC (gtk_false), NULL);
      gtk_signal_connect (GTK_OBJECT (window), "destroy",
			  (GtkSignalFunc) gtk_widget_destroyed,
			  &window);

      box = gtk_hbox_new (FALSE, 5);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox),
			  box, FALSE, FALSE, 10);

#ifdef HAVE_IMLIB
      {
	 GtkWidget *frame;
      
	 frame = gtk_frame_new (NULL);
	 gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	 gtk_container_border_width (GTK_CONTAINER (frame), 0);
	 gtk_container_add (GTK_CONTAINER (frame),
			    make_pixmap (PKGDATADIR "/themes.org.jpg", 0, 0,
					 NULL));
	 gtk_box_pack_start (GTK_BOX (box), frame, TRUE, FALSE, 10);
      }
#endif  /* not HAVE_IMLIB */

      box = gtk_hbox_new (FALSE, 0);
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox),
			  box, TRUE, FALSE, 5);

      for (tile = tiles; *tile; tile++)
      {
	 GtkWidget *vbox, *hbox, *button, *frame;

	 vbox = gtk_vbox_new (FALSE, 0);
	 gtk_box_pack_start (GTK_BOX (box), vbox, TRUE, FALSE, 5);

	 hbox = gtk_hbox_new (FALSE, 0);
	 gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 5);
	 frame = gtk_frame_new (NULL);
	 gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_IN);
	 gtk_container_border_width (GTK_CONTAINER (frame), 0);
	 pixmap_tileofday [tile - tiles] = make_pixmap (*tile, 64, 64, NULL);
	 gtk_container_add (GTK_CONTAINER (frame),
			    pixmap_tileofday [tile - tiles]);
	 gtk_box_pack_start (GTK_BOX (hbox), frame, TRUE, FALSE, 5);

	 button = gtk_radio_button_new_with_label (group, *tile);
	 gtk_object_set_user_data (GTK_OBJECT (button), (gpointer) wtexture);
	 group  = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
	 gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 5);
	 gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button),
				      streq (tile_string, *tile));
	 gtk_signal_connect (GTK_OBJECT (button), "toggled",
			     GTK_SIGNAL_FUNC (tile_toggle), *tile);
      }
      
#if defined(HAVE_POPEN) && defined(CRONTAB)
      {
	 GtkWidget *hbox;
	 GtkWidget *vbox2;
	 GtkWidget *button;
	 GtkWidget *label;

	 box = GTK_DIALOG (window)->vbox;
	 gtk_box_pack_start (GTK_BOX (box), gtk_hseparator_new (),
			     FALSE, TRUE, 0);
	 
	 hbox = gtk_hbox_new (FALSE, 10);
	 gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, TRUE, 10);
      
	 vbox2 = gtk_vbox_new (FALSE, 0);
	 gtk_box_pack_start (GTK_BOX (hbox), vbox2, FALSE, TRUE, 5);
	 toggle_button 
	    	= gtk_check_button_new_with_label ("Periodically get tile at");
	 button = toggle_button;
	 gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button), cron_set);
	 gtk_signal_connect (GTK_OBJECT (button), "toggled",
			     GTK_SIGNAL_FUNC (set_toggle), (gpointer) wtexture);
	 gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, TRUE, 5);
	 gtk_tooltips_set_tip (tooltips, button,
			       "Add/remove entry in your crontab file to"
			       " automatically get selected tile"
			       " at specified time and day.", NULL);  
	 
	 vbox2 = gtk_vbox_new (FALSE, 0);
	 gtk_box_pack_start (GTK_BOX (hbox), vbox2, FALSE, TRUE, 5);
      	 label = gtk_label_new ("Hour:");
	 gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	 gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, TRUE, 0);

	 {
	    GList	*cbitems = NULL;
	    GtkWidget	*cb;
    
	    cbitems = g_list_append (cbitems, "0");
	    cbitems = g_list_append (cbitems, "4");
	    cbitems = g_list_append (cbitems, "8");
	    cbitems = g_list_append (cbitems, "12");
	    cbitems = g_list_append (cbitems, "16");
	    cbitems = g_list_append (cbitems, "20");
	    cbitems = g_list_append (cbitems, "*");
	    
	    combo [0] = cb = gtk_combo_new ();
	    gtk_tooltips_set_tip (tooltips, GTK_COMBO(cb)->entry,
				  "* matches every hour", NULL);  
	    gtk_combo_set_popdown_strings (GTK_COMBO (cb), cbitems);
	    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(cb)->entry), cron_hour);
	    gtk_widget_set_usize (GTK_COMBO (cb)->entry, 80, 0);
	    gtk_widget_set_usize (cb, 80, 0);
	    gtk_box_pack_start (GTK_BOX (vbox2), cb, FALSE, TRUE, 0);
	    gtk_signal_connect (GTK_OBJECT (GTK_COMBO(cb)->entry), "changed",
				GTK_SIGNAL_FUNC (set_combo), &cron_hour);
	    g_list_free (cbitems);
	 }

	 vbox2 = gtk_vbox_new (FALSE, 0);
	 gtk_box_pack_start (GTK_BOX (hbox), vbox2, FALSE, TRUE, 5);
      	 label = gtk_label_new ("Minute:");
	 gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	 gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, TRUE, 0);

	 {
	    GList	*cbitems = NULL;
	    GtkWidget	*cb;
    
	    cbitems = g_list_append (cbitems, "0");
	    cbitems = g_list_append (cbitems, "10");
	    cbitems = g_list_append (cbitems, "20");
	    cbitems = g_list_append (cbitems, "30");
	    cbitems = g_list_append (cbitems, "40");
	    cbitems = g_list_append (cbitems, "50");
	    
	    combo [1] = cb = gtk_combo_new ();
	    gtk_combo_set_popdown_strings (GTK_COMBO (cb), cbitems);
	    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(cb)->entry), cron_minute);
	    gtk_widget_set_usize (GTK_COMBO (cb)->entry, 80, 0);
	    gtk_widget_set_usize (cb, 80, 0);
	    gtk_box_pack_start (GTK_BOX (vbox2), cb, FALSE, TRUE, 0);
	    gtk_signal_connect (GTK_OBJECT (GTK_COMBO(cb)->entry), "changed",
				GTK_SIGNAL_FUNC (set_combo), &cron_minute);
	    g_list_free (cbitems);
	 }
	 {
	    GtkWidget *button;

	    button = gtk_button_new_with_label ("Default");
	    gtk_box_pack_start (GTK_BOX (vbox2), button, FALSE, TRUE, 10);
	    gtk_signal_connect (GTK_OBJECT (button), "clicked",
				GTK_SIGNAL_FUNC (set_default_time), NULL);
	    gtk_tooltips_set_tip (tooltips, button,
				  "Set a default time for cronjob based on "
				  "the selected tile.", NULL);  
	 }
	 
	 
	 vbox2 = gtk_vbox_new (FALSE, 0);
	 gtk_box_pack_start (GTK_BOX (hbox), vbox2, FALSE, TRUE, 5);
      	 label = gtk_label_new ("Day of week:");
	 gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
	 gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, TRUE, 0);

	 {
	    GList *cbitems = NULL;
	    GtkWidget *cb;
	    
	    cbitems = g_list_append (cbitems, "*");
	    cbitems = g_list_append (cbitems, "Sun");
	    cbitems = g_list_append (cbitems, "Mon");
	    cbitems = g_list_append (cbitems, "Tue");
	    cbitems = g_list_append (cbitems, "Wed");
	    cbitems = g_list_append (cbitems, "Thu");
	    cbitems = g_list_append (cbitems, "Fri");
	    cbitems = g_list_append (cbitems, "Sat");
	    
	    combo [2] = cb = gtk_combo_new ();
	    gtk_tooltips_set_tip (tooltips, GTK_COMBO(cb)->entry,
				  "* matches every day", NULL);  
	    gtk_combo_set_popdown_strings (GTK_COMBO (cb), cbitems);
	    gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(cb)->entry), cron_day);
	    gtk_widget_set_usize (GTK_COMBO (cb)->entry, 80, 0);
	    gtk_widget_set_usize (cb, 80, 0);
	    gtk_box_pack_start (GTK_BOX (vbox2), cb, FALSE, TRUE, 0);
	    gtk_signal_connect (GTK_OBJECT (GTK_COMBO(cb)->entry), "changed",
				GTK_SIGNAL_FUNC (set_combo), &cron_day);
	    g_list_free (cbitems);
	 }
      }
#endif /*  defined(HAVE_POPEN) && defined(CRONTAB) */
      
      box = gtk_hbutton_box_new ();
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->action_area),
			  box, FALSE, TRUE, 5);

      button = gtk_button_new_with_label ("Use selected tile");
      gtk_box_pack_start (GTK_BOX (box), button, FALSE, TRUE, 5);
      gtk_object_set_user_data (GTK_OBJECT (button), button);
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (set_tileofday), (gpointer) wtexture);
      gtk_tooltips_set_tip (tooltips, button,
			    "Set IconBack pixmap to selected tile.", NULL);  
      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (window));

      button = gtk_button_new_with_label ("Get selected tile");
      gtk_box_pack_start (GTK_BOX (box), button, FALSE, TRUE, 5);
      gtk_object_set_user_data (GTK_OBJECT (button), button);
      gtk_tooltips_set_tip (tooltips, button,
			    "Get current tile via HTTP request - "
			    "this may take some time ...", NULL);  
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (get_tileofday), (gpointer) wtexture);

      button = gtk_button_new_with_label ("Close");
      gtk_box_pack_start (GTK_BOX (box), button, FALSE, TRUE, 5);
      gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
				 GTK_SIGNAL_FUNC (gtk_widget_destroy),
				 GTK_OBJECT (window));
   }
   
   if (!GTK_WIDGET_VISIBLE (window))
      gtk_widget_show_all (window);
   else
      gtk_widget_destroy (window);
}

void
check_valid_tileofday (wtexture_t *wtexture)
/*
 *  IconBack must be 'tile-of-the-day' if cron job is activated.
 *
 *  No return value.
 */
{
   if (strcaseeq (PLGetString (wtexture->key), "IconBack"))
   {
      char *txt;

      txt = gtk_entry_get_text (GTK_ENTRY (wtexture->wpixmap [PIXMAP_NAME]));
      if (cron_set
	  && !streq (txt, "tile-of-the-day")
	  && !streq (txt, "random-tile")
	  && !streq (txt, "guest-tile"))
      {
	 dialog_popup (DIALOG_WARNING,
		       "Tile cronjob has been removed because\n"
		       "IconBack pixmap is neither\n"
		       "`tile-of-the-day', `guest-tile',\n"
		       "nor `random-tile'.", NULL, NULL);
	 cron_set = NO;
	 cron_changed  = YES;
	 PLInsertDictionaryEntry (plist_changed, wtexture->key, pl_yes);
	 if (toggle_button)
	    gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (toggle_button),
					 cron_set);
      }
   }
}

#if defined(HAVE_POPEN) && defined(CRONTAB)
void
read_crontab (void)
/*
 *  Read cron table and set local variables.
 *
 *  No return value.
 *
 *  Side effects:
 *	locals 'cron_day', 'crontab_file', 'cron_set', 'cron_minute',
 *	'cron_hour' are set.
 */
{
   FILE *cronpipe;

   cron_day     = strdup ("*");
   cron_minute	= strdup ("0");
   cron_hour	= strdup ("4");
   
   cronpipe = popen (CRONTAB " -l", "r");
   if (!cronpipe)
   {
      dialog_popup (DIALOG_ERROR, "Can't connect to crontab utility.",
		    NULL, NULL);
      return;
   }
   else
   {
      unsigned	lines = 0;		/* number of lines in crontab */
      char	tmp [MAXSTRLEN];

      while (fgets (tmp, MAXSTRLEN, cronpipe))
	 if (*tmp != '#' && !strstr (tmp, "gettile.pl"))
	    lines++;
      pclose (cronpipe);

      cron_table = Calloc (lines + 1, sizeof (char *));
      /*
       *  Read and store lines of crontab
       */
      cronpipe = popen (CRONTAB " -l", "r");
      if (!cronpipe)
      {
	 dialog_popup (DIALOG_ERROR, "Can't connect to crontab utility.",
		       NULL, NULL);
	 return;
      }
      else
      {
	 lines = 0;
	 while (fgets (tmp, MAXSTRLEN, cronpipe))
	 {
	    if (*tmp == '#')
	       continue;
	    if (strstr (tmp, "gettile.pl")) /* read cronjob date and time */
	    {
	       char dummy1 [MAXSTRLEN];
	       char dummy2 [MAXSTRLEN];
	       char dummy3 [MAXSTRLEN];
	       char dummy4 [MAXSTRLEN];
	       char dummy5 [MAXSTRLEN];

	       Free (cron_minute);
	       Free (cron_hour);
	       Free (cron_day);

	       sscanf (tmp, "%s %s %s %s %s",
		       dummy1, dummy2, dummy3, dummy4, dummy5);
	       cron_minute = strdup (dummy1);
	       cron_hour   = strdup (dummy2);
	       cron_day    = strdup (dummy5);
	       cron_set    = YES;
	       if (strstr (tmp, "tile-of-the-day"))
		  tile_string = "tile-of-the-day";
	       else if (strstr (tmp, "random-tile"))
		  tile_string = "random-tile";
	       else
		  tile_string = "guest-tile";
	    }
	    else
	       cron_table [lines++] = strdup (tmp);
	 }
	 cron_table [lines] = NULL;
	 pclose (cronpipe);
      }
   }
}

void
save_crontab (void)
/*
 *  Save new cron table if required, i.e. cron_changed == TRUE.
 *
 *  Return value:
 *	TRUE if file has been saved
 */
{
   if (!cron_changed)
      return;
   else
   {
      char *path = get_pixmap_path (tile_string);
   
      if (path)
      {
	 if (cron_set)
	 {
	    char *tmp_name 	= tmpnam (NULL);
	    char *cron_filename = tmp_name ? tmp_name : "/tmp/cron.tmp";
	    FILE *cronfile      = fopen (cron_filename, "w");
	 
	    if (!cronfile)
	    {
	       dialog_popup (DIALOG_ERROR,
			     "Can't write to temporary crontab file.",
			     NULL, NULL);
	       return;
	    }
	    if (cron_table)		/* non-gettile entries */
	    {
	       char **line;
	       
	       for (line = cron_table; *line; line++)
		  fputs (*line, cronfile);
	    }
	    fprintf (cronfile, "%s %s * * %s %s %s %s\n",
		     cron_minute, cron_hour, cron_day,
		     PERL " " PKGDATADIR "/gettile.pl", wmpath, path);
	    fclose (cronfile);

	    {
	       char *tmp = Calloc (strlen (CRONTAB " ") + strlen (cron_filename)
				   + 1, sizeof (char));
	       strcpy (tmp, CRONTAB " ");
	       strcat (tmp, cron_filename);
	       if (system (tmp))
		  dialog_popup (DIALOG_ERROR, "Can't replace crontab file.",
				NULL, NULL);
	       else
		  message ("Crontab file changed.");
	       Free (tmp);
	    }
	 }
	 Free (path);
      }
      else
	 dialog_popup (DIALOG_ERROR,
		       "Can't find selected tile in PixmapPath.", NULL, NULL);
   }
   cron_changed = NO;
}

#endif /*  defined(HAVE_POPEN) && defined(CRONTAB) */

/*******************************************************************************

				private code
  
*******************************************************************************/

static void
set_tileofday (GtkWidget *widget, gpointer ptr)
/*
 *  Set pixmap of IconBack to 'tile-of-the-day'.
 *  Update corresponding widgets accordingly.
 *
 *  No return value.
 */
{
   char	*path = get_pixmap_path (tile_string);

   if (path)
   { 
      wtexture_t *wtexture = (wtexture_t *) ptr;
      proplist_t *array = PLGetDictionaryEntry (windowmaker, wtexture->key);
   
      Free (path);
      PLRemoveArrayElement (array, 1);	/* filename */
      PLInsertArrayElement (array, PLMakeString (tile_string), 1);

      make_pixmap (tile_string, 128, 128, wtexture->wpixmap [PIXMAP_PIXMAP]);
      gtk_entry_set_text (GTK_ENTRY (wtexture->wpixmap [PIXMAP_NAME]),
			  tile_string);
      PLInsertDictionaryEntry (plist_changed, wtexture->key, pl_yes);
      changed = YES;
   }
   else
      dialog_popup (DIALOG_ERROR, "Can't find selected tile in PixmapPath.",
		    NULL, NULL);
}

static void
get_tileofday (GtkWidget *widget, gpointer ptr)
/*
 *  Call perl script gettile.pl that tries to get themes.org current
 *  tile of the day.
 *
 *  No return value.
 */
{
   char	*path = get_pixmap_path (tile_string);

   if (path)
   {
      char *argv = Calloc (strlen (PERL " " PKGDATADIR "/gettile.pl ")
			   + strlen (path) + 1 + 3, sizeof (char));
      
      strcpy (argv, PERL " " PKGDATADIR "/gettile.pl ");
      strcat (argv, " no ");
      strcat (argv, path);

      if (system (argv))
	 dialog_popup (DIALOG_ERROR,
		       "Can't connect to http://themes.org/.\n"
		       "Please check `stderr' for more details.",
		       NULL, NULL);
      else
      {
	 if (streq (tile_string, "tile-of-the-day"))
	    make_pixmap (tile_string, 64, 64, pixmap_tileofday [0]);
	 else if (streq (tile_string, "random-tile"))
	    make_pixmap (tile_string, 64, 64, pixmap_tileofday [1]);
	 else
	    make_pixmap (tile_string, 64, 64, pixmap_tileofday [2]);
	 if (cron_set)
	    set_tileofday (widget, ptr);
      }
      Free (argv);
      Free (path);
   }
   else
      dialog_popup (DIALOG_ERROR, "Can't find selected tile in PixmapPath.",
		    NULL, NULL);
}

static void
set_combo (GtkWidget *widget, gpointer ptr)
{
   char **value = (char **) ptr;
   
   Free (*value);
   *value       = strdup (gtk_entry_get_text (GTK_ENTRY (widget)));
   cron_changed = YES;
}

static void
set_toggle (GtkWidget *widget, gpointer ptr)
{
   cron_set = GTK_TOGGLE_BUTTON (widget)->active;
   cron_changed = YES;

   if (cron_set)
      set_tileofday (widget, ptr);
}

static void
set_default_time (GtkWidget *widget, gpointer ptr)
{
   if (streq (tile_string, "tile-of-the-day"))
   {
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [0])->entry), "6");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [1])->entry), "0");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [2])->entry), "*");
   }
   else if (streq (tile_string, "random-tile"))
   {
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [0])->entry), "*");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [1])->entry), "1");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [2])->entry), "*");
   }
   else
   {
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [0])->entry), "6");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [1])->entry), "0");
      gtk_entry_set_text (GTK_ENTRY (GTK_COMBO(combo [2])->entry), "Tue");
   }
}

static void
tile_toggle (GtkWidget *widget, gpointer ptr)
{
   if (GTK_TOGGLE_BUTTON (widget)->active)
   {
      tile_string = (char *) ptr;
      if (cron_set)
      {
	 wtexture_t *wtexture = gtk_object_get_user_data (GTK_OBJECT (widget));
	 
	 set_tileofday (widget, wtexture);
	 cron_changed = YES;
      }
   }
}


#endif /* TILE_OF_DAY */
