#include	"qpage.h"


/*
** global variables
*/
#ifndef lint
static char	sccsid[] = "@(#)config.c  1.16  06/26/97  tomiii@mtu.edu";
#endif
service_t	*Services;
pager_t		*Pagers;
pgroup_t	*Groups;
char		*Administrator;
char		*ForceHostname;
int		IdentTimeout;
int		SNPPTimeout;


/*
** check_service()
**
** This function sets appropriate defaults for any unspecified parameters
** of a paging service.  If the default_service parameter is not NULL, the
** unspecified parameters are copied from that service.  If the service
** and default_service parameters point to the same structure then the
** compiled-in defaults are used.
**
**	Input:
**		service - the service to set the defaults for
**		default_service - a pointer to the default service
**
**	Returns:
**		an integer status code (0=success, -1=error)
*/
int
check_service(service_t *service, service_t *default_service)
{
	if ((default_service == NULL) || (service == default_service)) {
		if (service->device == NULL) {
			qpage_log(LOG_ERR, "no modem device for service=%s",
				service->name);

			service->device = "FATAL_CONFIGURATION_ERROR";
			return(-1);
		}

		if (service->dialcmd == NULL) {
			qpage_log(LOG_ERR, "no dial command for service=%s",
				service->name);

			service->dialcmd = "FATAL_CONFIGURATION_ERROR";
			return(-1);
		}

		if (service->password == NULL)
			service->password = strdup(DEFAULT_PASSWORD);

		if (service->baudrate == 0)
			service->baudrate = DEFAULT_BAUDRATE;

		if (service->maxmsgsize == 0)
			service->maxmsgsize = DEFAULT_MAXMSGSIZE;

		if (service->maxpages == 0)
			service->maxpages = DEFAULT_MAXPAGES;

		if (service->maxtries == 0)
			service->maxtries = DEFAULT_MAXTRIES;
	}
	else {
		if (service->device == NULL)
			service->device = strdup(default_service->device);

		if (service->dialcmd == NULL)
			service->dialcmd = strdup(default_service->dialcmd);

		if (service->password == NULL)
			service->password = strdup(default_service->password);

		if (service->baudrate == 0)
			service->baudrate = default_service->baudrate;

		if (service->maxmsgsize == 0)
			service->maxmsgsize = default_service->maxmsgsize;

		if (service->maxpages == 0)
			service->maxpages = default_service->maxpages;

		if (service->maxtries == 0)
			service->maxtries = default_service->maxtries;
	}

	return(0);
}


/*
** check_pager()
**
** This function verifies that a pager specification has all the
** appropriate information.  If a required field is missing, the
** entry for this pager is freed.  Note that we are aways passed
** the head of a linked list so we don't need to worry about changing
** a pointer pointing at this pager.
**
**	Input:
**		pager - a pointer to the pager structure
**		default_service - the default service to use
**
**	Returns:
**		an integer status code (0=success, -1=error)
*/
int
check_pager(pager_t **pager, service_t *default_service)
{
	pager_t	*tmp;


	tmp = *pager;

	if (tmp->pagerid == NULL) {
		qpage_log(LOG_ERR, "no pagerid for pager=%s", tmp->name);
		*pager = tmp->next;
		free(tmp->name);
		free(tmp);
		return(-1);
	}

	if (tmp->service == NULL) {
		if (default_service == NULL) {
			qpage_log(LOG_ERR, "no service for pager=%s", tmp->name);
			*pager = tmp->next;
			free(tmp->name);
			free(tmp);
			return(-1);
		}

		tmp->service = default_service;
	}

	return(0);
}



/*
** open_config_file()
**
** This function opens the configuration file.
**
**	Input:
**		nothing
**
**	Returns:
**		a FILE pointer to the open configuration file
*/
FILE *
open_config_file(char *filename)
{
	FILE	*fp;
#ifdef MAYBE_NEXT_RELEASE
	char	buff[1024];
	char	*homedir;


	/*
	** First check the user's home directory
	*/
	if ((homedir = getenv("HOME")) != NULL) {
		(void)sprintf(buff, "%s/%s", homedir, USER_CONFIG);

		if ((fp = fopen(buff, "r")) != NULL)
			return(fp);
	}
#endif

	/*
	** Use the system default configuration file
	*/
	fp = fopen(filename, "r");

	return(fp);
}


/*
** read_config_file()
**
** This function reads the configuration file.
**
**	Input:
**		nothing
**
**	Returns:
**		an integer status code
**
**	Side effects:
**		This function changes the current working directory to
**		that of the page queue as specified in the configuration
**		file.  It is an error if no such specification is present.
**
**		The following global variables are modified:
**
**			Pagers
**			Services
**			Groups
**			IdentTimeout
**			SNPPTimeout
**			Administrator
**			ForceHostname
*/
int
read_config_file()
{
	FILE		*fp;
	service_t	*default_service;
	service_t	*service;
	pager_t		*pager;
	pgroup_t	*group;
	member_t	*tmp;
	char		buff[1024];
	char		value[1024];
	char		*ptr;
	int		gotqueuedir;
	int		haserrors;
	int		line;
	int		pos;


	IdentTimeout = DEFAULT_IDENTTIMEOUT;
	SNPPTimeout = DEFAULT_SNPPTIMEOUT;
	Administrator = NULL;
	ForceHostname = NULL;
	Services = NULL;
	Pagers = NULL;
	Groups = NULL;

	default_service = NULL;
	service = NULL;
	gotqueuedir = 0;
	haserrors = 0;
	pager = NULL;
	group = NULL;
	line = 0;
	pos = 0;

	if ((fp = open_config_file(ConfigFile)) == NULL) {
		qpage_log(LOG_ERR, "cannot open configuration file");
		return(-1);
	}

	for (;;) {
		if (pos && buff[pos]) {
#ifdef CONFDEBUG
			printf("Leftover characters: <%s>\n", &buff[pos]);
#endif
			(void)strcpy(buff, &buff[pos]);
		}
		else {
			if (fgets(buff, sizeof(buff), fp) == NULL) {
#ifdef COMMENTS_ANYWHERE
				if ((ptr = strchr(buff, '#')) != NULL)
					*ptr = '\0';
#endif

				break;
			}
			else
				line++;
		}

		/*
		** Comments and blank lines cause the termination of any
		** current service, pager, or group specification.
		*/
		if ((buff[0] == '#') || (buff[0] == '\n')) {
			pos = 0;
			continue;
		}

		/*
		** strip trailing newlines
		*/
		if ((ptr = strchr(buff, '\n')) != NULL)
			*ptr = '\0';

		/*
		** This is the location of the queue directory.  This
		** keyword causes the termination of any current service,
		** pager, or group specification.
		*/
		if (sscanf(buff, "queuedir=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
			printf("queuedir=<%s>\n", value);
#endif
			if (gotqueuedir) {
				qpage_log(LOG_WARNING, "duplicate queuedir specification");
				continue;
			}

			if (chdir(value) < 0) {
				qpage_log(LOG_ERR, "chdir(%s) failed: %s",
					value, strerror(errno));
				haserrors++;
			}
			else
				gotqueuedir++;

			if (service) {
				if (check_service(service, default_service))
					haserrors++;

				service = NULL;
			}

			if (pager) {
				if (check_pager(&Pagers, default_service))
					haserrors++;

				pager = NULL;
			}

			if (group)
				group = NULL;

			continue;
		}

		/*
		** This is the ident timeout in seconds.  This keyword
		** causes the termination of any current service, pager,
		** or group specification.
		*/
		if (sscanf(buff, "identtimeout=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
			printf("identtimeout=<%s>\n", value);
#endif
			IdentTimeout = atoi(value);

			if (IdentTimeout < 0)
				IdentTimeout = 0;

			if (service) {
				if (check_service(service, default_service))
					haserrors++;

				service = NULL;
			}

			if (pager) {
				if (check_pager(&Pagers, default_service))
					haserrors++;

				pager = NULL;
			}

			if (group)
				group = NULL;

			continue;
		}

		/*
		** This is the SNPP command timeout in seconds.  This keyword
		** causes the termination of any current service, pager, or
		** group specification.
		*/
		if (sscanf(buff, "snpptimeout=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
			printf("snpptimeout=<%s>\n", value);
#endif
			SNPPTimeout = atoi(value);

			if (SNPPTimeout < 2)
				SNPPTimeout = 2;

			if (service) {
				if (check_service(service, default_service))
					haserrors++;

				service = NULL;
			}

			if (pager) {
				if (check_pager(&Pagers, default_service))
					haserrors++;

				pager = NULL;
			}

			if (group)
				group = NULL;

			continue;
		}

		/*
		** This specifies the e-mail addresses of the qpage
		** administrator.  If defined, status notification is
		** sent to this address (regardless of the service level)
		** for all failed pages.  This keyword causes the termination
		** of any current service, pager, or group specification.
		*/
		if (sscanf(buff, "administrator=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
			printf("administrator=<%s>\n", value);
#endif
			Administrator = strdup(value);

			if (service) {
				if (check_service(service, default_service))
					haserrors++;

				service = NULL;
			}

			if (pager) {
				if (check_pager(&Pagers, default_service))
					haserrors++;

				pager = NULL;
			}

			if (group)
				group = NULL;

			continue;
		}

		/*
		** This specifies whether qpage should qualify the
		** addresses used to send e-mail notification to
		** page submitters.  If true (and the CALLerid info
		** does not contain "@host.name") then the submitter's
		** hostname will be appended to the notification e-mail
		** address.  If the vaue of this keyword starts with
		** '@' then it is appended as-is to unqualified addresses.
		** This keyword causes the termination of any current
		** service, pager, or group specification.
		**
		** Disclaimer: Using "true" for this keyword should
		** be a last resort.  I strongly believe that all e-mail
		** addresses for a given domain should use the domain
		** name ONLY (with no leading hostname).  Administrators
		** who implement public user@host e-mail addresses (where
		** "host" is the name of the user's workstation) should
		** be forced to scrub toilets for a living.
		*/
		if (sscanf(buff, "forcehostname=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
			printf("forcehostname=<%s>\n", value);
#endif
			if (service) {
				if (check_service(service, default_service))
					haserrors++;

				service = NULL;
			}

			if (pager) {
				if (check_pager(&Pagers, default_service))
					haserrors++;

				pager = NULL;
			}

			if (group)
				group = NULL;

			if (value[0] == '@') {
				ForceHostname = strdup(value);
				continue;
			}

			if (!strcasecmp(value, "on"))
				ForceHostname = "true";

			if (!strcasecmp(value, "yes"))
				ForceHostname = "true";

			if (!strcasecmp(value, "true"))
				ForceHostname = "true";

			continue;
		}

		/*
		** This is a paging service specification.  Add another
		** node to the services linked-list.  Note that we are
		** building the linked-list in reverse order deliberately;
		** we want later specifications to take priority over
		** earlier specifications.  This keyword causes the
		** termination of any current pager or group specification.
		*/
		if (sscanf(buff, "service=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
			printf("service=<%s>\n", value);
#endif

			if (service)
				if (check_service(service, default_service))
					haserrors++;

			service = (void *)malloc(sizeof(*service));
			(void)memset((char *)service, 0, sizeof(*service));
			service->next = Services;
			Services = service;
			service->name = strdup(value);
			service->identfrom = TRUE;
			service->parity = -1;

			if (!strcmp(value, "default"))
				default_service = service;

			if (pager) {
				if (check_pager(&Pagers, default_service))
					haserrors++;

				pager = NULL;
			}

			if (group)
				group = NULL;

			continue;
		}

		/*
		** This is a pager specification.  Add another node to the
		** pagers linked-list.  Note that we are building the list
		** in reverse order deliberately; we want later specifications
		** to take priority over earlier specifications.  This keyword
		** causes the termination of any current service or group
		** specification.
		*/
		if (sscanf(buff, "pager=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
			printf("pager=<%s>\n", value);
#endif

			if (pager)
				if (check_pager(&Pagers, default_service))
					haserrors++;

			pager = (void *)malloc(sizeof(*pager));
			(void)memset((char *)pager, 0, sizeof(*pager));
			pager->next = Pagers;
			Pagers = pager;
			pager->name = strdup(value);

			if (service) {
				if (check_service(service, default_service))
					haserrors++;

				service = NULL;
			}

			if (group)
				group = NULL;

			continue;
		}

		/*
		** This is a group specification.  Add another node to the
		** groups linked-list.  Note that we are building the list
		** in reverse order deliberately; we want later specifications
		** to take priority over earlier specifications.  This keyword
		** causes the termination of any current service or pager
		** specification.
		*/
		if (sscanf(buff, "group=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
			printf("group=<%s>\n", value);
#endif

			group = (void *)malloc(sizeof(*group));
			(void)memset((char *)group, 0, sizeof(*group));
			group->next = Groups;
			Groups = group;
			group->name = strdup(value);

			if (service) {
				if (check_service(service, default_service))
					haserrors++;

				service = NULL;
			}

			if (pager) {
				if (check_pager(&Pagers, default_service))
					haserrors++;

				pager = NULL;
			}

			continue;
		}

		if (service && isspace(buff[0])) {
			if (sscanf(buff, " device=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:device=<%s>\n", service->name,
					value);
#endif
				service->device = strdup(value);
				continue;
			}

			if (sscanf(buff, " dialcmd=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:dialcmd=<%s>\n", service->name,
					value);
#endif
				service->dialcmd = strdup(value);
				continue;
			}

			if (sscanf(buff, " password=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:password=<%s>\n", service->name,
					value);
#endif
				service->password = strdup(value);
				continue;
			}

			if (sscanf(buff, " baudrate=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:baudrate=<%s>\n", service->name,
					value);
#endif
				switch (atoi(value)) {
					case 300:
						service->baudrate = B300;
						break;

					case 1200:
						service->baudrate = B1200;
						break;

					case 2400:
						service->baudrate = B2400;
						break;

					case 9600:
						service->baudrate = B9600;
						break;

					case 19200:
						service->baudrate = B19200;
						break;

					case 38400:
						service->baudrate = B38400;
						break;

					default:
						qpage_log(LOG_WARNING, "unrecognized baud rate %s", value);
						service->baudrate = B300;
						break;
				}
				continue;
			}

			if (sscanf(buff, " parity=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:parity=<%s>\n", service->name,
					value);
#endif
				if (!strcasecmp(value, "even"))
					service->parity = 2;
				else
					if (!strcasecmp(value, "odd"))
						service->parity = 1;
					else
						if (!strcasecmp(value, "none"))
							service->parity = 0;
						else
							qpage_log(LOG_WARNING,
								"unrecognized parity %s", value);

				continue;
			}

			if (sscanf(buff, " maxmsgsize=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:maxmsgsize=<%s>\n", service->name,
					value);
#endif
				service->maxmsgsize = atoi(value);
				continue;
			}

			if (sscanf(buff, " maxpages=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:maxpages=<%s>\n", service->name,
					value);
#endif
				service->maxpages = atoi(value);

				if (service->maxpages < 1)
					service->maxpages = 1;

				if (service->maxpages > 9)
					service->maxpages = 9;

				continue;
			}

			if (sscanf(buff, " maxtries=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:maxtries=<%s>\n", service->name,
					value);
#endif
				service->maxtries = atoi(value);
				continue;
			}

			if (sscanf(buff, " identfrom=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:identfrom=<%s>\n", service->name,
					value);
#endif
				if (!strcasecmp(value, "no"))
					service->identfrom = FALSE;

				if (!strcasecmp(value, "off"))
					service->identfrom = FALSE;

				if (!strcasecmp(value, "false"))
					service->identfrom = FALSE;

				continue;
			}

			if (sscanf(buff, " allowpid=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:allowpid=<%s>\n", service->name,
					value);
#endif
				if (!strcasecmp(value, "on"))
					service->allowpid = TRUE;

				if (!strcasecmp(value, "yes"))
					service->allowpid = TRUE;

				if (!strcasecmp(value, "true"))
					service->allowpid = TRUE;

				continue;
			}
		}

		if (pager && isspace(buff[0])) {
			if (sscanf(buff, " pagerid=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:pagerid=<%s>\n", pager->name,
					value);
#endif
				pager->pagerid = strdup(value);
				continue;
			}

			if (sscanf(buff, " service=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:service=<%s>\n", pager->name,
					value);
#endif
				pager->service = lookup(Services, value);

				if (pager->service == NULL) {
					haserrors++;

					qpage_log(LOG_ERR,
						"no service %s for pager=%s",
						value, pager->name);

					Pagers = pager->next;

					if (pager->pagerid)
						free(pager->pagerid);

					free(pager->name);
					free(pager);
					pager = NULL;
				}

				continue;
			}
		}

		if (group && isspace(buff[0])) {
			if (sscanf(buff, " member=%s%n", value, &pos) == 1) {
#ifdef CONFDEBUG
				printf("%s:member=<%s>\n", group->name,
					value);
#endif
				/*
				** check for a duty schedule
				*/
				if ((ptr = strchr(value, '/')) != NULL) {
					*ptr++ = '\0';

					if (on_duty(ptr, 0) < 0)
						continue;
				}

				if ((pager = lookup(Pagers, value)) != NULL) {
					/*
					** The pager is valid.  Add to group.
					*/
					tmp = (void *)malloc(sizeof(*tmp));
					(void)memset((char *)tmp, 0, sizeof(*tmp));
					tmp->pager = pager;

					if (ptr)
						tmp->schedule = strdup(ptr);

					tmp->next = group->members;
					group->members = tmp;

					pager = NULL;
				}
				else
					qpage_log(LOG_ERR,
						"no such pager %s for group=%s",
						value, group->name);

				continue;
			}
		}

		/*
		** check for a line consisting of nothing but whitespace
		*/
		for (ptr=buff; *ptr; ptr++) {
			if (!isspace(*ptr))
				break;
		}

		if (*ptr == (char)NULL) {
#ifdef CONFDEBUG
			printf("<just whitespace>\n");
#endif
			pos = 0;
			continue;
		}

		qpage_log(LOG_WARNING, "syntax error in config file: line %d",
			line);

		haserrors++;
		pos = 0;
	}

	(void)fclose(fp);

	/*
	** make sure the last service we read is complete
	*/
	if (service)
		if (check_service(service, default_service))
			haserrors++;

	/*
	** make sure the last pager we read is complete
	*/
	if (pager)
		if (check_pager(&Pagers, default_service))
			haserrors++;


	if (!gotqueuedir) {
		qpage_log(LOG_ERR, "no queuedir specification!");
		return(-1);
	}

	return(haserrors);
}


#if 0
void
print_services(service_t *list)
{
	if (list->next)
		print_services(list->next);

	printf("Service name: <%s>\n", list->name);
	printf("\tdevice=<%s>\n", list->device);
	printf("\tdialcmd=<%s>\n", list->dialcmd);
	printf("\tpassword=<%s>\n", list->password);
	printf("\tbaudrate=<%d>\n", list->baudrate);
	printf("\tmaxmsgsize=<%d>\n", list->maxmsgsize);
	printf("\tmaxpages=<%d>\n", list->maxpages);
	printf("\tmaxtries=<%d>\n", list->maxtries);
	printf("\n");
}


void
print_pagers(pager_t *list)
{
	if (list->next)
		print_pagers(list->next);

	printf("Pager name: <%s>\n", list->name);
	printf("\tpagerid=<%s>\n", list->pagerid);
	printf("\tservice=<%s>\n", list->service->name);
	printf("\n");
}


int
main(void)
{
	read_config_file();

	if (Services)
		print_services(Services);

	if (Pagers)
		print_pagers(Pagers);

	return(0);
}
#endif
