/*
 * Created on May 16, 2004
 * Created by Olivier Chalouhi
 * Copyright (C) 2004 Aelitis, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 * AELITIS, SARL au capital de 30,000 euros
 * 8 Allee Lenotre, La Grille Royale, 78600 Le Mesnil le Roi, France.
 *
 */

package org.gudy.azureus2.update;

import java.io.*;
import java.util.*;
import java.net.*;

public class 
Updater 
{
	private static final String MAIN_CLASS		= "org.gudy.azureus2.ui.swt.Main";
	private static final String UPDATER_JAR 	= "Updater.jar";

	public static final String		LOG_FILE_NAME		= "update.log";
	public static final String		UPDATE_PROPERTIES	= "update.properties";
	
  		// change these and you'll need to change the UpdateInstaller !!!!
  
	protected static final String	UPDATE_DIR 	= "updates";
	protected static final String	ACTIONS		= "install.act";
	
	public static void 
	main(
		String[] args) 
	{   
		new Updater().update( args );
	}
	
	protected void
	update(
		String[]	args )
	{
		if (args.length < 3) {
			System.out.println("Usage: Updater full_classpath full_librarypath full_userpath [config override]");
			System.exit(-1);
		}
    
		if ( args[0].equals( "restart" )){
	    	
			restart( args, true );
				
		}else if ( args[0].equals( "updateonly" )){
			    	
			restart( args, false );
						
		}else{
    	
			System.out.println( "Old restart mechanism no longer supported");
		}
	}
  
  	protected void
  	restart(
		String[]	args,
		boolean		restart_azureus )
	{
  		String command	    	= args[0];
  		String app_path			= args[1];
  		String user_path   		= args[2];
  		String config_override	= args.length>3?args[3]:"";
  		
  		boolean	do_restart	= restart_azureus;
  		
  		
  		// 2.1.0.0 shagged restart example:
  		// "C:\Program Files\Java\jdk1.5.0\jre\bin\javaw" 
  		// -classpath "blah" 
  		// -Djava.library.path="C:\Program Files\Azureus" org.gudy.azureus2.update.Updater 
  		// "restart" "C:\Program Files\Azureus\" "C:\Documents and Settings\stuff\Application Data\Azureus" ""
  		// 
  		// trailing \ escapes quote -> app_path has a \ " in it
  		// note, only win32 as the user.dir trailing \ is added by the new azureus.exe launcher
  
  		String	log_dir;
  		boolean	shagged_restart = false;
  		
  		int	shag_pos = app_path.indexOf("\" ");
  		
  		if (  shag_pos != -1 ){
  			
  				// shagged
  			
  			log_dir = app_path.substring(0,shag_pos);
  		
  			shagged_restart = true;
  			
  		}else{
  			
  			log_dir	= user_path;
  		}
  		  		
	    File logFile = new File( log_dir, LOG_FILE_NAME );
  		    
	    if ( !logFile.exists()){
	    	
	    	try{
	    		logFile.createNewFile();
	    		
	    	}catch( IOException e ){
	    		
	    		e.printStackTrace();
	    	}
	    	
	    	if ( !logFile.exists()){
	    		
	    		try{
	    		
	    			logFile = File.createTempFile( "AZU", ".log" );
	    			
	    		}catch( IOException e ){
	    			
	    			e.printStackTrace();
	    			
	    				// last ditch attempt, use file relative to current location
	    			
	    			logFile = new File("update.log");
	    		}
	    	}
	    }
	    
	    PrintWriter	log = null;
	    
	    Properties	update_properties	= new Properties();
	    
  		try{
  		    log = new PrintWriter(new FileWriter(logFile, true));

  		    Calendar now = GregorianCalendar.getInstance();

  		    log.println( "Update Starts: " + now.get(Calendar.HOUR_OF_DAY) + ":" + format(now.get(Calendar.MINUTE)) + ":" + format(now.get(Calendar.SECOND)));        

  		    log.println( "app  dir = " + app_path );
  		    log.println( "user dir = " + user_path );
  		    log.println( "config override = " + config_override );
  			
  	  		if ( shagged_restart ){
  	  			
  	  			log.println( "invalid restart parameters, abandoning restart" );
  	  			
  	  			log.println( "**** PLEASE MANUALLY INSTALL AZUREUS FROM THE WIN32 .EXE SETUP PACKAGE ****");
  
  	  			do_restart = false;
  	  			
  	  			log.close();
  	  			
  	  			System.exit(1);
  	  			
  	  			return;
  	  			
  	  		}else{
  	  			
  				if ( isWindows ){
  	  				
  		  				// 2.2.0.0 shagged restart. unicode still not working. we end up with a user dir
  		  				// of, say, C:\Documents and Settings\t?stt?st\Application Data\Azureus
  		  				// with the unicode chars replaced with something unspeakable
  		  			
  		  			try{
  		  				File	up_file = new File( user_path );
  		  				
  		  				if ( !up_file.exists()){
  		  				
  		  					log.println( "user path not found, attempting to fix" );
  		  					
  		  						// see if we can patch things up

  		  					String	user_home = System.getProperty( "user.home" );
  		  					
  		  					File	az_config = findConfig(new File( user_home ));
  		  					
  		  					if ( az_config != null ){
  		  						
  		  						log.println( "    found config in " + az_config.toString() + ", using parent as user dir" );
  		  						
  		  						user_path	= az_config.getParent();
  		  					}
  		  					
  		  				}
  		  			}catch( Throwable e ){
  		  			
  		  				e.printStackTrace( log );
  		  			}
  				}
  	  		}
  	  
  			waitForAzureusClose( log );
		    
  			FileInputStream	fis = null;
  			
  			try{
  			
  				File	props_file = new File( user_path, UPDATE_PROPERTIES );
  				
  				if ( props_file.exists()){
  					  					
  					update_properties.load( new FileInputStream( props_file ));
  					
  					Iterator	it = update_properties.keySet().iterator();
  					
  					log.println("Loaded 'update.properties'" );
  					
  					while( it.hasNext()){
  						
  						String	key = (String)it.next();
  						
  						log.println( "    " + key + " = " + update_properties.get(key));
  					}
			
  				}else{
  					
  		 		    log.println( "No update.properties found" );
  		 		 
  				}
  			}catch( Throwable e ){
  				
  				log.println( "Failed to read update.properties" );
  				
  				e.printStackTrace( log );
  				
  			}finally{
  				if ( fis != null ){
  					
  					try{
  						fis.close();
  						
  					}catch( Throwable e ){
  						
  						e.printStackTrace( log );
  					}
  				}
  			}
  			
  		    File	update_dir = new File( user_path, UPDATE_DIR );
  		    
  		    File[]	inst_dirs = update_dir.listFiles();
  		    
  		    if ( inst_dirs == null ){
  		    	
  		    	inst_dirs = new File[0];
  		    }
  		    
  		    for (int i=0;i<inst_dirs.length;i++){
  		    	
  		    	File	inst_dir = inst_dirs[i];
  		    	
  		    	if ( inst_dir.isDirectory()){
  		    		
  		    		processUpdate( log, inst_dir );
  		    	}
  		    }
  		    
  		}catch( Throwable e ){
  			
  			if ( log != null ){
  				
  				log.println( "Update Fails" );
  				
  				e.printStackTrace( log );
  			}
  		}finally{
  			
  			if ( do_restart ){
  				
	  			if ( log != null ){
	  			
	  				log.println( "Restarting Azureus" );
	  			}
	  			
	  			String	vendor = System.getProperty( "java.vm.vendor" );
	  			
	  			Vector	vm_args = new Vector();
	  			
	  				// only for Sun JVM
	  			
	  			if ( vendor != null && vendor.toLowerCase().startsWith( "sun" )){
	  				
		  			String	max_mem = (String)update_properties.get( "max_mem" );
		  			
		  				// for previous versions we default to 128m
		  			
		  			long	max_mem_l = 0;
		  					  				
		  			if ( max_mem == null ){
		  				
		  				max_mem_l = 128*1024*1024;
		  				
		  			}else{
		  				
		  				try{
		  					
		  					max_mem_l = Long.parseLong( max_mem );
		  					
		  				}catch( Throwable e ){
		  					
		  					e.printStackTrace( log );
		  				}
		  				
		  			}
		  			
		  				// don't allow < 64m to be specified
		  			
		  			if ( max_mem_l < 64*1024*1024 ){
		  				
		  				max_mem_l = 64*1024*1024;
		  			}
		  			
		  			vm_args.add( "-Xmx" + (max_mem_l/(1024*1024)) + "m" );
	  			}
	  			
	  			vm_args.add( "-Duser.dir=\"" + app_path + "\"" );
	  				  			
	  			if ( config_override.length() > 0 ){
	  				
	  				vm_args.add( "-Dazureus.config.path=\"" + config_override + "\"" );
	  			}
	  				
	  			String[]	props = new String[vm_args.size()];
	  		
	  			vm_args.toArray( props );
	  			
	            restartAzureus( log, MAIN_CLASS, props, new String[0] );
	            
	 			if ( log != null ){
	 	  			
	 				log.println( "Restart initiated" );
	 			}
  			}else{
  				
	  			if ( log != null ){
		  			
		  			log.println( "Not restarting Azureus" );
		  		}
  			}
	 		
  			if ( log != null ){
	 	  				 	  			
	 	  		log.close();
	 	  	}
  		}
	}
  	
  	protected File
  	findConfig(
  		File	dir )
  	{
  		File[]	files = dir.listFiles();
  		
  		if ( files != null ){
  			
  			for (int i=0;i<files.length;i++){
  				
  				File	f = files[i];
  				
  				if ( f.isDirectory()){
  					
  					File	res = findConfig( f );
  					
  					if ( res != null ){
  						
  						return( res );
  					}
  				}else{
  					
  					if ( f.getName().equals( "azureus.config" )){
  						
  						return( f );
  					}
  				}
  			}
  		}
  		
  		return( null );
  	}
  	
  	protected void
	processUpdate(
		PrintWriter	log,
		File		inst_dir )
	{
		log.println( "processing " + inst_dir.toString());
	
		try{
			
			File	commands = new File( inst_dir, ACTIONS );
			
			if ( !commands.exists()){
				
				log.println( "    command file '" + ACTIONS + "' not found, aborting");
				
				return;
			}
			
			LineNumberReader	lnr = new LineNumberReader( new FileReader( commands ));
				
			boolean	failed = false;
			
			while(true){
					
				String	line = lnr.readLine();
					
				if ( line == null ){
						
					break;
				}
					
				log.println( "    command:" + line );
				
				StringTokenizer tok = new StringTokenizer(line, ",");
				
				String	command = tok.nextToken();
				
				if ( command.equals( "move" )){
					
					File	from_file 	= new File(tok.nextToken());
					File	to_file		= new File(tok.nextToken());
					
					if ( to_file.exists()){
						
						deleteDirOrFile( log, to_file );
					}
					
					if ( !renameFile( log, from_file, to_file )){
						
						failed	= true;
					}
					
				}else if ( command.equals( "remove" )){
					
					File	file 	= new File(tok.nextToken());
					
					if ( file.exists()){
						
						if ( !deleteDirOrFile( log, file )){
							
							failed	= true;
						}
					}
					
				}else if ( command.equals("chmod")){
					
					String rights = tok.nextToken();
					
					String fileName = tok.nextToken();
					
					chMod(fileName,rights,log);          
				}else{
					
					log.println( "unrecognised command '" + command + "'" );
					
					failed	= true;
				}
			}
			
			lnr.close();
			
			if ( !failed ){
				
				deleteDirOrFile( log, inst_dir );
			}
		}catch( Throwable e ){
  			 				
  			log.println( "processing fails" );
  				
  			e.printStackTrace( log );
		}
	}

  	private boolean
	renameFile(
		PrintWriter	log,
		File		from_file,
		File		to_file )
	{			
 		try{
 			Throwable 	last_error = null;
  	  	    	  	      
  			for (int i=0;i<10;i++){
  				
  				if ( i > 0 ){
  					
	 				try{
	  					Thread.sleep(1000);
	  	      		
	  				}catch( InterruptedException e ){
	  				} 
	  	 	      	
	  				log.println( "   rename of '" + from_file.toString() + "' failed, retrying" );
 				}
  				
  				if ( from_file.renameTo( to_file )){
  	      		  	  					
  					return( true );
  					
  				}else{
  					
  						// can't rename across file systems under Linux - try copy+delete
  					
					FileInputStream		fis = null;
						
					FileOutputStream	fos = null;

  					try{
  						fis = new FileInputStream( from_file );
  						
  						fos = new FileOutputStream( to_file );

  						byte[]	buffer = new byte[65536];
  						
  						while( true ){
  							
  							int	len = fis.read( buffer );
  							
  							if ( len <= 0 ){
  								
  								break;
  							}
  							
  							fos.write( buffer, 0, len );
  						}
  						
  						fos.close();
  						
  						fos	= null;
  						
  						fis.close();
  						
  						fis = null;
  						
  						from_file.delete();
  						
  						return( true );
  						
  					}catch( Throwable e ){
  						
  						last_error	= e;
  						
  					}finally{
  						
  						if ( fis != null ){
  							
  							try{
  								fis.close();
  							}catch( Throwable e ){
  							}
  						}
  						
  						if ( fos != null ){
							try{
  								fos.close();
  							}catch( Throwable e ){
  							}
  						}
  					}
  				}
   			}
  			  				
  			log.println( "failed to rename '" + from_file.toString() + "' to ' " + to_file.toString() + "'");
  			
  			if ( last_error != null ){
  				
  				last_error.printStackTrace( log );
  			}
  			
  	    }catch( Throwable e ){
  	    	
  	    	e.printStackTrace(log);
  	    }
  	    
  	    return( false );
  	}
  	
  	private boolean 
  	deleteDirOrFile(
  		PrintWriter	log,
		File 		f) 
	{
  		try{
  			if ( f.isDirectory()){
  	      	
  				File[] files = f.listFiles();
  	        
  				if ( files != null ){
  					
 	  				for (int i = 0; i < files.length; i++){
		  	
 	  						// don't propagate failures here, if something goes wrong
 	  						// then it'll fail higher up the hierarchy
 	  					
	  					deleteDirOrFile(log, files[i]);
  					}
  				}
  			} 
  	      	      
  			for (int i=0;i<10;i++){
  				
  				if ( i > 0 ){
  					
	 				try{
	  					Thread.sleep(1000);
	  	      		
	  				}catch( InterruptedException e ){
	  				} 
	  	 	      	
	  				log.println( "   delete of '" + f.toString() + "' failed, retrying" );
 				}
  				
  				if ( f.delete()){
  	      		
  					return( true );
  				}
   			}
  			  				
  			log.println( "Failed to delete '" + f.toString() + "'" );
 
  	    }catch( Exception e ){
  	    	
  	    	e.printStackTrace(log);
  	    }
  	    
  	    return( false );
  	}
  	
  	private void
	waitForAzureusClose(
		PrintWriter			log )
	{
  		log.println( "Waiting to bind to port 6880" );
  	     
  	    boolean ok = false;
  	     
  	    int		loop	= 0;
  	    
  	    try{
  
	  	    while(!ok){
	  	    	try{
	  	    		ServerSocket server = new ServerSocket(6880, 50, InetAddress.getByName("127.0.0.1"));
	  	    		
	  	    		ok = true;
	  	    		
	  	    		server.close();
	  	    		
	  	        }catch(Exception e){
	  	        	
	  	        	log.println( "Exception while trying to bind on port 6880 : " + e);
	  	          
	  	        	loop++;
	  	        	
	  	        	if ( loop == 30 ){
	  	        		
	  	        		log.println( "Giving up on bind" );
	  	        		
	  	        		return;
	  	        	}
	  	        	
	  	          	Thread.sleep(1000);
	  	        }
	  	    }
	  	      
	  	    	//Wait 1 sec more anyway.
	  	    
	  	    Thread.sleep(1000);
	  	    
  	    }catch( InterruptedException e ){
  	    	
  	    	e.printStackTrace(log);
  	    }
  	}
  	
	private String 
	format(
		int n) 
	{
	   if(n < 10) return "0".concat(String.valueOf(n));
	   return String.valueOf(n);
	}  
	
	
	private String
	getClassPath()
	{
		String _classPath = System.getProperty("java.class.path");
	    	// we've got to remove Updater.jar from the classpath else if there's a
			// updater to Updater.jar we're stuffed
		
		StringTokenizer tok = new StringTokenizer( _classPath, System.getProperty("path.separator"));
		
		String	classPath = "";
			
		while( tok.hasMoreTokens()){
			
			String	bit = tok.nextToken();
			
			if ( !bit.endsWith( File.separator + UPDATER_JAR )){
				
				classPath += (classPath.length()==0?"":System.getProperty("path.separator")) + bit;
			}
		}
			    
	    return( "-classpath \"" + classPath + "\" " );
	}
	
	private boolean
	win32NativeRestart(
		PrintWriter	log,
		String		exec )
	{
		return( false );
	}
	
	public static final String  OSName = System.getProperty("os.name");
	  
	public static final boolean isOSX			= OSName.equalsIgnoreCase("Mac OS X");
	public static final boolean isLinux			= OSName.equalsIgnoreCase("Linux");
	public static final boolean isWindows		= !(isOSX || isLinux);
	
	private boolean
	isOSX()
	{
		return( isOSX );
	}
	
	private boolean
	isLinux()
	{
		return( isLinux );
	}
	
	 // ****************** This code is copied into Restarter / Updater so make changes there too !!!
	  
	  //Beware that for OSX no SPECIAL Java will be used with
	  //This method.
	  
	  private static final String restartScriptName = "restartScript";
	  
	  public void 
	  restartAzureus(
	      PrintWriter log, 
	    String    mainClass,
	    String[]  properties,
	    String[]  parameters ) 
	  {
	    if(isOSX()){
	    	
	    	restartAzureus_OSX(log,mainClass,properties,parameters);
	    	
	    }else if( isLinux() ){
	    	
	    	restartAzureus_Linux(log,mainClass,properties,parameters);
	      
	    }else{
	    	
	    	restartAzureus_win32(log,mainClass,properties,parameters);
	    }
	  }
	  
	  private void 
	  restartAzureus_win32(
	      PrintWriter log,
	    String    mainClass,
	    String[]  properties,
	    String[]  parameters) 
	  {
	    
	    //Classic restart way using Runtime.exec directly on java(w)
	     String javaPath = System.getProperty("java.home")
	                    + System.getProperty("file.separator")
	                    + "bin"
	                    + System.getProperty("file.separator");
	    
	    String exec = "\"" + javaPath + "javaw\" "+ getClassPath() +
	            getLibraryPath();
	    
	    for (int i=0;i<properties.length;i++){
	      exec += properties[i] + " ";
	    }
	    
	    exec += mainClass;
	    
	    for(int i = 0 ; i < parameters.length ; i++) {
	      exec += " \"" + parameters[i] + "\"";
	    }
	    
	    if ( log != null ){
	      log.println( "  " + exec );
	    }
	    
	    if ( !win32NativeRestart( log, exec )){
	      
	      // hmm, try java method - this WILL inherit handles but might work :)
	          
	        try{
	          Runtime.getRuntime().exec(exec);
	          
	        }catch(Throwable f){
	          
	          f.printStackTrace( log );
	        }
	    }
	  }
	  
	  private void 
	  restartAzureus_OSX(
	      PrintWriter log,
	    String mainClass,
	    String[]  properties,
	    String[] parameters) 
	  {
	    String userPath = System.getProperty("user.dir");
	    String javaPath = System.getProperty("java.home")
	                    + System.getProperty("file.separator")
	                    + "bin"
	                    + System.getProperty("file.separator");
	    
	    String exec =   "#!/bin/bash\n" + 
	                  	"ulimit -H -S -n 8192\n\"" +
						userPath + "/Azureus.app/Contents/MacOS/java_swt\" " + getClassPath() +
						getLibraryPath();
	    
	    for (int i=0;i<properties.length;i++){
	      exec += properties[i] + " ";
	    }
	    
	    exec += mainClass ;
	    
	    for(int i = 0 ; i < parameters.length ; i++) {
	      exec += " \"" + parameters[i] + "\"";
	    }
	    
	    if ( log != null ){
	      log.println( "  " + exec );
	    }
	    String fileName = userPath + "/Azureus.app/" + restartScriptName;
	    
	    File fUpdate = new File(fileName);
	    
	    try {
	      FileOutputStream fosUpdate = new FileOutputStream(fUpdate,false);
	      fosUpdate.write(exec.getBytes());
	      fosUpdate.close();
	      chMod(fileName,"755",log);      
	      Process p = Runtime.getRuntime().exec("Azureus.app/" + restartScriptName);
	    } catch(Exception e) {
	      log.println(e);
	      e.printStackTrace(log);
	    }
	  }
	  
	  private void 
	  restartAzureus_Linux(
	    PrintWriter log,
	  String    mainClass,
	  String[]  properties,
	  String[]  parameters) 
	  {
	    String userPath = System.getProperty("user.dir"); 
	    String javaPath = System.getProperty("java.home")
	                    + System.getProperty("file.separator")
	                    + "bin"
	                    + System.getProperty("file.separator");
	    
	    String exec =   "#!/bin/bash\n\"" + javaPath + "java\" " + getClassPath() +
	            		getLibraryPath();
	    
	    for (int i=0;i<properties.length;i++){
	      exec += properties[i] + " ";
	    }
	    
	    exec += mainClass ;
	    
	    for(int i = 0 ; i < parameters.length ; i++) {
	      exec += " \"" + parameters[i] + "\"";
	    }
	    
	    if ( log != null ){
	      log.println( "  " + exec );
	    }
	    
	    String fileName = userPath + "/" + restartScriptName;
	    
	    File fUpdate = new File(fileName);
	    try {
	      FileOutputStream fosUpdate = new FileOutputStream(fUpdate,false);
	      fosUpdate.write(exec.getBytes());
	      fosUpdate.close();
	      chMod(fileName,"755",log);
	      Process p = Runtime.getRuntime().exec("./" + restartScriptName);
	    } catch(Exception e) {
	      log.println(e);  
	      e.printStackTrace(log);
	    }
	  }
	  
	  private String
	  getLibraryPath()
	  {
	    String libraryPath = System.getProperty("java.library.path");
	    
	    if ( libraryPath == null ){
	    	
	      libraryPath = "";
	      
	    }else{
	    	
	    		// remove any quotes from the damn thing
	    	
	    	String	temp = "";
	    	
	    	for (int i=0;i<libraryPath.length();i++){
	    		
	    		char	c = libraryPath.charAt(i);
	    		
	    		if ( c != '"' ){
	    			
	    			temp += c;
	    		}
	    	}
	    	
	    	libraryPath	= temp;
	    	
	    		// remove trailing separator chars if they exist as they stuff up
	    		// the following "
	    	
	    	while( libraryPath.endsWith(File.separator)){
	    	
	    		libraryPath = libraryPath.substring( 0, libraryPath.length()-1 );
	    	}
	    	
	    	if ( libraryPath.length() > 0 ){
	  
	    		libraryPath = "-Djava.library.path=\"" + libraryPath + "\" ";
	    	}
	    }
	    
	    return( libraryPath );
	  }
	  
	  private void logStream(String message,InputStream stream,PrintWriter log) {
	    BufferedReader br = new BufferedReader (new InputStreamReader(stream));
	    String line = null;
	    log.println(message);
	    try {
	      while((line = br.readLine()) != null) {
	        log.println(line);
	      }
	    } catch(Exception e) {
	       log.println(e);
	       e.printStackTrace(log);
	    }
	  }
	  
	  private void chMod(String fileName,String rights,PrintWriter log) {
	    String[] execStr = new String[3];
	    execStr[0] = "chmod";
	    execStr[1] = rights;
	    execStr[2] = fileName;
	    log.println("About to execute : "  + execStr[0] + " " + execStr[1] + " " + execStr[2]);
	    try {
	      Process pChMod = Runtime.getRuntime().exec(execStr);
	      pChMod.waitFor();
	      logStream("Execution Output",pChMod.getInputStream(),log);
	      logStream("Execution Error",pChMod.getErrorStream(),log);
	    } catch(Exception e) {
	      log.println(e);
	      e.printStackTrace(log);
	    }
	  }
	  
	}