
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.component;

import java.awt.*;
import javax.swing.JPanel;

/**
 * Used to produce a new component that consists of some child component
 * with padding around the edge.
 * <p>
 * This is a container class; the container is designed to hold only one
 * child, which is passed in the constructor.  The child will be placed
 * in the container with the requested padding around the edge, and will
 * grow and shrink with the container if desired.
 * <p>
 * The container is put together when the constructor is called; no
 * components should be explicitly added or removed.
 */
public class PaddedPane extends JPanel {
    /**
     * The component held in this container.
     * @serial
     */
    private Component contents;
    
    /**
     * The distances in pixels between the child component and each edge
     * of the container.
     * @serial
     */
    private Insets insets;

    /**
     * Creates a <tt>PaddedPane</tt> containing the given component and
     * with the same padding on all four sides.  The component will grow
     * and shrink with the container.
     *
     * @param contents the component to be placed in the container.
     * @param inset the distance in pixels to place between the given
     * component and the edge of the container; this will be the same
     * for all four sides of the component.
     */
    public PaddedPane(Component contents, int inset) {
        this(contents, inset, true);
    }

    /**
     * Creates a <tt>PaddedPane</tt> containing the given component and
     * with the same padding on all four sides.
     *
     * @param contents the component to be placed in the container.
     * @param inset the distance in pixels to place between the given
     * component and the edge of the container; this will be the same
     * for all four sides of the component.
     * @param grow <tt>true</tt> if the component should grow and shrink
     * with the container, or <tt>false</tt> if it should remain its
     * preferred size.
     */
    public PaddedPane(Component contents, int inset, boolean grow) {
        this.contents = contents;
        insets = new Insets(inset, inset, inset, inset);
        init(grow);
    }

    /**
     * Creates a <tt>PaddedPane</tt> containing the given component and
     * with the requested padding on each side.  The component will
     * grown and shrink with the container.
     *
     * @param contents the component to be placed in the container.
     * @param top the distance in pixels to place between the given
     * component and the top of the container.
     * @param left the distance in pixels to place between the given
     * component and the left of the container.
     * @param bottom the distance in pixels to place between the given
     * component and the bottom of the container.
     * @param right the distance in pixels to place between the given
     * component and the right of the container.
     */
    public PaddedPane(Component contents, int top, int left, int bottom,
            int right) {
        this(contents, top, left, bottom, right, true);
    }

    /**
     * Creates a <tt>PaddedPane</tt> containing the given component and
     * with the requested padding on each side.
     *
     * @param contents the component to be placed in the container.
     * @param top the distance in pixels to place between the given
     * component and the top of the container.
     * @param left the distance in pixels to place between the given
     * component and the left of the container.
     * @param bottom the distance in pixels to place between the given
     * component and the bottom of the container.
     * @param right the distance in pixels to place between the given
     * component and the right of the container.
     * @param grow <tt>true</tt> if the component should grow and shrink
     * with the container, or <tt>false</tt> if it should remain its
     * preferred size.
     */
    public PaddedPane(Component contents, int top, int left, int bottom,
            int right, boolean grow) {
        this.contents = contents;
        insets = new Insets(top, left, bottom, right);
        init(grow);
    }
    
    /**
     * Creates a <tt>PaddedPane</tt> containing the given component and
     * with the requested padding on each side.  The component will
     * grown and shrink with the container.
     *
     * @param contents the component to be placed in the container.
     * @param insets contains the distances in pixels to place between the
     * given component and each edge of the container.
     */
    public PaddedPane(Component contents, Insets insets) {
        this(contents, insets, true);
    }

    /**
     * Creates a <tt>PaddedPane</tt> containing the given component and
     * with the requested padding on each side.
     *
     * @param contents the component to be placed in the container.
     * @param insets contains the distances in pixels to place between the
     * given component and each edge of the container.
     * @param grow <tt>true</tt> if the component should grow and shrink
     * with the container, or <tt>false</tt> if it should remain its
     * preferred size.
     */
    public PaddedPane(Component contents, Insets insets, boolean grow) {
        this.contents = contents;
        this.insets = (Insets)insets.clone();
        init(grow);
    }

    /**
     * Put together the container.
     *
     * @param grow <tt>true</tt> if the component should grow and shrink
     * with the container, or <tt>false</tt> if it should remain its
     * preferred size.
     */
    private void init(boolean grow) {
        this.setLayout(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        c.insets = insets;
        c.fill = grow ? c.BOTH : c.NONE;
        c.weightx = 1.0;
        c.weighty = 1.0;
        c.anchor = c.CENTER;
        add(contents, c);
    }
    
    /**
     * Returns the child component of this container.
     *
     * @return the child component.
     */
    public Component getContents() {
        return contents;
    }
    
    /**
     * Returns the padding between the child component and the edges of
     * the container.
     *
     * @return the distances in pixels to each edge of the container.
     */
    public Insets getInsets() {
        return insets;
    }
}
