/*
 * $Header: /home/cvspublic/jakarta-commons/httpclient/src/java/org/apache/commons/httpclient/HttpClient.java,v 1.54 2002/08/09 04:44:16 sullis Exp $
 * $Revision: 1.54 $
 * $Date: 2002/08/09 04:44:16 $
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "HttpClient", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.commons.httpclient;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.io.IOException;
import java.net.URL;
import javax.net.ssl.SSLSocketFactory;


/**
 * <p>
 * An HTTP "user-agent", containing an {@link HttpState} and
 * one or more {@link HttpConnection}s, to which
 * {@link HttpMethod}s can be applied.
 * </p>
 * @author <a href="mailto:remm@apache.org">Remy Maucherat</a>
 * @author <a href="mailto:rwaldhoff@apache.org">Rodney Waldhoff</a>
 * @author Sean C. Sullivan
 * @author <a href="mailto:dion@apache.org">dIon Gillard</a>
 * @author Ortwin Glck
 * @version $Revision: 1.54 $ $Date: 2002/08/09 04:44:16 $
 */

public class HttpClient {


    // -------------------------------------------------------------- Constants

    /** Log object for this class. */
    private static final Log log = LogFactory.getLog(HttpClient.class);

    // ----------------------------------------------------------- Constructors

    /**
     * Constructor.
     */
    public HttpClient() {
    }

    // ----------------------------------------------------- Instance Variables

    /**
     * My {@link HttpConnection connection}.
     */
    private HttpConnection connection = null;

    /**
     * My {@link HttpState state}.
     */
    private HttpState state;

    private SSLSocketFactory sslSocketFactory = null;

    private int timeoutInMilliseconds = 0;

    // ------------------------------------------------------------- Properties

    /**
     * Get my {@link HttpState state}.
     *
     * @see #setState(HttpState)
     * @return the shared client state
     */
    public HttpState getState() {
        if (null == state) {
            state = new HttpState();
        }
        return state;
    }

    /**
     * Set my {@link HttpState state}.
     *
     * @see #getState()
     * @param state the new state for the client
     */
    public void setState(HttpState state) {
        this.state = state;
    }

    /**
     * Specifies an alternative factory for SSL sockets.
     * @see HttpConnection#setSSLSocketFactory(SSLSocketFactory) HttpConnection.setSSLSocketFactory
     * @param sslSocketFactory a living instance of the alternative SSLSocketFactory
     */
    public void setSSLSocketFactory(SSLSocketFactory sslSocketFactory) {
        this.sslSocketFactory = sslSocketFactory;
    }

    /**
     * Sets the SO_TIMEOUT which is the timeout for waiting for data.
     * 
     * A timeout value of zero is interpreted as an infinite timeout.
     * 
     * @param newTimeoutInMilliseconds Timeout in milliseconds
     * 
     */
    public void setTimeout(int newTimeoutInMilliseconds) {
        this.timeoutInMilliseconds = newTimeoutInMilliseconds;
    }

    // --------------------------------------------------------- Public Methods

    /**
     * Start an HTTP session with the server at the given
     * <i>host</i> and <i>port</i>.
     * @param host the host to connect to
     * @param port the port to connect to
     *
     * @see #endSession()
     */
    public void startSession(String host, int port) {
        log.trace("enter HttpClient.startSession(String, int)");
        startSession(host, port, false);
    }

    /**
     * Start an HTTP or HTTPS session with the server at the given
     * <i>host</i> and <i>port</i>.
     * @param host the host to connect to
     * @param port the port to connect to
     * @param https when <code>true</code>, create an HTTPS session
     *
     * @see #endSession()
     *
     */
    public void startSession(String host, int port, boolean https) {
        log.trace("enter HttpClient.startSession(String, int, boolean)");

        if (log.isDebugEnabled()) {
            log.debug("HttpClient.startSession(String,int,boolean): Host:"
                + host + " Port:" + port + " HTTPS:" + https);
        }
        connection = new HttpConnection(host, port, https);
    }

    /**
     * Start an HTTP session with the server at the given
     * <i>host</i> and <i>port</i> using the given default
     * default credentials.
     *
     * @param host the host to connect to
     * @param port the port to connect to
     * @param creds the default credentials to use
     * @see #endSession()
     */
    public void startSession(String host, int port, Credentials creds) {
        log.trace("enter HttpClient.startSession(String, int, Credentials)");
        startSession(host, port, creds, false);
    }


    /**
     * Start an HTTP or HTTPS session with the server at the given
     * <i>host</i> and <i>port</i> using the given default
     * default credentials.
     *
     * @param host the host to connect to
     * @param port the port to connect to
     * @param creds the default credentials to use
     * @param https when <code>true</code>, create an HTTPS session
     * @see #endSession()
     */
    public void startSession(String host, int port, Credentials creds, boolean https) {
        log.trace("enter HttpClient.startSession(String, int, Credentials, boolean)");

        if (log.isDebugEnabled()) {
            log.debug(
                "Starting HttpClient session" 
		+ " Host:" + host
	       	+ " Port:" + port + " Credentials:" + creds
                + " HTTPS:" + https);
        }
        getState().setCredentials(null, creds);
        connection = new HttpConnection(host, port, https);
    }

    /**
     * Start an HTTP or HTTPS session with the server specified
     * by the protocol, host and port of the given
     * <i>url</i>.
     * <p>
     * Note that everything but the protocol, host and port of the
     * given <i>url</i> is ignored.
     * </p>
     * @param url the {@link URL URL} from which the protocol, host,
     *            and port of the session are determined
     *
     * @see #endSession()
     *
     */
    public void startSession(URL url) {
        log.trace("enter HttpClient.startSession(String, int, Credentials, boolean)");

        if ("https".equalsIgnoreCase(url.getProtocol())) {
	    startSession(url.getHost(), url.getPort() == -1 ? 443
		  : url.getPort(), true);
        } else if ("http".equalsIgnoreCase(url.getProtocol())) {
            startSession(url.getHost(), url.getPort() == -1 ? 80
		  : url.getPort(), false);
        } else {
            throw new IllegalArgumentException("Protocol " + url.getProtocol()
		  + " not supported in URL " + url);
        }
    }

    /**
     * Start an HTTP or HTTPS session with the server specified
     * by the protocol, host and port of the given
     * <i>url</i>, using the given credentials by default.
     * <p>
     * Note that everything but the protocol, host and port of the
     * given <i>url</i> is ignored.
     * </p>
     * @param creds the default credentials to use
     * @param url the {@link URL URL} from which the protocol, host,
     *            and port of the session are determined
     *
     * @see #endSession()
     *
     */
    public void startSession(URL url, Credentials creds) {
        log.trace("enter HttpClient.startSession(URL, Credentials)");
        getState().setCredentials(null, creds);
        startSession(url);
    }

    /**
     * Start an HTTP session with the server specified
     * by the given <i>host</i> and <i>port</i>
     * via the given <i>proxyhost</i> and <i>proxyport</i>.
     * @param host the host to connect to
     * @param port the port to connect to
     * @param proxyhost the proxy host to connect via
     * @param proxyport the proxy port to connect via
     *
     * @see #endSession()
     *
     */
    public void startSession(String host, int port, String proxyhost, int proxyport) {
        log.trace("enter HttpClient.startSession(String, int, String, int)");
        startSession(host, port, proxyhost, proxyport, false);
    }


    /**
     * Start an HTTP session with the server specified
     * by the given <i>host</i> and <i>port</i>
     * via the given <i>proxyhost</i> and <i>proxyport</i>.
     * @param host the host to connect to
     * @param port the port to connect to
     * @param proxyhost the proxy host to connect via
     * @param proxyport the proxy port to connect via
     * @param secure whether or not to connect using HTTPS
     *
     * @see #endSession()
     *
     */
    public void startSession(String host, int port, String proxyhost, int proxyport, boolean secure) {
        log.trace("enter HttpClient.startSession(String, int, String, int, boolean)");
        connection = new HttpConnection(proxyhost, proxyport, host, port, secure);
    }

    /**
     * Execute the given {@link HttpMethod} using my current
     * {@link HttpConnection connection} and {@link HttpState}.
     *
     * @param method the {@link HttpMethod} to execute
     * @return the method's response code
     *
     * @throws IOException if an I/O error occurs
     * @throws HttpException if a protocol exception occurs
     * @throws IllegalStateException if the session has not been started
     */
    public synchronized int executeMethod(HttpMethod method)
        throws IOException, HttpException, IllegalStateException  {
        log.trace("enter HttpClient.executeMethod(HttpMethod)");

        if (null == method) {
            throw new NullPointerException("HttpMethod parameter");
        }

        if (null == connection) {
            throw new IllegalStateException(
                "The startSession method must be called before executeMethod");
        }

        if (!connection.isOpen()) {
            connection.setSSLSocketFactory(sslSocketFactory);
            connection.setSoTimeout(timeoutInMilliseconds);
            connection.open();
            if (connection.isProxied() && connection.isSecure()) {
                method = new ConnectMethod(method);
            }
        }
        return method.execute(getState(), connection);
    }

    /**
     * End the current session, closing my the associated
     * {@link HttpConnection connection} if any.
     *
     * @see #startSession(String, int)
     * @see #startSession(String, int, boolean)
     * @see #startSession(String, int, Credentials)
     * @see #startSession(String, int, Credentials, boolean)
     * @see #startSession(java.net.URL)
     * @see #startSession(java.net.URL, Credentials)
     * @see #startSession(String, int, String, int)
     * @throws IOException when i/o errors occur closing the connection
     */
    public void endSession() throws IOException {
        log.trace("enter HttpClient.endSession()");
        if (null != connection) {
            connection.close();
            connection = null;
        }
    }
}
