/*
 *  FreeGuide J2
 *
 *  Copyright (c) 2001-2004 by Andy Balaam and the FreeGuide contributors
 *
 *  Released under the GNU General Public License
 *  with ABSOLUTELY NO WARRANTY.
 *
 *  See the file COPYING for more information.
 *  Created on Jun 24, 2004
 */

package freeguide.gui.viewer;

import freeguide.*;
import freeguide.lib.fgspecific.FGPreferences;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import javax.swing.*;

/**
 * @author Yann Coupin
 * @version 2
 *
 * A class to store the channel information and maintain them together
 */
public class ChannelJLabel extends JLabel implements ComponentListener{

	private String id;
	private String name;
	private String cacheFileName;
	private String currentIconFileName;

	/**
	 * Reset the icon to the default one of the channel (or none if no default icon)
	 */
	public void setDefaultIcon() {
    	setIcon(cacheFileName);
    	currentIconFileName = cacheFileName;
	}
	
	/**
	 * Get the cache file path that stores the default icon
	 * @return the path to the cache file
	 */
	public String getCacheIconPath() {
		return cacheFileName;
	}
	
	/**
	 * Set the icon of the channel
	 * @param fileName the path to the image to use as an icon
	 */
	public void setIcon( String fileName ) {
        
		currentIconFileName = fileName;
        
		// Create the icon from the file
		ImageIcon icon = new ImageIcon( fileName );
        
        // Force the icon to have proportions 1.36 x 1
        
		int icon_width  = icon.getIconWidth();
        int icon_height = icon.getIconHeight();
        
        int new_icon_width  = -1;
        int new_icon_height = -1;
        
		int max_height = getHeight() - getInsets().top - getInsets().bottom;
        int max_width = (int)( 1.37 * (double)max_height );
        
        double new_over_old = (double)max_width / (double)icon_width;
            
        new_icon_width  = max_width;
        new_icon_height = (int)( (double)icon_height * new_over_old );

        if( new_icon_height > max_height ) {
            
            double new_over_old2 = (double)max_height / (double)new_icon_height;
            
            new_icon_height = max_height;
            new_icon_width  = (int)( (double)max_width * new_over_old2 );
            
        }
        
        super.setIcon( new ImageIcon( icon.getImage().getScaledInstance(
            new_icon_width, new_icon_height, Image.SCALE_AREA_AVERAGING ) ) );
        
    }
	
	/**
	 * @param id the Id of the channel
	 * @param name the displayed name of the channel
	 */
	public ChannelJLabel(String id, String name) {
		super(name);
		this.id = id;
		this.name = name;
		
		// Compute the cache fileName
		StringBuffer sb = FGPreferences.getIconCacheDir();
		sb.append( id.replace( '.', '_' ).replaceAll( "[^a-zA-Z0-9_]","-" ) );
		cacheFileName = sb.toString();
		addComponentListener(this);
	}
        
	/**
	 * @return Returns the id.
	 */
	public String getId() {
		return id;
	}
    
	/**
	 * @return Returns the name.
	 */
	public String toString() {
		return name;
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentHidden(java.awt.event.ComponentEvent)
	 */
	public void componentHidden(ComponentEvent e) {
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentMoved(java.awt.event.ComponentEvent)
	 */
	public void componentMoved(ComponentEvent e) {
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentResized(java.awt.event.ComponentEvent)
	 */
	public void componentResized(ComponentEvent e) {
		if( currentIconFileName != null )
			setIcon( currentIconFileName );
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentShown(java.awt.event.ComponentEvent)
	 */
	public void componentShown(ComponentEvent e) {
	}

	/* (non-Javadoc)
	 * @see javax.swing.JLabel#setIcon(javax.swing.Icon)
	 */
	public void setIcon(Icon icon) {
		currentIconFileName = null;
		super.setIcon(icon);
	}
    
    public int getRequiredWidth() {
        
        FontMetrics myFM = this.getFontMetrics( getFont() );
        int ans = myFM.stringWidth( getText() );
        
        Icon ic = super.getIcon();
        if( ic != null ) {
            ans += ic.getIconWidth();
        }
        
        return ans;
        
    }
    
}
