/*
 $Id: FieldNode.java,v 1.13 2005/02/03 20:37:05 blackdrag Exp $

 Copyright 2003 (C) James Strachan and Bob Mcwhirter. All Rights Reserved.

 Redistribution and use of this software and associated documentation
 ("Software"), with or without modification, are permitted provided
 that the following conditions are met:

 1. Redistributions of source code must retain copyright
    statements and notices.  Redistributions must also contain a
    copy of this document.

 2. Redistributions in binary form must reproduce the
    above copyright notice, this list of conditions and the
    following disclaimer in the documentation and/or other
    materials provided with the distribution.

 3. The name "groovy" must not be used to endorse or promote
    products derived from this Software without prior written
    permission of The Codehaus.  For written permission,
    please contact info@codehaus.org.

 4. Products derived from this Software may not be called "groovy"
    nor may "groovy" appear in their names without prior written
    permission of The Codehaus. "groovy" is a registered
    trademark of The Codehaus.

 5. Due credit should be given to The Codehaus -
    http://groovy.codehaus.org/

 THIS SOFTWARE IS PROVIDED BY THE CODEHAUS AND CONTRIBUTORS
 ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT
 NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 THE CODEHAUS OR ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 OF THE POSSIBILITY OF SUCH DAMAGE.

 */
package org.codehaus.groovy.ast;

import java.lang.reflect.Field;

import org.codehaus.groovy.ast.expr.Expression;
import org.codehaus.groovy.classgen.BytecodeHelper;
import org.objectweb.asm.Constants;

/**
 * Represents a field (member variable)
 * 
 * @author <a href="mailto:james@coredevelopers.net">James Strachan</a>
 * @version $Revision: 1.13 $
 */
public class FieldNode extends AnnotatedNode implements Constants {

    private String name;
    private int modifiers;
    private String type;
    private String owner;
    private Expression initialValueExpression;
    private boolean dynamicType;
    private boolean holder;

    public static FieldNode newStatic(Class theClass, String name) throws SecurityException, NoSuchFieldException {
        Field field = theClass.getField(name);
        String fldType = field.getType().getName();
        return new FieldNode(name, ACC_PUBLIC | ACC_STATIC, fldType, theClass.getName(), null);
    }

    public FieldNode(String name, int modifiers, String type, ClassNode owner, Expression initialValueExpression) {
        this(name, modifiers, type, owner.getName(), initialValueExpression);
    }

    public FieldNode(String name, int modifiers, String type, String owner, Expression initialValueExpression) {
        this.name = name;
        this.modifiers = modifiers;
        this.type = type;
        this.owner = owner;
        this.initialValueExpression = initialValueExpression;
        if (type == null || type.length() == 0) {
            this.dynamicType = true;
            if (initialValueExpression != null) {
                String initType = initialValueExpression.getType();
                if (initType != null && initType.length() > 0){
                    this.type = initType;
//                    this.dynamicType = false;
                }
                else {
                    this.type = "java.lang.Object";
//                    this.dynamicType = true;
                }
            }
            else {
                this.type = "java.lang.Object";
//                this.dynamicType = true;
            }
        }
        else {
            String boxedType = BytecodeHelper.getObjectTypeForPrimitive(type);
            this.type = boxedType;
            dynamicType = false;
        }
    }

    public Expression getInitialValueExpression() {
        return initialValueExpression;
    }

    public int getModifiers() {
        return modifiers;
    }

    public String getName() {
        return name;
    }

    public String getType() {
        return type;
    }

    public void setType(String type) {
        this.type = type;
    }
    
    public String getOwner() {
        return owner;
    }

    public boolean isHolder() {
        return holder;
    }

    public void setHolder(boolean holder) {
        this.holder = holder;
    }

    public boolean isDynamicType() {
        return dynamicType;
    }

    public void setModifiers(int modifiers) {
        this.modifiers = modifiers;
    }

    /**
     * @return true if the field is static
     */
    public boolean isStatic() {
        return (modifiers & Constants.ACC_STATIC) != 0;
    }
	/**
	 * @param owner The owner to set.
	 */
	public void setOwner(String owner) {
		this.owner = owner;
	}
}
