package com.ibm.bsf.engines.xslt;

import java.util.*;
import java.io.*;
import java.net.URL;

import org.w3c.dom.*;

import org.apache.xalan.xpath.XObject;
import org.apache.xalan.xpath.xml.XMLParserLiaison;
import org.apache.xalan.xslt.XSLTInputSource;
import org.apache.xalan.xslt.XSLTProcessor;
import org.apache.xalan.xslt.XSLTProcessorFactory;
import org.apache.xalan.xslt.XSLTResultTarget;
import org.apache.xalan.xslt.StylesheetRoot;

import com.ibm.bsf.*;
import com.ibm.bsf.util.BSFEngineImpl;
import com.ibm.bsf.util.BSFFunctions;

/**
 * Xerces XSLT interface to BSF. Requires Xalan and Xerces from Apache.
 * 
 * This integration uses the BSF registry to pass in any src document
 * and stylesheet base URI that the user may wish to set. 
 *
 * @author   Sanjiva Weerawarana
 * @author   Sam Ruby
 */
public class XSLTEngine extends BSFEngineImpl {
  // xsl processor used to eval xsl scripts
  XSLTProcessor xslp;

  /**
   * Initialize the engine.
   */
  public void initialize (BSFManager mgr, String lang,
                          Vector declaredBeans) throws BSFException {
    super.initialize (mgr, lang, declaredBeans);

    try {
      xslp = XSLTProcessorFactory.getProcessor();
    } catch (Exception e) {
      if (debug) {
        e.printStackTrace (debugStream);
      }
      throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
    "Exception from Xerces XSLT: " + e, e);
    }
  }

  /**
   * Evaluate an expression. In this case, an expression is assumed
   * to be a stylesheet of the template style (see the XSLT spec).
   */
  public Object eval (String source, int lineNo, int columnNo, 
          Object oscript) throws BSFException {
    // reset the xsl processor      
    xslp.reset ();

    // get the style base URI (the place from where Xerces XSLT will
    // look for imported/included files and referenced docs): if a
    // bean named "xslt:styleBaseURI" is registered, then cvt it
    // to a string and use that. Otherwise use ".", which means the
    // base is the directory where the process is running from
    Object sbObj = mgr.lookupBean ("xslt:styleBaseURI");
    String styleBaseURI = (sbObj == null) ? "." : sbObj.toString ();

    // Locate the stylesheet.
    XSLTInputSource styleSource;

    try {
      if (oscript instanceof Document) {
        styleSource = new XSLTInputSource((Document) oscript);
      } else {
        // cvt object to string and parse it. 
        styleSource = new XSLTInputSource(
          new StringReader (oscript.toString ()));
      }

      styleSource.setSystemId(styleBaseURI);
    } catch (Exception e) {
      if (debug) {
        e.printStackTrace (debugStream);
      }
        throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
              "Exception from Xerces XSLT: " + e, e);
      }

    // get the src to work on: if a bean named "xslt:src" is registered
    // and its a Node, then use it as the source. If its not a Node, then
    // if its a URL parse it, if not treat it as a file and make a URL and
    // parse it and go. If no xslt:src is found, use an empty document
    // (stylesheet is treated as a literal result element stylesheet)
    Object srcObj = mgr.lookupBean ("xslt:src");
    XSLTInputSource xis;
    if (srcObj != null) {
      if (srcObj instanceof Node) {
        xis = new XSLTInputSource((Node)srcObj);
      } else {
        try {
          String mesg = "as anything";
          if (srcObj instanceof URL) {
            xis = new XSLTInputSource(srcObj.toString());
            mesg = "as a URL";
          } else if (srcObj instanceof Reader) {
            xis = new XSLTInputSource ((Reader) srcObj);
            mesg = "as a Reader";
          } else {
            // try it as a file; if not then as an XML string
            String srcObjString = srcObj.toString ();
            Reader r = null;
            try {
              r = new FileReader (srcObjString);
              mesg = "as a file";
            } catch (Exception ee) {
              r = new StringReader (srcObjString);
              mesg = "as an XML string";
            } finally {
              xis = new XSLTInputSource (r);
              xis.setPublicId (srcObjString);
            }
          }

          if (xis == null) {
            throw new Exception ("Unable to get input from '" +
               srcObj + "' " + mesg);
          }
        } catch (Exception e) {
          throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
                "BSF:XSLTEngine: unable to get " +
                "input from '" + srcObj + "' as XML", e);
        }
      }
    } else {
      // create an empty document - real src must come into the 
      // stylesheet using "doc(...)" [see XSLT spec] or the stylesheet
      // must be of literal result element type
      xis = new XSLTInputSource();
    }
    
    // set all declared beans as parameters. Note that because we re-use
    // the XSLProcessor and because there's no "unsetStylesheetParam" 
    // method, this could leave extra garbage if the set of declared
    // beans were to dwindle across calls.
    for (int i = 0; i < declaredBeans.size (); i++) {
      BSFDeclaredBean b = (BSFDeclaredBean) declaredBeans.elementAt (i);
      xslp.setStylesheetParam (b.name, new XObject (b.bean));
    }

    // declare a "bsf" parameter which is the BSF handle so that 
    // the script can do BSF stuff if it wants to
    xslp.setStylesheetParam ("bsf", 
           new XObject (new BSFFunctions (mgr, this)));

    // do it
    try {
      XSLTResultTarget result = new XSLTResultTarget();
      xslp.process (xis, styleSource, result);
      return new XSLTResultNode (result.getNode());
    } catch (Exception e) {
      throw new BSFException (BSFException.REASON_EXECUTION_ERROR,
            "exception while eval'ing XSLT script" + e, e);
    }
  }

  /**
   * call the named method of the given object.
   */
  public Object call (Object object, String method, Object[] args) 
                                                        throws BSFException {
    throw new BSFException (BSFException.REASON_UNSUPPORTED_FEATURE,
          "BSF:XSLTEngine can't call methods");
  }

  public void declareBean (BSFDeclaredBean bean) throws BSFException {};

  public void undeclareBean (BSFDeclaredBean bean) throws BSFException {};
}
