/* 
 *   Creation Date: <1999/03/06 18:35:25 samuel>
 *   Time-stamp: <2000/10/29 16:46:15 samuel>
 *   
 *	<gc.c>
 *	
 *	Grand Central (GC)
 *
 *	This file adds the PCI device - the chips in the GC
 *	are added elsewhere.
 *
 *   Copyright (C) 1999, 2000 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include "verbose.h"
#include "promif.h"
#include "ioports.h"
#include "debugger.h"
#include "pci.h"
#include "gc.h"
#include "driver_mgr.h"

/* REG_COUNT needs to be a multiple of 4 in order not to
 * trigger add_io_range alignment error
 */
#define REG_COUNT	4
#define REG_OFFSET	0x30

static int gc_init( void );
static void gc_cleanup( void );

static ulong reg_io_read( ulong addr, int len, void *usr );
static void reg_io_write( ulong addr, ulong data, int len, void *usr );
static void reg_io_print( int isread, ulong addr, ulong data, int len, void *usr );
static void do_map( int regnum, ulong base, int add_map, void *usr );

driver_interface_t gc_driver =
{
    "gc", gc_init, gc_cleanup
};

static pci_dev_info_t gc_pci_config = {
	/* vendor, device ID, revision, class */
	0x106b, 0x0002, 0x02, 0xff00
};
static pci_dev_info_t heathrow_pci_config = {
	/* vendor, device ID, revision, class */
	0x106b, 0x0010, 0x02, 0xff00
};
static pci_dev_info_t ohare_pci_config = {
	/* vendor, device ID, revision, class */
	0x106b, 0x0001, 0x03, 0x0600
};

static io_ops_t macio_ops={
	NULL,
	reg_io_read,
	reg_io_write, 
	reg_io_print
};

static gc_range_t	*gc_regs_range = NULL;
static ulong 		registers[REG_COUNT];

static mol_device_node_t *gc_node = NULL;

static gc_range_t	*first_range = NULL;

int
gc_init( void )
{
	int bus, dev_fn;
	mol_device_node_t *dn;

	if( (dn = prom_find_devices( "gc" )) != NULL ) {
		if( pci_device_loc( dn, &bus, &dev_fn ) == 0) {
			// printm("GC, Grand Central, (%d:%02d) found\n", bus, (dev_fn>>3) );
			add_pci_device( bus, dev_fn, &gc_pci_config, NULL );
			set_pci_base( bus, dev_fn, 0, 0xfff80000, do_map );
			gc_node = dn;
		} else
			printm("gc: Couldn't get pci device number\n");
	}

	if( !dn && (dn = prom_find_devices( "ohare" )) != NULL ) {
		if( pci_device_loc( dn, &bus, &dev_fn ) == 0) {
			// printm("OHare, (%d:%02d) found\n", bus, (dev_fn>>3) );
			add_pci_device( bus, dev_fn, &ohare_pci_config, NULL );
				/* I guess these are endian reversed */
			registers[0] = 0x0000ff00;	/* from Starmax */
			registers[1] = 0x10000000;
			registers[2] = 0x00beff7a;
			gc_regs_range = add_gc_range( REG_OFFSET, REG_COUNT*4, "ohare regs", 0, &macio_ops, NULL);
			set_pci_base( bus, dev_fn, 0, 0xfff80000, do_map );
			gc_node = dn;
		} else
			printm("ohare: Couldn't get pci device number\n");
	}

	if( !dn && (dn = prom_find_devices( "mac-io" )) != NULL ) {
		if( pci_device_loc( dn, &bus, &dev_fn ) == 0) {
			// printm("Heathrow, (%d:%02d) found\n", bus, (dev_fn>>3) );

			add_pci_device( bus, dev_fn, &heathrow_pci_config, NULL );
			registers[0] = 0x0000FF00; 	/* from G3 */
			registers[1] = 0x391070F0;
			registers[2] = 0x61befffb; 	/* fcr */
			gc_regs_range = add_gc_range( REG_OFFSET, REG_COUNT*4, "mac-io regs", 0, &macio_ops, NULL);
			set_pci_base( bus, dev_fn, 0, 0xfff80000, do_map );
			gc_node = dn;
		} else
			printm("heathrow: Couldn't get pci device number\n");
	}
	
	return 1;
}


static void
gc_cleanup( void )
{
	gc_range_t *r, *next;

	for( r=first_range; r; r=next ){
		if( r->cleanup_fp )
			r->cleanup_fp( r->usr );
		next = r->next;
		free( r );
	}
	first_range = NULL;
}


static ulong 
reg_io_read( ulong addr, int len, void *usr )
{
	ulong base = get_mbase( gc_regs_range );

	if ((addr >= base) && (addr < (base + REG_COUNT*4))) {
		return read_mem( ((char *)registers) + (addr - base), len );
	}
	
	printm("heathrow: invalid reg read @0x%lx, len: %d\n", addr, len);
	return 0;
}

static void 
reg_io_write( ulong addr, ulong data, int len, void *usr )
{
	ulong base = get_mbase( gc_regs_range );

	if ((addr >= base) && (addr < (base + REG_COUNT*4))) {
		write_mem( ((char *)registers) + (addr - base), data, len);
		return;
	}
	
	printm("heathrow: invalid reg write @0x%lx, len: %d\n", addr, len);
}

static void 
reg_io_print( int isread, ulong addr, ulong data, int len, void *u_bridge )
{
	printm("heathrow/ohare: %08lX: (len=%d) %s %08lX\n", addr, len,
	       isread ? "read: " : "write:", data );
}

int 
is_gc_child( mol_device_node_t *dn )
{
	return is_ancestor( gc_node, dn );
}

extern ulong 
gc_offs( ulong mbase )
{
	return mbase - gc_node->addrs[0].address;
}

gc_range_t *
add_gc_range( int offs, size_t size, char *name, int flags,
		 io_ops_t *ops, void *usr )
{
	gc_range_t *r = calloc( sizeof( gc_range_t ),1 );
	
	r->next = first_range;
	first_range = r;
	
	r->offs = offs;
	r->size = size;
	r->name = name;
	r->flags = flags;
	r->ops = ops;
	r->cleanup_fp = r->ops->cleanup_fp;
	r->ops->cleanup_fp = NULL;

	r->usr = usr;

	r->is_mapped = 0;
	return r;
}

int
gc_is_present( void )
{
	return gc_node ? 1:0;
}

static void 
do_map( int regnum, ulong base, int add_map, void *usr )
{
	gc_range_t *r;       

	if( add_map )
		printm("Mapping GC at %08lX\n", base);
	else
		printm("Unmapping GC\n");

	for( r=first_range; r; r=r->next ) {
		if( r->is_mapped )
			remove_io_range( r->range_id );
		
		if( add_map ) {
			r->base = r->offs + base;
			r->range_id = add_io_range( r->base, r->size, r->name, 
						    r->flags, r->ops, r->usr );
		}
		r->is_mapped = add_map;
	}
}

