// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/train.cpp,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//



#include "platform.h"
#include "train.h"
#include "card_table.h"
#include "gc_consts.h"
#include "gc_plan.h"
#include "gc_hooks.h"

Train::Train(unsigned long    train_number,
             Gc_Fast_Hooks    *p_gc_hooks,
             Gc_Plan          *p_gc_plan,
		     Train_Generation *p_container,
             Card_Table       *p_card_table,
		     Block_Store      *p_block_store)

		     : Gc_Space(train_number,
                        p_gc_hooks,
                        p_gc_plan,
		                p_container,
					    NULL,
                        p_card_table,
					    p_block_store)
{
    _p_train_generation = p_container;
	//
	// Cars are created lazily as needed.
	//
	_number_of_cars     = 0;

#if (GC_DEBUG>0)
	train_creation_hook(this);
#endif
}

Train::~Train()
{
#if (GC_DEBUG>0)
	train_deletion_hook(this);
#endif

	for (unsigned long car_index = 0; 
	              car_index < _number_of_cars; 
				  car_index++){
		delete _p_cars[car_index];
	}

	return;
}


void
Train::_add_car()
{
    _p_cars[_number_of_cars] = 
                    new Car(_number_of_cars,
                            _p_gc_hooks,
                            _p_gc_plan,
		                    this,
                            _p_train_generation,
                            _p_card_table,
		                    _p_block_store);
	_number_of_cars++;
    _p_cars[_number_of_cars] = NULL;    // Always keep the next higher car
                                        // invalid. 
#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - Adding car number " << (_number_of_cars - 1) 
        << " to train " << this << endl;
#endif
	//
	// Blocks get added lazily.
	//
}

//
// Do any required cleanup at the end of the collection.
//
void 
Train::cleanup() 
{
	assert(0); // stub
}
    
//
// The oldest car has been completely evacuated.
// Delete it.
// For Debug -  check that all cards are 0 when this car is released.
//***************************************** NOTDONE ***************
// Clear the last_object_allocated table for all cards used in this car.
//
//
void
Train::_free_oldest_car()
{
#if (GC_DEBUG>0)
	assert(_number_of_cars > 0);

//    inspect (0);
//    orp_cout << " -----START---------- Freeing oldest car " << _p_cars[0] << endl;
#endif // _DEBUG

    delete _p_cars[0];

#if (GC_DEBUG>0)
//    orp_cout << " ------END----------- Freeing oldest car " << _p_cars[0] << endl;
#endif

    _p_cars[0] = NULL;

	for (unsigned long index = 0; 
         index < (_number_of_cars - 1); 
         index++) {

		_p_cars[index] = _p_cars[index + 1];
		_p_cars[index]->set_id(index);
    }

    _p_cars[_number_of_cars - 1] = NULL;
    // Make sure the next oldest if null or someone didn't clear it.
    assert (_p_cars[_number_of_cars] == NULL);
    
    _number_of_cars = _number_of_cars - 1;

#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - deleting car from train " << this << endl;
#endif // GC_TRAIN_TRACE

#ifdef GC_SAPPHIRE
	if (_number_of_cars == 0) {
		_p_train_generation->free_empty_oldest_train();
	}
#endif // GC_SAPPHIRE
	return; 
}

#if (GC_DEBUG>0)
//
// Debugging routine to display relevant information.
// Detail controlled by level.
//
void Train::inspect(unsigned int level)
{
	cout << "    Train " << this << " ID" << get_id ();
	cout << " Contains " << _number_of_cars << " cars." << endl;

	for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
		cout << "    Inspecting car # " << idx << endl;
		_p_cars[idx]->inspect(level);
	}
}
#endif // _DEBUG

//
// Membership checking routines for trains need to be overridden.
//
bool 
Train::is_my_object(Java_java_lang_Object *p_obj)
{
	Gc_Space *p_container = _p_block_store->p_get_object_container(p_obj);

	for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
		if (_p_cars[idx] == p_container) {
			return true;
		}
	}
	return false;
}

bool 
Train::is_my_reference(Java_java_lang_Object **pp_obj_ref)
{
	Gc_Space *p_container = 
		_p_block_store->p_get_reference_container(pp_obj_ref);

	for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
		if (_p_cars[idx] == p_container) {
			return true;
		}
	}
	return false;
}

//
// RLH-TRAIN This routine determines if the pointer is into the train, 
// This is needed to note that the train is alive.
//
bool 
Train::is_my_address(void *p_addr)
{
	Gc_Space *p_container = 
		_p_block_store->p_get_address_container(p_addr);

	for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
		if (_p_cars[idx] == p_container) {
			return true;
		}
	}
	return false;
}

bool
Train::is_reference_in_my_train(Java_java_lang_Object **pp_obj_ref)
{
	Gc_Space *p_space = 
		_p_block_store->p_get_address_container((void *)pp_obj_ref);

    if (p_space->is_car()) {

		if (((Car *)p_space)->p_get_train() == this) {
			return true;
		}
	}

	return false;
}
	
#if 0 // unused?
void
Train::move_object_to_end_of_train(Java_java_lang_Object **pp_ref)
{
    p_scavenge_object(pp_ref);
	// 
	// Eagerly complete any pending Cheney scans.
	//
    p_container->execute_pending_cheney_scans();
    return;
}
#endif // 0

//
// Check if the train is a candidate for reclamation.
// Note that this does not check the stack roots or the YOS.
// The YOS and the stack roots only need to indicate if there
// is a reference into the train so the boolean is_train_alive is
// useed.
bool 
Train::no_incoming_refs()
{
	for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
		if (_p_cars[idx]->inter_train_refs_exist()) {
			return false;
		}
	}
	return true;
}
//
// RLH-TRAIN why doesn't this reclaim all the cars in the train??
//
bool
Train::reclaim_train(Remembered_Set *p_all_outside_refs,
                     Remembered_Set *p_weak_refs)
{
    assert (_number_of_cars > 0);
    
	// Only reclaim the focus car...
	assert (_p_cars[0] == ((Mrl_Gc_V1 *)p_gc)->p_get_focus_car() );
	//
	// Reclaim the oldest car.
	//
    _p_cars[0]->reclaim_car(p_all_outside_refs, p_weak_refs);
    
    //
	// Free this car.
	//
    _free_oldest_car();

#if (GC_DEBUG>0)
    ((Train_Generation *)p_container)->verify_trains_cars_write_barriers();
#endif // _DEBUG
	//
	// If there are now no inter-train pointers into this train,
	// then signal the container generation that this train is
	// filled with garbage.
	//

    //
    // This can happen before the reclaimation of the car and it is 
    // is probable a good idea. It can also happen here. Two checks might
    // take a little more time but compared to evacuating a complete car the
    // check is not a big cost.
    //

    if (no_incoming_refs()) {
		return true;
	} 
#ifndef GC_TRAIN_V5
#ifdef GC_GEN_V3
	// For the GC_GEN_V3, which should have only one car.
    orp_cout << "Train that should have been collected has cars left in it." << endl;
    assert (0);
#endif // GC_GEN_V3
#endif // GC_TRAIN_V5

	return false;
}

//
// Run all finalizers at exit time if required by app.
//
void
Train::run_all_finalizers()
{
	for (unsigned long car_index = 0; 
	              car_index < _number_of_cars; 
				  car_index++){
		_p_cars[car_index]->run_all_finalizers();
	}
}

//
// Our generation is telling us that the specified
// object needs to be tenured out of the highest
// step of the next lower generation. We happen to
// be the highest numbered train. Let's pull it
// in:
// RLH-TRAIN This routine is called when an object is
// evicted from the focus car as well as when it is
// evicted from the lowest step.
Java_java_lang_Object *
Train::p_scavenge_object(Java_java_lang_Object **pp_obj)
{
	//
	// Lazily add a car if needed.
	//
	if (_number_of_cars == 0) {
		_add_car();
	}
    //
    // Try to add it to the last car.
    //
	Java_java_lang_Object *p_return_object = 
		_p_cars[_number_of_cars - 1]->p_scavenge_object(pp_obj);
    //
    // Oops - the last car is full: add another
    // and try again.
    //
	if (!p_return_object) {

#ifdef GEN_GC_V3
        // We should never need more than one car with the simple generational 
        // collector. If we do then one of two things is true.
        // We are out of heap space or we have incorrectly set up the
        // gc_plan.h plan so that cars are too small. They should be
        // about half the size of the older generations to allow for
        // both a to and a from space in the older generation.
        assert (0);
#endif // GEN_GC_V3

        _add_car();
		p_return_object = 
			_p_cars[_number_of_cars - 1]->p_scavenge_object(pp_obj);
	}

#if (GC_DEBUG>2)
	assert(p_return_object);
#endif // _DEBUG
 
    gc_trace (p_return_object, "scavenged into a train."); 

	return p_return_object;
}


bool
Train::cheney_scan_cars (bool doing_mos_collection)
{ 
#ifdef GC_TRAIN_TRACE
    orp_cout << "Train trace - Cheney scanning all " << _number_of_cars << " cars." << endl;
#endif // GC_TRAIN_TRACE
    // Scan each of the cars once.
    bool result = false;
    for (unsigned long car_idx = 0; car_idx < _number_of_cars; car_idx++) {
        if (_p_cars[car_idx]->cheney_scan_block_list(doing_mos_collection)) {
#ifdef GC_SAPPHIRE
    Car *p_focus_car = p_global_bs->get_mature_generation()->p_nominate_focus_car();
    // no objects should be moved into focus car so no work should need to be done.
    assert (p_focus_car != _p_cars[car_idx]); 
#endif // GC_SAPPHIRE
#ifdef GC_TRAIN_TRACE
            orp_cout << " Car number " << car_idx << " did some work." << endl;
#endif
            result = true;
        }
#ifdef GC_TRAIN_TRACE
        else {
            //orp_cout << " Car number " << car_idx << " did NO work." << endl;
        }
#endif
	}
    return result;
}

bool
Train::cheney_scan_pending ()
{
    // Check each of the cars.
    for (unsigned long car_idx = 0; car_idx < _number_of_cars; car_idx++) {
        if (_p_cars[car_idx]->cheney_scan_pending()) {
            return true;
        }
	}
    return false;
}
// #endif  // END NEW CHENEY CODE

void 
Train::enumerate_reference_queues () 
{
    for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
        _p_cars[idx]->enumerate_reference_queues();
    }
}

bool
Train::walk_train(bool (*func)(Object_Gc_Header *, Remembered_Set *p_rs))
{
    for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
        _p_cars[idx]->walk_car(func);
    }
    return true;
}


#if (GC_DEBUG>0)
//
// Debug routine to verify none of the consituent cars have invalid
// entries into their write barriers.
//
void Train::verify_cars_write_barriers()
{
    for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
        _p_cars[idx]->verify_car_write_barrier();
    }
}
#endif // _DEBUG

#if (GC_DEBUG>3)
//
// Each space provides a routine to verify that it is consistent.
// i.e. contains only well-formed objects.
//
bool Train::verify_space()
{
	for (unsigned long idx = 0; idx < _number_of_cars; idx++) {
		if (_p_cars[idx]->verify_space()) {
			continue;
		} else {
			return false;
		}
	}
	return true;
}
#endif // _DEBUG

// 
// sapphire_select_to_Train
// Input  - a slot holding a reference. Use the slot to determine where to move an object.
// Output - the train that this object should be moved to, it will go into the last car.
// If this is called and the pp_slot is in the focus car then we need to evict the
// object into the youngest train. This is reasonable since we can end up here as a 
// result of a write barrier call.
// 

Train *
Train::sapphire_select_to_train(Java_java_lang_Object **pp_slot)
{
    Train *result = p_global_bs->get_mature_generation()->p_get_youngest_train();
    Car *slot_car;
	if (p_global_bs->is_address_in_heap ((void *)pp_slot)) {
		if (p_global_bs->is_address_in_mature_generation((void *)pp_slot)) {
			result = (Train_Generation::slot_to_train (pp_slot));
            slot_car = (Train_Generation::slot_to_car (pp_slot));
            if (slot_car == ((Mrl_Gc_V1 *)p_gc)->p_get_focus_car() ) {
                // Don't move objects into focus car.
                result = p_global_bs->get_mature_generation()->p_get_youngest_train();
            }
		}
	}
    assert (result == p_global_bs->get_mature_generation()->p_get_youngest_train());
    return result;
}

//
// reserve_to_space
// Input   - The object that is going to be replicated in to space.
// Output  - The uninitialized replicated object.
// Comment - Use the object to figure out how much space to allocate.  
Java_java_lang_Object *
Train::p_reserve_space(Java_java_lang_Object **pp_obj)
{
	// Lazily add a car if needed.
	if (_number_of_cars == 0) {
		_add_car();
	}
    // Try to add it to the last car.
	Java_java_lang_Object *p_return_object = 
		_p_cars[_number_of_cars - 1]->p_reserve_space(pp_obj);

    // Oops - the last car is full: add another
    // and try again.
    //
	if (!p_return_object) {
#ifdef GEN_GC_V3
        // We should never need more than one car with the simple generational 
        // collector. If we do then one of two things is true.
        // We are out of heap space or we have incorrectly set up the
        // gc_plan.h plan so that cars are too small. They should be
        // about half the size of the older generations to allow for
        // both a to and a from space in the older generation.
        assert (0);
#endif // GEN_GC_V3
        _add_car();
		p_return_object = 
			_p_cars[_number_of_cars - 1]->p_reserve_space(pp_obj);
	}

#if (GC_DEBUG>2)
	assert(p_return_object);
#endif // _DEBUG

	return p_return_object;

}

//
// p_reserve_bytes
// Input   - The size in bytes
// Output  - An uninitialized replicated object of the appropriate size.
// Comment - Use the object to figure out how much space to allocate.
Java_java_lang_Object *
Train::p_reserve_bytes(unsigned size)
{
	// Lazily add a car if needed.
	if (_number_of_cars == 0) {
		_add_car();
	}
    // Try to add it to the last car.
	Java_java_lang_Object *p_return_object = 
		_p_cars[_number_of_cars - 1]->p_reserve_bytes(size);

    // Oops - the last car is full: add another
    // and try again.
    //
	if (!p_return_object) {
#ifdef GEN_GC_V3
        // We should never need more than one car with the simple generational 
        // collector. If we do then one of two things is true.
        // We are out of heap space or we have incorrectly set up the
        // gc_plan.h plan so that cars are too small. They should be
        // about half the size of the older generations to allow for
        // both a to and a from space in the older generation.
        assert (0);
#endif // GEN_GC_V3
		if (_number_of_cars > 4) {
			// Trigger a collection if you have allocated 2 cars.
			return NULL;
		}
        _add_car();
		p_return_object = 
			_p_cars[_number_of_cars - 1]->p_reserve_bytes(size);
	}

#if (GC_DEBUG>2)
	assert(p_return_object);
#endif // _DEBUG

	return p_return_object;

}


// end file train.cpp

