// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/include/Class.h,v 1.20 2002/01/10 22:51:33 ssubram5 Exp $
//

#ifndef _CLASS_H_
#define _CLASS_H_

#include <iostream.h>
#include "String_Pool.h"
#include "Signature.h"
#include "orp_types.h"
#include "jit_intf.h"

#include "orp_for_gc.h"


//
// forward declarations
//
class Field;
class Method;
class Package;
typedef struct Class Class;
class JIT;
class Class_Loader;
class ByteReader;

//
// external declarations
//
class Global_Env;	// defined in Environment.h

#ifdef CLI_TESTING
class CLI_TypeDef;
class CLI_Type;
class CLI_FieldOrMethod;
class CLI_Field;
class CLI_Method;
class CLI_Exception_Clause;
#endif



///////////////////////////////////////////////////////////////////////////////
// Constant Java values
///////////////////////////////////////////////////////////////////////////////
union Const_Java_Value {
    uint32 i;
    int64 j;
    struct {
        uint32 lo_bytes;
        uint32 hi_bytes;
    } l;
    float f;
    double d;
    String *string;
    void *object;
    Const_Java_Value() {l.lo_bytes=l.hi_bytes=0;}
};



///////////////////////////////////////////////////////////////////////////////
// Constant pool entries.
///////////////////////////////////////////////////////////////////////////////
union Const_Pool {
    unsigned char  *tags;       // entry 0 of const pool only
    uint16 name_index;          // CONSTANT_Class
    uint16 string_index;        // CONSTANT_String
   
    struct {                    // CONSTANT_{Field,Method,InterfaceMethod}ref
        uint16 class_index;
        uint16 name_and_type_index;
    } cp_tag;

    uint32      int_value;      // CONSTANT_Integer
    float       float_value;    // CONSTANT_Float
#ifdef POINTER64
    struct cp64x {
        uint32 low_bytes;  // each Const_Pool element is 64bit in this case 
        uint32 high_bytes; // we pack all 8 bytes of long/double in one
                           // Const_Pool element (and leave the second
                           // Const_Pool element of the long/double unused)
    } cp64x;
#else
    uint32	low_bytes;
    uint32	high_bytes;     // CONSTANT_{Long,Double}
#endif
    struct {                // CONSTANT_NameAndType
        uint16 name_index;
        uint16 descriptor_index;
    } ni_di;
    String *string;			// CONSTANT_Utf8
    //
    // after resolution, constant pool entries look like this
    //
    Signature	*signature;	// CONSTANT_NameAndType
    Class		*clss;		// CONSTANT_Class
    Field		*field;		// CONSTANT_Fieldref
    Method		*method;	// CONSTANT_{Interface}Methodref
};



///////////////////////////////////////////////////////////////////////////////
// Types of constant pool entries.  These are defined by a seperate byte array
// pointed to by the first constant pool entry.
///////////////////////////////////////////////////////////////////////////////
enum Const_Pool_Tags {
	CONSTANT_Class				=7,
	CONSTANT_Fieldref			=9,
	CONSTANT_Methodref			=10,
	CONSTANT_InterfaceMethodref	=11,
	CONSTANT_String				=8,
	CONSTANT_Integer			=3,
	CONSTANT_Float				=4,
	CONSTANT_Long				=5,
	CONSTANT_Double				=6,
	CONSTANT_NameAndType		=12,
	CONSTANT_Utf8				=1,
	CONSTANT_Tags				=0,	// pointer to tags array
};
#if 0
#define	CONSTANT_Class				7
#define	CONSTANT_Fieldref			9
#define	CONSTANT_Methodref			10
#define	CONSTANT_InterfaceMethodref	11
#define	CONSTANT_String				8
#define	CONSTANT_Integer			3
#define	CONSTANT_Float				4
#define	CONSTANT_Long				5
#define	CONSTANT_Double				6
#define	CONSTANT_NameAndType		12
#define	CONSTANT_Utf8				1

#define CONSTANT_Tags				0	// pointer to tags array
#endif

#define	TAG_MASK		0x0F	// 4 bits is sufficient for tag
#define	RESOLVED_MASK	0x80	// msb is resolved flag

#define	cp_tag(cp,i)			(cp[0].tags[i] & TAG_MASK)
#define	cp_is_resolved(cp,i)	(cp[0].tags[i] & RESOLVED_MASK)
#define cp_set_resolved(cp,i)	(cp[0].tags[i] |= RESOLVED_MASK)

#define cp_is_utf8(cp,i)	(cp_tag(cp,i) == CONSTANT_Utf8)
#define cp_is_class(cp,i)	(cp_tag(cp,i) == CONSTANT_Class)
#define cp_is_cnst(cp,i)	((cp_tag(cp,i) >= CONSTANT_Integer &&  cp_tag(cp,i) <= CONSTANT_Double) ||	cp_tag(cp,i) == CONSTANT_String)

#define cp_is_string(cp,i)				(cp_tag(cp,i) == CONSTANT_String)
#define cp_is_fieldref(cp,i)			(cp_tag(cp,i) == CONSTANT_Fieldref)
#define cp_is_methodref(cp,i)			(cp_tag(cp,i) == CONSTANT_Methodref)
#define cp_is_interfacemethodref(cp,i)	(cp_tag(cp,i) == CONSTANT_InterfaceMethodref)

#define cp_resolve_to_class(cp,i,c)	cp_set_resolved(cp,i); cp[i].clss=c;

#define cp_resolve_to_field(cp,i,f)	cp_set_resolved(cp,i); cp[i].field=f;

#define cp_resolve_to_method(cp,i,m) cp_set_resolved(cp,i); cp[i].method=m;

#if 0
// for ease of cutting and pasting switch statements
case CONSTANT_Class:
case CONSTANT_Fieldref:
case CONSTANT_Methodref:
case CONSTANT_InterfaceMethodref:
case CONSTANT_String:
case CONSTANT_Integer:
case CONSTANT_Float:
case CONSTANT_Long:
case CONSTANT_Double:
case CONSTANT_NameAndType:
case CONSTANT_Utf8:
#endif

#ifndef ORP_POSIX // for Linux, move to Loader_Result.h    
//
// class loader error conditions
//
enum Loader_Result {
	LD_OK,
	LD_ParseError,	// parse error occured
	LD_ConstPoolError,	// bad constant pool format
};
//
// class loader exceptions -- see pg 611 of the language definition to see list
// of exceptions
//
// see pg 39 of the JVM spec for an overview of the LinkageError exceptions
// see pg 45 of the JVM spec for an overview of the 
// IncompatibleClassChangeError exceptions
//
// Error
//   LinkageError
//     ClassCircularityError		Loader
//	   ClassFormatError				Loader
//	   NoClassDefFoundError			Loader
//	   IncompatibleClassChangeError			Linker				pg 45
//       AbstractMethodError						Preperation
//		 IllegalAccessError					Linker
//		 InstantiationError					Linker
//		 NoSuchFieldError					Linker
//		 NoSuchMethodError					Linker
//     UnsatisfiedLinkError									
//     VerifyError
//
// IncompatibleClassChangeError exceptions
//
// IllegalAccessError occurs when a a field, method or class reference 
// occurs and the referenced field (method) is private, protected or not public
// or the accessed class is not public.
//
// InstantiationError occurs when an interface or abstract class is 
// being instantiated by the byte codes.
//
// NoSuchField(Method)Error occurs when a reference refers to a specific field
// (method) of a specific field or interface, but the class or interface does 
// not declare a field of that name (it is specifically not sufficient for it 
// simply to be an inherited field (method) of that class or interface).
//
//
enum Loader_Exception {
	LD_NoException,
	LD_NoClassDefFoundError,
	LD_ClassFormatError,
	LD_ClassCircularityError,
	LD_IncompatibleClassChangeError,
	LD_AbstractMethodError,		// occurs during preparation
	LD_IllegalAccessError,
	LD_InstantiationError,
	LD_NoSuchFieldError,
	LD_NoSuchMethodError,
	LD_UnsatisfiedLinkError,
	LD_VerifyError
};

#endif // for Linux, move to Loader_Result.h


///////////////////////////////////////////////////////////////////////////////
// virtual method table of a class
///////////////////////////////////////////////////////////////////////////////
extern "C" {

typedef struct {
	unsigned char	**table;// pointer into methods array of VTable below
	unsigned intfc_id;		// id of interface
} Intfc_Table_Entry;

#ifdef GC_REWORK
typedef struct Intfc_Table  {
#ifdef POINTER64
    // see INTFC_TABLE_OVERHEAD
	uint32 dummy;   // padding
#endif
	uint32 n_entries;
	Intfc_Table_Entry entry[1];
} Intfc_Table;
#else
typedef struct Intfc_Table {
#ifdef POINTER64
    // see INTFC_TABLE_OVERHEAD
	uint32 dummy;   // padding
#endif
	uint32 n_entries;
	Intfc_Table_Entry entry[1];
} Intfc_Table;
#endif

#define INTFC_TABLE_OVERHEAD	(sizeof(void *))


extern Intfc_Table *create_intfc_table(unsigned n_entries);

#ifdef GC_REWORK
// This definition has been moved to orp_types.h
#else
/////////////////////////////////////////////////////////////////////
//////// The first field can not be changed without reflecting
//////// the changes in the type Partial_VTable typedef in orp_for_gc.h
////////////////////////////////////////////////////////////////////

typedef struct VTable {
	Class			*clss;          // the class - see above before change
	Intfc_Table		*intfc_table;	// interface table; NULL if no intfc table
//#ifdef FAST_INSTOF
	Class *superclasses[MAX_FAST_INSTOF_DEPTH]; //::
//#endif
	unsigned char	*methods[1];	// code for method
} VTable;

#endif // GC_REWORK

#define VTABLE_OVERHEAD	(sizeof(VTable) - sizeof(void *)) 
//::(2 * sizeof(void *))

// extern VTable *create_vtable(unsigned n_vtable_entries);
extern VTable *create_vtable(Class *p_class, unsigned n_vtable_entries);

};
///////////////////////////////////////////////////////////////////////////////
// A Java class
///////////////////////////////////////////////////////////////////////////////
extern "C" {

//
// state of this class
//
enum Class_State {
	ST_Start,				// initial state
	ST_LoadingAncestors,	// loading super class and super interfaces
	ST_Loaded,				// successfully loaded
	ST_Prepared,			// successfully prepared
	ST_Initializing,		// initializing the class
	ST_Initialized,			// class initialized
    ST_Error,               // bad class or the initializer failed
};

typedef union {
		String *name;
		Class  *clss;
} Class_Superinterface;



typedef struct Registered_Native {
    String *name;
    String *sig;
    void   *func;
    Registered_Native *next;
} Registered_Native;



typedef struct Class {

/////////////////////////////////////////////////////////////////////
//////// The first few fields can not be changed without reflecting
//////// the changes in the type Partial_Class in orp_for_gc.h
////////////////////////////////////////////////////////////////////

    VTable  *p_vtable;  // NOTE: this field must be first since it points to vtable 

#ifndef OLD_OBJ_LAYOUT
    POINTER_SIZE_INT obj_info;
#endif
	Java_java_lang_Object *signers_field_located_in_java_lang_class_in_gnu_classpath;


	//
	// super class of this class; initiially, it is the string name of super
	// class; after super class is loaded, it becomes a pointer to class
	// structure of the super class.
    //
    // The offset of this field is returned by class_get_super_offset.
    // Make sure to update this function if the field is moved around.
	//
	union {
		String  *super_name;
		Class	*super_class;
	};


    //
    // See the masks in orp_for_gc.h.
    //

    uint32 class_properties;


 

    // This typically holds an array of offsets to the pointers in 
    // and instance of this class. What is here is totally up to the
    // garbage collector side of the interface.
    unsigned int *gc_information;

    // Offset from the top by CLASS_ALLOCATED_SIZE_OFFSET
    // The number of bytes allocated for this object. It is the same as
    // instance_data_size with the constraint bit cleared. This includes
    // the OBJECT_HEADER_SIZE as well as the OBJECT_VTABLE_POINTER_SIZE
    unsigned int allocated_size;

    unsigned int array_element_size;
 

/////////////////////////////////////////////////////////////////////
//////// Fields above this point can not be moved without redefining
//////// the data structure Partial_Class in orp_for_gc.h
/////////////////////////////////////////////////////////////////////

    //
    // How should objects of this class be aligned by GC.
    //
    int alignment;


	//
	// unique class id
	//
	unsigned id;


    //
    // The class loader used to load this class.
    // Null for the system class loader.
    //
    Class_Loader *class_loader;


    //
    // Does it represent a primitive type?
    //
	unsigned is_primitive : 1;

	//
	// array information
	//
	unsigned is_array : 1;
	unsigned is_array_of_primitives : 1;


    //
    // Does the class have a finalizer that is not inherited from
    // java.lang.Object?
    //
	unsigned has_finalizer : 1;

	//
	// Is this a soft, weak or phantom reference.
	//
	unsigned is_soft_ref : 1;
	unsigned is_weak_ref : 1;
	unsigned is_phantom_ref : 1;


#ifdef CLI_TESTING
    Boolean is_value_type : 1;
    Boolean is_enum : 1;
#endif

	//
	// number of dimensions in array; current ORP limitation is 255
	//
	// Note, that you can derive the base component type of the array 
	// by looking at name->bytes[n_dimensions].
	//
	unsigned char n_dimensions;


	//
	// for non-primitive arrays only, array_base_class is the base class 
	// of an array
	//
	Class *array_base_class;


	Class *array_element_class;

	uint16 access_flags;
	uint16 cp_size;
	uint16 n_superinterfaces;
	uint16 n_fields;
	uint16 n_methods;

    // for inner class support	
	uint16 declaringclass_index;
	uint16 n_innerclasses;
	uint16 *innerclass_indexes;
	
	Const_Pool *const_pool;	// constant pool array; size is _cp_size
	Field	*fields;			// array of fields; size is _n_fields
	Method	*methods;			// array of methods; size is _n_methods
	//
	// array of interfaces this class implements; size is _n_superinterfaces
	// initially, it is an array of string names of interfaces and then
	// after superinterfaces are loaded, this becomes an array pointers
	// to superinterface class structures.
	//
	Class_Superinterface *superinterfaces;

	const String	*name;		// string name of this name
	const String	*class_file_name;   // string name of file from which 
                                        // this class has been loaded
	const String	*src_file_name;     // string name of file from which 
                                        // this class has been compiled
	Class_State   state;		// state of this class
	Class	*table_next;		// link in class table

	Package	*package;			// package to which this class belongs
	Class	*next_class_in_package;	// link for list of classes in a package
	//
	// the following sizes are all in bytes
	//

    unsigned n_instance_refs;  // number of instance variables that are references RDS
//#if _DEBUG
    unsigned n_static_fields;  // number of vars that are not refs. RDS
//#endif // _DEBUG

    unsigned n_virtual_method_entries;	// number virtual methods in vtable
	unsigned n_intfc_method_entries;	// number interface methods in vtable
	unsigned n_intfc_table_entries;	    // number intfc tables in _intfc_table
										// same as _vtable->_intfc_table->n_entries

	Method	*finalize_method;			// NULL if none exists

	Method	*static_initializer;		// if it exists, NULL otherwise

	unsigned static_data_size;		// size of this class' static data block
	void *static_data_block;	// block containing array of static data fields
#ifdef NEW_DATA_ALLOC
    JavaArrayOfBoolean *static_fields_boolean;
    JavaArrayOfByte    *static_fields_byte;
    JavaArrayOfChar    *static_fields_char;
    JavaArrayOfShort   *static_fields_short;
    JavaArrayOfInt     *static_fields_int;
    JavaArrayOfLong    *static_fields_long;
    JavaArrayOfFloat   *static_fields_float;
    JavaArrayOfDouble  *static_fields_double;
    JavaArrayOfObject  *static_fields_ref;
#endif

	unsigned static_method_size;	// size in bytes of this class' static method block
	unsigned char **static_method_block;	// array of pointers to code of
											// static methods

 
    // This is the size of an instance without any alignment padding. 
    // It can be used while calculating the field offsets of subclasses.
    // It does not include the OBJECT_HEADER_SIZE but does include the
    // OBJECT_VTABLE_POINTER_SIZE.
    // The allocated_size field will be this field properly aligned.

    unsigned unpadded_instance_data_size;

    //
    // Try to keep instance_data_size near vtable since the are used at the same time
    // by the allocation routines and sharing a cache line seem to help.
    //
    // The next to high bit is set if allocation needs to consider class_properties.
    // (mumble->instance_data_size & NEXT_TO_HIGH_BIT_CLEAR_MASK) will always return the
    // actual size of and instance of class mumble. 
    // Use get_instance_data_size() to get the actual size of an instance.
    // Use set_instance_data_size_constraint_bit() to set this bit.

    unsigned instance_data_size;	// For most classes the size of a class instance's 
                                    // data block. This is what is passed to the GC.
                                    // See above for details.
	VTable	*vtable;			// virtual method table; NULL for interfaces



	//
	// _vtable_descriptor is an array of pointers to Method descriptors, one
	// descriptor for each corresponding entry in _vtable.methods[].
	//
	Method	**vtable_descriptors;
	//
	// _intfc_table_descriptor is an array of pointers to Class descriptors,
	// one descriptor for each corresponding entry in intf_table.entry[];
	//
	Class	**intfc_table_descriptors;		// Class struture of interface
    void *class_object;

#ifdef _DEBUG
    Boolean printed_in_dump_jit;
#endif

    void *p_initializing_thread;        // this really points to OS thread data struct
    void *p_initializer_error_object;   // this really points to a java object

#ifdef ORP_STATS
    uint64 num_class_init_checks;

    // For subclasses of java.lang.Throwable only.
    uint64 num_throws;

    // Number of instanceof/checkcast calls both from the user code
    // and the VM that were not subsumed by the fast version of instanceof.
    uint64 num_instanceof_slow;
#endif

    Registered_Native *registered_natives;

//#ifdef FAST_INSTOF
	int depth;
	//::Class *superclasses[MAX_FAST_INSTOF_DEPTH];
//#endif

#ifdef CLI_TESTING
    CLI_TypeDef *cli_type_def;
    CLI_Type    *super_type;
#ifdef CLI_OCL
	// Field to access runtime System.Type object to enable CLI reflection
	// void * for now; will need to include header file neatly to code actual type.
	void *system_type;
#endif // CLI_OCL
#endif // CLI_TESTING
} Class;


};


//
// access modifiers
//
#define class_is_public(clss)       ((clss)->access_flags & ACC_PUBLIC)
#define class_is_final(clss)        ((clss)->access_flags & ACC_FINAL)
#define class_is_super(clss)        ((clss)->access_flags & ACC_SUPER)
#define class_is_interface(clss)    ((clss)->access_flags & ACC_INTERFACE)
#define class_is_abstract(clss)     ((clss)->access_flags & ACC_ABSTRACT)

Class *new_class(const String *name);



//
// Look up of methods and fields in class.
// Functions with the "_recursive" suffix also check superclasses.
//
ORPExport Field *class_lookup_field(Class *clss, Signature *sig);
ORPExport Field *class_lookup_field_recursive(Class *clss, const char *name, const char *descr);
ORPExport Method *class_lookup_method(Class *clss, Signature *sig);
ORPExport Method *class_lookup_method_recursive(Class *clss, Signature *sig);
ORPExport Method *class_lookup_method_recursive(Class *clss, const char *name, const char *descr);

ORPExport Java_java_lang_Object * class_alloc_new_object(Class *c);

Java_Type class_get_cp_const_type(Class *clss, unsigned cp_index);
const void *class_get_addr_of_constant(Class *clss, unsigned cp_index);
Field *class_resolve_static_field(Class *clss, unsigned cp_index);
Field *class_resolve_nonstatic_field(Class *clss, unsigned cp_index);
Method *class_resolve_static_method(Class *clss, unsigned cp_index);
Method *class_resolve_virtual_method(Class *clss, unsigned cp_index);
Method *class_resolve_special_method(Class *clss, unsigned cp_index);

Method *class_resolve_nonstatic_method(Class *clss, unsigned cp_index);
Method *class_resolve_interface_method(Class *clss, unsigned cp_index);
Class *class_resolve_class(Class *clss, unsigned cp_index);

Class *resolve_class_array_of_class(Global_Env *env,
                                    Class *cc,
                                    Loader_Exception *exc);


//
// parses in class description from a class file format
//
Loader_Result class_parse(Class *clss,
                          ByteReader &cfs,
                          Global_Env& env,
                          unsigned *super_class_cp_index);


//
// preparation phase of class loading
//
ORPExport Loader_Result class_prepare(Class *clss);
#ifdef CLI_TESTING
ORPExport Loader_Result cli_class_prepare(Class *clss);
#endif



//
// Load a class and perform the first two parts of the link process: verify
// and prepare.  The last stage of linking, resolution, is done at JIT-time.
// See the ORP spec 2.16.3.
//
ORPExport Class *class_load_verify_prepare(Global_Env *env,
                                           const String *classname,
                                           Loader_Exception *exception);


ORPExport Class *class_load_verify_prepare_by_loader(Global_Env *env,
                                                     const String *classname,
                                                     Class_Loader *cl,
                                                     Loader_Exception *exception);

ORPExport Class *class_load_verify_prepare_from_jni(Global_Env *env,
                                                     const String *classname,
                                                     Loader_Exception *exception);

//
// execute static initializer of class
//

ORPExport void class_initialize1(Class *clss, bool throw_exception);
ORPExport void class_initialize_from_jni(Class *clss, bool throw_exception);
ORPExport void class_initialize(Class *clss);
#ifdef CLI_TESTING
ORPExport void cli_class_initialize(Class *clss);
#endif




///////////////////////////////////////////////////////////////////////////////
// A class' members are its fields and methods.  Class_Member is the base
// class for Field and Method, and factors out the commonalities in these
// two classes.
///////////////////////////////////////////////////////////////////////////////
class Class_Member {
public:
	//
	// access modifiers
	//
	unsigned is_public()		{return (_access_flags&ACC_PUBLIC);} 
	unsigned is_private()		{return (_access_flags&ACC_PRIVATE);} 
	unsigned is_protected()		{return (_access_flags&ACC_PROTECTED);} 
	unsigned is_static()		{return (_access_flags&ACC_STATIC);} 
	unsigned is_final()			{return (_access_flags&ACC_FINAL);} 
	unsigned get_access_flags()	{return _access_flags;}


#ifdef CLI_TESTING
    bool is_cli() { return _is_cli; };
#endif

	//
	// field get/set methods
	//
	unsigned      get_offset()     {return _offset;}

	Class        *get_class()      {return _class;}

    const String *get_name()       {return _signature->name;}

    // Get the ORP type descriptor (Sec. 4.3.2)
    const char   *get_descriptor() {return _signature->descriptor->bytes;}

	Signature    *get_signature()  {return _signature;}


	//
	// friends: class_prepare touches offset
	//
	friend ORPExport Loader_Result class_prepare(Class*);

    // This function examines offsets in order to process field access data.
	friend void assign_offsets_to_class_fields(Class *);
	friend void build_gc_class_ref_map(Class *);
	friend void intialize_static_fields_for_interface(Class *);

protected:
	Class_Member() 
	{
		_access_flags=0;_signature=NULL;_class=NULL;_offset=0;
#ifdef CLI_TESTING
        _is_cli = false;
        _cli_member = 0;
#endif
#ifdef ORP_STATS
        num_accesses=0;
#endif
	}


#ifdef NEW_DATA_ALLOC
    // add a new comment
	unsigned _offset;

    void *_static_field_block; // for static fields only
#else
	//
	// offset of class member; 
	// for virtual methods, this is the method's offset within the vtable
	// for static methods, this is the method's offset within the class' static
	// method table
	// for instance data, this is the offset within the instance's data block
	// for static data, this is the offset withing the class' static data block
	unsigned _offset;
#endif


	uint16 _access_flags;
	Signature	*_signature;
	Class		*_class;
    Loader_Result parse(ByteReader &cfs,
                        Const_Pool *cp,
                        unsigned cp_size,
                        Signature_Table& sig_table,
                        Class *clss);

#ifdef CLI_TESTING
    bool _is_cli;
    CLI_FieldOrMethod *_cli_member;
#endif

#ifdef ORP_STATS
public:
    uint64 num_accesses;
#endif
}; //Class_Member



///////////////////////////////////////////////////////////////////////////////
// Fields within Class structures.
///////////////////////////////////////////////////////////////////////////////
class Field : public Class_Member{
public:
    //-----------------------

    // For static fields
    ORPExport void *get_address();

    // Return the type of this field.
    Java_Type get_java_type() {
        return (Java_Type)(get_signature()->descriptor->bytes[0]); 
    };

    Const_Java_Value get_const_value() { return const_value; };
    uint16 get_const_value_index() { return _const_value_index; };

    //-----------------------

	Field() {
        _const_value_index = 0;
    }
	void set(unsigned short af, Signature *sig, Class *cl) {
		_access_flags = af; _signature = sig; _class = cl;
	}
	//
	// access modifiers
	//
	unsigned is_volatile()	{return (_access_flags&ACC_VOLATILE);} 
	unsigned is_transient()	{return (_access_flags&ACC_TRANSIENT);} 
 
    Loader_Result parse(ByteReader &cfs,
                        Const_Pool *cp,
                        unsigned cp_size,
                        Signature_Table& sig_table,
                        Class *clss);

#ifdef CLI_TESTING
    Loader_Result cli_parse(Class *clss, CLI_Field *cli_field, Global_Env *env, Loader_Exception *exception);
    CLI_Field *get_cli_field();
#endif

private:

    //
    // The initial values of static fields.  This is defined by the 
    // ConstantValue attribute in the class file.  
    //
    // If there was not ConstantValue attribute for that field then _const_value_index==0
    //
    uint16 _const_value_index;
    Const_Java_Value const_value;
}; // Field



///////////////////////////////////////////////////////////////////////////////
// Handler represents a catch block in a method's code array
///////////////////////////////////////////////////////////////////////////////
class Handler {
public:
	Handler();
	Loader_Result parse(ByteReader &cfs,
                        unsigned code_length,
                        Const_Pool *cp,
                        unsigned cp_size);
    uint32 get_start_pc() {return _start_pc;}
    uint32 get_end_pc() {return _end_pc;}
    uint32 get_handler_pc() {return _handler_pc;}
    uint32 get_catch_type_index() {return _catch_type_index;}

#ifdef CLI_TESTING
    Loader_Result cli_parse(CLI_Exception_Clause *e, Global_Env *env, Loader_Exception *exception);
    enum Kind {
        Handler_Filter  = 1,
        Handler_Finally = 2,
        Handler_Fault   = 4
    };
    Kind get_kind() {return _kind;}
    uint32 get_handler_len() {return _handler_len;}
    uint32 get_filter_pc() {return _filter_pc;}
#endif

private:
	uint32 _start_pc;
	uint32 _end_pc;
	uint32 _handler_pc;
	uint32 _catch_type_index;  // CP idx
	String *_catch_type;

#ifdef CLI_TESTING
    bool _is_cli;                        // for debugging only???
    CLI_Exception_Clause *_cli_handler;  // for debugging only
	uint32 _handler_len;
	uint32 _filter_pc;
    Kind _kind;
#endif
}; //Handler



// Representation of target handlers in the generated code.
class Target_Exception_Handler {
public:
	Target_Exception_Handler(void  *start_ip,
                             void  *end_ip,
                             void  *handler_ip,
                             Class *exc,
                             bool   exc_obj_is_dead
                             ) {
        _start_ip        = start_ip;
        _end_ip          = end_ip;
        _handler_ip      = handler_ip;
        _exc             = exc;
        _exc_obj_is_dead = exc_obj_is_dead;
    };

    void *get_start_ip()   {return _start_ip;}
    void *get_end_ip()     {return _end_ip;}
    void *get_handler_ip() {return _handler_ip;}
    Class *get_exc()       {return _exc;}
    bool is_exc_obj_dead() {return _exc_obj_is_dead;}

    Boolean is_in_range(void *eip, Boolean is_first);
    Boolean is_assignable(Class *exc);

private:
    void *_start_ip;
    void *_end_ip;
    void *_handler_ip;
    Class *_exc;
    bool _exc_obj_is_dead;
}; //Target_Exception_Handler

typedef Target_Exception_Handler *Target_Exception_Handler_Ptr;



///////////////////////////////////////////////////////////////////////////////
// Line number entry for method's code array; the set of line numbers of a 
// of a method are linked together in a list.  
// Note, that line number entries are debugging information and thus optional.
///////////////////////////////////////////////////////////////////////////////
class Line_Number_Entry {
public:
	unsigned short start_pc;
	unsigned short line_number;
	Line_Number_Entry *next;
	Line_Number_Entry(unsigned short p,unsigned short l,Line_Number_Entry *n)
		: start_pc(p), line_number(l), next(n) {}
};



struct Line_Number_Table_Entry {
    uint16 start_pc;
    uint16 line_number;
#ifdef STACK_TRACE_ADDITIONAL_INFO
	uint16 npc_offset;
#endif
};


struct Line_Number_Table {
    uint16 length;
    Line_Number_Table_Entry table[1];
};


///////////////////////////////////////////////////////////////////////////////
// Local variable debugging entry for method.  The set of local variable 
// entries of a method are linked together in a list.
// Note, that local var entries are debugging information and thus optional.
///////////////////////////////////////////////////////////////////////////////
class Local_Var_Entry {
public:
	Signature *sig;		// variables signature (name and type strings)
	unsigned short start_pc;
	unsigned short length;
	Local_Var_Entry *next;
	Local_Var_Entry(Signature *s,unsigned short p,unsigned short l,
		            Local_Var_Entry *x) :
	sig(s), start_pc(p), length(l), next(x) {}
};



#define MAX_VTABLE_PATCH_ENTRIES 10

class VTable_Patches {
public:
    void *patch_table[MAX_VTABLE_PATCH_ENTRIES];
    VTable_Patches *next;
};



/////////////////////////////////////////////////////////////////
// begin multiple-JIT support

struct JIT_Data_Block {
    JIT_Data_Block *next;
    char bytes[1];
};



class JIT_Specific_Info {

friend class Method;

public:
    JIT_Specific_Info();

    void set_jit(JIT *jit) { _jit = jit; }
    JIT *get_jit() { return _jit; }

    void set_method(Method *m) { _method = m; }
    Method *get_method() { return _method; }

    void *get_code_block_addr() { return _code_block; }
    unsigned get_code_block_size() { return _code_block_size; }

    void set_rw_data_block(void *d) { _rw_data_block = d; }
    void *get_rw_data_block() { return _rw_data_block; }

    unsigned get_num_target_exception_handlers()
    {
        return _num_target_exception_handlers;
    } //get_num_target_exception_handlers

    Target_Exception_Handler_Ptr get_target_exception_handler_info(unsigned eh_num)
    {
        return _target_exception_handlers[eh_num];
    } //get_target_exception_handler_info

private:
    JIT *_jit;
    Method *_method;

    // "Target" handlers.
    unsigned _num_target_exception_handlers;
    Target_Exception_Handler_Ptr *_target_exception_handlers;

public:
    void *_code_block;
    void *_rw_data_block;
    void *_jit_info_block;
    unsigned _code_block_size;
    unsigned _rw_data_block_size;
    unsigned _jit_info_block_size;
    JIT_Data_Block *_data_blocks;
    JIT_Specific_Info *_next;



#ifdef ORP_VTUNE_SUPPORT
public:
    void set_loaded_for_vtune(bool v) { _has_been_loaded_for_vtune = v; };
    bool get_loaded_for_vtune() { return _has_been_loaded_for_vtune; };
private:
    bool _has_been_loaded_for_vtune;
#endif


#ifdef ORP_STATS
public:
    uint64 num_throws;
    uint64 num_catches;
    uint64 num_unwind_java_frames_gc;
    uint64 num_unwind_java_frames_non_gc;
#endif
};

// end multiple-JIT support
/////////////////////////////////////////////////////////////////


struct Inline_Record;



///////////////////////////////////////////////////////////////////////////////
// Methods defined in a class.
///////////////////////////////////////////////////////////////////////////////
class Method : public Class_Member {
    //-----------------------
public:
	//
	// state of this method
	//
	enum State {
		ST_NotCompiled,				// initial state
        ST_BeingCompiled,           // jitting
        ST_Compiled
	};
	State get_state() {return _state;}
	void set_state(State st) {_state=st;}

    // "Bytecode" exception handlers, i.e., those from the class file
    unsigned num_bc_exception_handlers();
    Handler *get_bc_exception_handler_info(unsigned eh_number);

    // "Target" exception handlers, i.e., those in the code generated by the JIT.
    void set_num_target_exception_handlers(JIT *jit, unsigned n);
    unsigned get_num_target_exception_handlers(JIT *jit);


    //
    // Arguments:
    //  ...
    //  catch_clss  -- class of the exception or null (for "catch-all")
    //  ...
    //
    void set_target_exception_handler_info(JIT *jit,
                                           unsigned eh_number,
                                           void *start_ip,
                                           void *end_ip,
                                           void *handler_ip,
                                           Class *catch_clss,
                                           bool exc_obj_is_dead = false
                                           );

    Target_Exception_Handler_Ptr get_target_exception_handler_info(JIT *jit,
                                                                   unsigned eh_num);


    ORPExport unsigned num_exceptions_method_can_throw();
    ORPExport String *get_exception_name (int n);

    // Address of the memory block containing bytecodes.  For best performance
    // the bytecodes should not be destroyed even after the method has been
    // jitted to allow re-compilation.  However the interface allows for such
    // deallocation.  The effect would be that re-optimizing JITs would not
    // show their full potential, but that may be acceptable for low-end systems
    // where memory is at a premium.
    // The value returned by getByteCodeAddr may be NULL in which case the
    // bytecodes are not available (presumably they have been garbage collected
    // by the ORP).
    const Byte  *get_byte_code_addr() {return _byte_codes;}
    size_t       get_byte_code_size() {return _byte_code_length;}

    // From the class file (Sec. 4.7.4)
    unsigned get_max_stack() {return _max_stack;}
    unsigned get_max_locals() {return _max_locals;}

    // Returns an iterator for the argument list.
    Arg_List_Iterator get_argument_list();

    // Returns number of bytes of arguments pushed on the stack.
    // This value depends on the descriptor and the calling convention.
    unsigned get_num_arg_bytes();


    // Returns number of arguments.  For non-static methods, the this pointer
    // is included in this number
    unsigned get_num_args();


    // Number of arguments which are references.
    unsigned get_num_ref_args();

#ifdef CLI_TESTING
    char * get_parameter_java_type(Arg_List_Iterator it);

    void *get_unboxer_addr()                 { return unboxer; }
    void set_unboxer_addr(void *unboxer_addr)   { unboxer = unboxer_addr; }
#endif //CLI_TESTING

    // Return the return type of this method.
    Java_Type get_return_java_type();

    // For non-primitive types (i.e., classes) get the class type information.
    Class *get_return_class_type();

    // Address of the memory location containing the address of the code.
    // Used for static and special methods which have been resolved but not jitted.
    // The call would be:
    //      call dword ptr [addr]
    void *get_indirect_address()          { return &_code; }

    // Entry address of the method.  Points to an appropriate stub or directly
    // to the code if no stub is necessary.
    void *get_code_addr()                 { return _code; }
    void set_code_addr(void *code_addr)   { _code = code_addr; }

    void add_vtable_patch(void *);
    void apply_vtable_patches();

    //
    // Only one of each blocks can be allocated per method.
    //

    // Code block is R/W?
    // This contains jitted code.  Not used for native methods.
    void *allocate_code_block(size_t size, JIT *jit);
    void *get_code_block_addr() {return _code_block;}
    unsigned get_code_block_size() {return _code_block_size;}

    // Read/Write data block.
    void *allocate_rw_data_block(size_t size, JIT *jit);

    // The JIT can store some information in a JavaMethod object.
    void *allocate_jit_info_block(size_t size, JIT *jit);

    // Return the memory block allocated with allocateJITInfoBlock.
    inline void *get_jit_info_block() { return _jit_info_block; };
    unsigned get_jit_info_block_size() { return _jit_info_block_size; }

    void set_jit(JIT *jit);
    JIT *get_jit() { return _jit; };

    // JIT-specific data blocks.
    void *allocate_JIT_data_block(size_t size, JIT *jit);
    void *get_first_JIT_data_block(JIT *jit);
    void *get_first_JIT_data_block(void *info_block);
    static void *get_next_JIT_data_block(void *block);

    JIT_Specific_Info *get_first_JIT_specific_info() { return _jits; };
    JIT_Specific_Info *get_JIT_specific_info(JIT *jit);
    JIT_Specific_Info *get_JIT_specific_info_no_create(JIT *jit);

    Method_Side_Effects get_side_effects() { return _side_effects; };
    void set_side_effects(Method_Side_Effects mse) { _side_effects = mse; };

private:

    State _state;
    void *_code;
    void *_compile_me_stub;
    VTable_Patches *_vtable_patch;
    void *_code_block;
#ifdef CLI_TESTING
    void *unboxer;
#endif //CLI_TESTING
    void *_jit_info_block;
    unsigned _code_block_size;
    unsigned _jit_info_block_size;

    JIT_Specific_Info *_jits;

    JIT *_jit;

    Method_Side_Effects _side_effects;

    //-----------------------
public:

	Method();
	//
	// access modifiers
	//
	unsigned is_synchronized()  {return (_access_flags&ACC_SYNCHRONIZED);} 
	unsigned is_native()        {return (_access_flags&ACC_NATIVE);} 
	unsigned is_abstract()      {return (_access_flags&ACC_ABSTRACT);} 
	unsigned is_init()          {return _flags.is_init;}
	unsigned is_clinit()        {return _flags.is_clinit;}
	unsigned is_finalize()      {return _flags.is_finalize;}
	unsigned is_overridden()    {return _flags.is_overridden;}
	Boolean is_nop()            {return _flags.is_nop;}

    unsigned get_index()     {return _index;}

    Loader_Result parse(ByteReader &cfs,
                        Const_Pool *cp,
                        unsigned cp_size,
                        Global_Env& env,
                        Class *clss);
#ifdef CLI_TESTING
    Loader_Result cli_parse(Class *clss, CLI_Method *cli_method, Global_Env *env, Loader_Exception *exception);
    CLI_Method *get_cli_method();
#endif

	//
	// friends: class_prepare touches index and _flags.is_overridden
	//
	friend ORPExport Loader_Result class_prepare(Class *);
	friend void assign_offsets_to_class_methods(Class *clss);



private:
	unsigned _index;				// index in method table
	uint16 _max_stack;
	uint16 _max_locals;
	uint16 _n_exceptions;	// num exceptions method can throw
	uint16 _n_handlers;		// num exception handlers in byte codes
	String	**_exceptions;			// array of exceptions method can throw
	uint32 _byte_code_length;		// num bytes of byte code
	Byte *_byte_codes;		// method's byte codes
	Handler	*_handlers;				// array of exception handlers in code
	struct {
		unsigned is_init        : 1;
		unsigned is_clinit      : 1;
		unsigned is_finalize    : 1;	// is finalize() method
		unsigned is_overridden  : 1;	// has this virtual method been 
										// overridden by a loaded subclass?
		unsigned is_nop         : 1;
	} _flags;


	//
	// private methods for parsing methods
	//
	Loader_Result _parse_code(ByteReader &cfs,
							  Const_Pool *cp,
							  unsigned cp_size,
							  Signature_Table& sig_table);

	Loader_Result _parse_exceptions(ByteReader &cfs,
									Const_Pool *cp,
									unsigned cp_size,
                                    unsigned attr_len);

    void _set_nop();
public:

    //
	// debugging info
	//
    Line_Number_Entry *get_line_number_entries() { return _line_number_entries; };
    Line_Number_Table   *_line_number_table;
	Line_Number_Entry	*_line_number_entries;	// list of line number entries
	Local_Var_Entry		**_local_vars_table;	// list of local var entries


    Inline_Record *inline_records;
    void set_inline_assumption(JIT *jit, Method *caller);
    void method_was_overridden();
}; //Method



///////////////////////////////////////////////////////////////////////////////
// class file attributes
///////////////////////////////////////////////////////////////////////////////
enum Attributes {
	ATTR_SourceFile,			// Class (no more than 1 in each class file)
	ATTR_InnerClasses,          // inner class
	ATTR_ConstantValue,			// Field (no more than 1 for each field)
	ATTR_Code,					// Method
	ATTR_Exceptions,			// Method
	ATTR_LineNumberTable,		// Code
	ATTR_LocalVariableTable,	// Code
	N_ATTR,
	ATTR_UNDEF,
	ATTR_ERROR
};

#define N_FIELD_ATTR	1
#define N_METHOD_ATTR	2
#define N_CODE_ATTR		2
#define N_CLASS_ATTR	2

#if 0
case ATTR_SourceFile:
case ATTR_InnerClasses: 
case ATTR_ConstantValue:
case ATTR_Code:
case ATTR_Exceptions:
case ATTR_LineNumberTable:
case ATTR_LocalVariableTable:
case ATTR_UNDEF:
case ATTR_ERROR:
#endif

///////////////////////////////////////////////////////////////////////////////
// Table of class' loaded by the class loader.
///////////////////////////////////////////////////////////////////////////////
class Class_Table {
public:
	Class_Table();
	Class *lookup(const String *name);
	void insert(Class *clss);
	void remove(Class *clss);
	Class **get_first_class();
	Class **get_next_class(Class *);
private:
	Class *_remove(Class *curr,Class *elem);

#define CLASS_TABLE_SIZE 1024

	Class *_table[CLASS_TABLE_SIZE];
};

//
// magic number, and major/minor version numbers of class file
//
#define	CLASSFILE_MAGIC	0xCAFEBABE
#define CLASSFILE_MAJOR	45
#define CLASSFILE_MINOR	3


ORPExport Class *load_class(Global_Env *env,
				  const String *classname,
				  Loader_Exception *exception);


Class *load_class_by_system_loader(Global_Env *env,
                                   const String *classname,
                                   Loader_Exception *exception);

ORPExport Class *load_class_by_loader(Global_Env *env,
                                      String *classname,
                                      Class_Loader *cl,
                                      Loader_Exception *exception);

Class *find_loaded_class(Global_Env *env,
				         const String *classname,
				         Loader_Exception *exception);


 
#define BITS_PER_BYTE 8
// We want to use signed arithmetic when we do allocation pointer/limit compares.
// In order to do this all sizes must be positive so when we want to overflow instead of 
// setting the high bit we set the next to high bit. If we set the high bit and the allocation buffer
// is at the top of memory we might not detect an overflow the unsigned overflow would produce
// a small positive number that is smaller then the limit.

#define NEXT_TO_HIGH_BIT_SET_MASK (1<<((sizeof(unsigned) * BITS_PER_BYTE)-2))
#define NEXT_TO_HIGH_BIT_CLEAR_MASK ~NEXT_TO_HIGH_BIT_SET_MASK

inline unsigned int get_instance_data_size (Class *c) 
{
//    assert ((c->instance_data_size & HIGH_BIT_CLEAR_MASK) == c->allocated_size);
    return (c->instance_data_size & NEXT_TO_HIGH_BIT_CLEAR_MASK);
}

inline void set_instance_data_size_constraint_bit (Class *c)
{
    c->instance_data_size = (c->instance_data_size | NEXT_TO_HIGH_BIT_SET_MASK);
}

// Setter functions for the class property field.
inline void set_prop_alignment_mask (Class *c, unsigned int the_mask)
{
    c->class_properties = (c->class_properties | the_mask);
#ifdef GC_REWORK
    c->vtable->class_properties = c->class_properties;
#endif
    set_instance_data_size_constraint_bit (c);
}
inline void set_prop_non_ref_array (Class *c)
{
    c->class_properties = (c->class_properties | CL_PROP_NON_REF_ARRAY_MASK);
#ifdef GC_REWORK
    c->vtable->class_properties = c->class_properties;
#endif
}
inline void set_prop_array (Class *c)
{
    c->class_properties = (c->class_properties | CL_PROP_ARRAY_MASK);
#ifdef GC_REWORK
    c->vtable->class_properties = c->class_properties;
#endif
}
inline void set_prop_pinned (Class *c)
{
    c->class_properties = (c->class_properties | CL_PROP_PINNED_MASK);
#ifdef GC_REWORK    
    c->vtable->class_properties = c->class_properties;
#endif
    set_instance_data_size_constraint_bit (c);
}
inline void set_prop_finalizable (Class *c)
{
    c->class_properties = (c->class_properties | CL_PROP_FINALIZABLE_MASK);

#ifdef GC_REWORK
    c->vtable->class_properties = c->class_properties;
#endif
    set_instance_data_size_constraint_bit (c);
}

// get functions for the class property field.
inline unsigned int get_prop_alignment (unsigned int class_properties)
{
    return (unsigned int)(class_properties & CL_PROP_ALIGNMENT_MASK);
}
inline unsigned int get_prop_non_ref_array (unsigned int class_properties)
{
    return (class_properties & CL_PROP_NON_REF_ARRAY_MASK);
}
inline unsigned int get_prop_array (unsigned int class_properties)
{
    return (class_properties & CL_PROP_ARRAY_MASK);
}
inline unsigned int get_prop_pinned (unsigned int class_properties)
{
    return (class_properties & CL_PROP_PINNED_MASK);
}
inline unsigned int get_prop_finalizable (unsigned int class_properties)
{
    return (class_properties & CL_PROP_FINALIZABLE_MASK);
}

#endif // _CLASS_H_
