
/*
	For the non-constructors and non-destructor function definitions, see rpggame.cpp
*/
#ifndef __RPGGAME_H__
#define __RPGGAME_H__

#include "cube.h"
#include "rpgcommand.h"

#define SPAWNNAMELEN 64

// Constants (placed here for easy access)
#define DEFAULTMODEL "rc"
#define DEFAULTMAP "title"

enum                            // static entity types
{
	NOTUSED = ET_EMPTY,         // entity slot not in use in map
	LIGHT = ET_LIGHT,           // lightsource, attr1 = radius, attr2 = intensity
	MAPMODEL = ET_MAPMODEL,     // attr1 = angle, attr2 = idx
	PLAYERSTART,                // attr1 = angle, attr2 = team
	ENVMAP = ET_ENVMAP,         // attr1 = radius
	PARTICLES = ET_PARTICLES,
	MAPSOUND = ET_SOUND,
	SPOTLIGHT = ET_SPOTLIGHT,
	SPAWN = ET_GAMESPECIFIC,
	TELEDEST, //attr1 = yaw, attr2 = from
	JUMPPAD, //attr1 = Z, attr2 = Y, attr3 = X, attr4 = radius
	CHECKPOINT,
	BLIP,
	MAXENTTYPES
};

struct rpgentity : extentity
{
    char name[SPAWNNAMELEN];

    rpgentity() { memset(name, 0, SPAWNNAMELEN); }
};

struct dynlightcache
{
	vec o, initcol, col;
	int flags, initrad, rad, fade, peak;

	dynlightcache(const vec &_o, float _rad, const vec &_col, int _fade = 0, int _peak = 0, int _flags = 0, float _initrad = 0, const vec &_initcol = vec(0,0,0)) :
	o(_o), initcol(_initcol), col(_col), flags(_flags), initrad(_initrad), rad(_rad), fade(_fade), peak(_peak)
	{}
	~dynlightcache() {};
};

enum
{
	PFX_TRAIL = 1,
	PFX_DYNLIGHT = 2,
	PFX_MAX = 4
};

struct projeffect
{
	//these respond to the particle's trail/projectile
	//TODO sounds

	const char *mdl;
	int basevel, kickback;
	int flags;

	int trailpart, trailcol, trailfade, gravity; //note for gravity, lower is faster
	float trailsize;

	int projpart, projcol;
	float projsize;

	int lightradius, lightcolour;

	int deathpart, deathpartcol, deathfade, deathlightinitcol, deathlightflags, deathdecal, deathlightfade;
	float deathpartsize;

	projeffect()
	{
		mdl = NULL;
		basevel = 200;
		kickback = 50;
		flags = PFX_TRAIL|PFX_DYNLIGHT;

		trailpart = PART_STEAM;
		trailcol = 0x7F7F7F;
		trailsize = 1.0f;
		trailfade = 500;
		gravity = 200;

		projpart = PART_FIREBALL1;
		projcol = 0xFFBF00;
		projsize = 4;

		lightradius = 64;
		lightcolour = 0xFFBF00;

		deathpart = PART_EXPLOSION;
		deathpartcol = 0xFFBF00;
		deathpartsize = 4.0f;
		deathfade = 800;
		deathdecal = DECAL_BURN;
		deathlightinitcol = lightcolour;
		deathlightflags = DL_EXPAND;
		deathlightfade = 200;
	}
	~projeffect()
	{
		if(mdl) delete[] mdl;
	}
};

struct rpgent;

struct projectile
{
	//TODO dedicated soundchannel
	rpgent *weapon; //the item it takes it's effect/damage stack from
	vec o, d, lastemit;
	int speed, radius;
	int effect, gravity; //ignore projpart, projcol and projsize if magic is false
	entitylight light;

	bool deleted;

	void addlight();
	void addtrail(bool &emitparts);

	projectile(rpgent *ent) : weapon(ent)
	{
		o = d = lastemit = vec(0, 0, 0);
		speed = 100;
		radius = 32;
		effect = 0, gravity = 0;
		deleted = false;
	}
	~projectile() {}
};

enum //status effects, both negative and positive
{
	STATUS_HEALTH = 0, //attributes
	STATUS_MOVE,
	STATUS_STRENGTH,
	STATUS_INTELLIGENCE,
	STATUS_CHARISMA,
	STATUS_ENDURANCE,
	STATUS_AGILITY,
	STATUS_LUCK,
	STATUS_CRIT,
	STATUS_HREGEN,
	STATUS_MREGEN,
	STATUS_FIRE, //these are resistances
	STATUS_WATER,
	STATUS_AIR,
	STATUS_EARTH,
	STATUS_MAGIC,
	STATUS_SLASH,
	STATUS_BLUNT,
	STATUS_PIERCE,
	STATUS_DISPELL, //negative strength dispells positive magics, positive strength bad magics
	STATUS_DOOM,
	STATUS_REFLECT,
	STATUS_INVIS,
	STATUS_LIGHT,
	STATUS_DEATH, //rather a status for withdrawal symptoms of death, as in, your stats are temporarily lowered due to your recent death
	/*Additional Spells worth considerings
	STATUS_STUN,
	STATUS_POLYMORPH, <-- this dispells all other polymorphs present
	*/
	STATUS_MAX
};



struct status
{
	int type, strength, applied, duration, startmillis, resists; //note applied is mostly for DoT attacks/healings resists is the applicable resistances, see the ATTACK* enum

	status(int t, int s, int d, int at = -1) : type(t), strength(s), startmillis(lastmillis), resists(at)
	{
		duration = max(d, 1);
		applied = 0;
	}
	~status() {};
};

//note, none-elemental attacks only use the respective magic/damage resistance
//otherwise if the attack is elemental, use a two step algorithm, eg, if both resistances are 25%, take 56.25% damage
//also note that elemental attacks will do 25% extra damage to compensate for having two resistances, so in the above case, 70.3125% damage will be taken
//note that a resistance is maxed at 80% which corresponds to 5% in elental attacks and 20% for regular attacks

enum
{
	RESIST_FIRE = 0,
	RESIST_WATER,
	RESIST_AIR,
	RESIST_EARTH,
	RESIST_MAGIC,
	RESIST_SLASH,
	RESIST_BLUNT,
	RESIST_PIERCE,
	/* Resistances under consideration

	RESIST_POISON

	*/
	RESIST_MAX
};

enum
{
	ATTACK_SLASHING = 0,
	ATTACK_BLUNT = 1,
	ATTACK_PIERCE = 2,
	ATTACK_MAGIC = 3,

	//keep a bit above the above, eg, if the above goes to ATTACK_* = 6, then use 8, the rest will automatically update the numbers
	ATTACK_FIRE = 4,
	ATTACK_WATER = ATTACK_FIRE<<1, //8
	ATTACK_AIR = ATTACK_FIRE|ATTACK_WATER, //12
	ATTACK_EARTH = ATTACK_FIRE<<2, //16

	/* Attack Types under consideration

	ATTACK_POISON = ATTACK_EARTH|ATTACK_FIRE,   //20

	*/
};

enum
{
	CROSS_TARGET = 0,
	CROSS_EDIT,
	CROSS_TALK,
	CROSS_PICKUP
	/*
	CROSS_SNEAK
	*/
};

//merge unarmed and blade -> melee?
//merge throw, marksman and gun -> range?
enum
{
	LEVEL_OVERALL = 0,
	LEVEL_MELEE,
	LEVEL_RANGED,
	LEVEL_REPAIR,
	LEVEL_HEAL,
	LEVEL_MAGIC, //break up into colleges?
	LEVEL_BARTER,
	LEVEL_SNEAK,
	LEVEL_MAX
};

struct stats
{
	//stats; max at 100
	int strength, intelligence, charisma, endurance, agility, luck;

	//skills
	struct level
	{
		int level, exp;
	} levels[LEVEL_MAX];

	//attributes
	int resistance[RESIST_MAX], maxhealth, maxmana, maxweight, movespeed, jumpvel;
	float healthregen, manaregen;

	void calcattrs();
	void limits();

	bool operator>= (const stats &s);
	bool operator< (const stats &s);
	bool operator== (const stats &s);
	bool operator!= (const stats &s);

	stats &add(int n);
	stats &add(const stats &s);
	stats &addall(int n);
	stats &addall(const stats &s);
	stats &sub(int n);
	stats &sub(const stats &s);
	stats &suball(int n);
	stats &suball(const stats &s);
	stats &div(float n);
	stats &divall(float n);
	stats &mul(float n);
	stats &mulall(float n);

	stats(bool empty = false)
	{
		if(!empty)
		{
			strength = intelligence = charisma = endurance = agility = luck = 20;
			loopi(LEVEL_MAX) {levels[i].level = 1; levels[i].exp = 0;}
			calcattrs();
		}
		else
		{
			strength = intelligence = charisma = endurance = agility = luck = maxhealth = healthregen = maxmana = manaregen = maxweight = movespeed = jumpvel = 0;
			loopi(LEVEL_MAX) {levels[i].level = 0; levels[i].exp = 0;}
			loopi(RESIST_MAX) resistance[i] = 0;
		}
	}
	~stats() {}
};

enum
{
	EQUIP_LHAND = 0,
	EQUIP_RHAND,
	EQUIP_FEET,
	EQUIP_LEGS,
	EQUIP_TORSO,
	EQUIP_MISC1,
	EQUIP_MISC2,
	EQUIP_HEAD,
	EQUIP_MAX
};

enum
{
	SLOT_LHAND = 1 << EQUIP_LHAND,
	SLOT_RHAND = 1 << EQUIP_RHAND,
	SLOT_FEET  = 1 << EQUIP_FEET,
	SLOT_LEGS  = 1 << EQUIP_LEGS,
	SLOT_TORSO = 1 << EQUIP_TORSO,
	SLOT_MISC1 = 1 << EQUIP_MISC1,
	SLOT_MISC2 = 1 << EQUIP_MISC2,
	SLOT_HEAD  = 1 << EQUIP_HEAD,
	SLOT_MAX   = 1 << EQUIP_MAX
};

struct response
{
	const char *talk;
	int dest;
	const char *script;

	response(const char *t, int d, const char *s) : talk(t), dest(d), script(s)
	{}
	~response()
	{
		delete [] talk;
		delete [] script;
	}
};

struct rpgchat
{
	const char *script; //when the dialogue is launched, this is executed to generate the instance, with any racial, ability or otherwise mutations of the dialogue
	const char *talk; //the words the object will present to the player

	vector<response *> dests;

	void close()
	{
		dests.deletecontentsp();
	}

	void open() { execute(script); }
	rpgchat(const char *s, const char *t) : script(s), talk(t) {}
	~rpgchat()
	{
		dests.deletecontentsp();
		delete [] script;
		delete [] talk;
	};
};

struct rpgchar
{
	stats base, attributes;
	//attributes
	float health, mana;
	int lastpain, lastaction, lastdeath, statpoints;
	int lastmana, lasthealth;
	bool updatestats, firstattack, secondattack;
	vector<projectile *> projs;
	rpgent *selected[EQUIP_MAX], *selectedspell;
	vector<rpgent *> inventory;
	vector<rpgent *> spellbook;

	rpgchar()
	{
		base = attributes = stats();
		health = base.maxhealth;
		mana = base.maxmana;
		statpoints = 0;
		loopi(EQUIP_MAX) selected[i] = NULL;
		selectedspell = NULL;
		lastmana = lasthealth = lastmillis;
		firstattack = secondattack = false;
		updatestats = true;
	}
	~rpgchar()
	{
		projs.deletecontentsp();
		//note inventory and spellbook is deleted inside rpgent::cleansubtypes();
	}
};

//for damage, use status health, with a negative amount and a millis of 0.
//don't compress spell into item, it's a seperate type, they can't coexist within an entity with the current setup
enum
{
	ITYPE_ORNAMENT = 0,
	ITYPE_CONSUMABLE,
	ITYPE_QUEST,
	ITYPE_MELEE,
	ITYPE_RANGED,
	ITYPE_THROWN
};

struct rpgitem
{
	stats requirements, equipbonus, invbonus;
	int cooldown, range, type, slots;
	int value, weight;
	vector<status *> owneffects;

	const char *drop, *equip;

	rpgitem()
	{
		drop = equip = NULL;
		requirements = equipbonus = invbonus = stats(true);
		cooldown = 250;
		range = 24;
		type = ITYPE_MELEE;
		slots = SLOT_RHAND|SLOT_LHAND;
		weight = 1;
		value = 1;
	}
	~rpgitem()
	{
		owneffects.deletecontentsp();
		if(drop) delete[] drop;
		if(equip) delete[] equip;
	}
};

enum
{
	STYPE_TARGET = 0,
	STYPE_CONE,
	STYPE_SELF
};

struct rpgspell
{
	stats requirements;
	int cooldown, range, type, cost, castdelay, effect, fade, gravity;

	rpgspell()
	{
		requirements = stats(true);
		cooldown = 500;
		range = 128;
		type = STYPE_TARGET;
		cost = 12;
		castdelay = 100;
		effect = 0;
		fade = 500;
		gravity = 200;
	}
	~rpgspell() {}
};

struct rpgobject
{
	bool active, triggered;
	int lastaction;

	rpgobject()
	{
		triggered = false;
		active = true;
		lastaction = 0;
	}
	~rpgobject() {}
};

enum
{
	ENT_CHAR = 0,
	ENT_ITEM,
	ENT_SPELL,
	ENT_OBJECT,
	ENT_MAX
};

//ent temporaries, things which update each frame
struct tempent
{
	int dist;
	float fade;
	int id;
	//int groupid; //e.attr2 of spawnent?

	tempent() : dist(0), fade(1), id(-1) {}
	~tempent() {}
};

struct rpgent : dynent
{
	tempent temp;

	int etype, spawn; //spawn is used by respawnable monsters
	const char *model;
	const char *name, *description, *icon;

	vector<status *> effects;

	int chatpos;
	int lastchat;
	vector<rpgchat *> dialogue; //note, we can script dialogues and other miscelaneous events onto the entity, such as portals with multiple destinations, or items which might have regrettable effects

	/*
		subtypes can be one of the items inside the ENT_* enum;
	*/
	void *subtype;


	const char *interact, //script to execute when interacted with
		*approach, //script to execute when another creature is within range
		*death; //script to execute upon death

	/*
		These functions are to be used for retrieval of specific subtypes;
		mainly for convenience and avoidance of silly problems
	*/
	rpgchar *getchar();
	rpgitem *getitem();
	rpgspell *getspell();
	rpgobject *getobject();

	void addexp(int amount, int skill);
	void cleansubtypes();
	bool dequip(rpgent *item);
	void die();
	bool equip(rpgent *item, int slot = -1);
	float geteffectmul(rpgent &wep);
	int gettotalweight();
	bool drop(rpgent *item);
	void newmodel(bool init = false);
	void pickup(rpgent *item);
	void respawn(bool death = false);
	void reset();
	void takedamage(float amount, int type);
	int transfer();
	void itemeffects(vector<status *> &efc);
	void updateeffects(stats &stat, vector<status *> &efx);
	void updateinv(stats &s);

	rpgent(const char *m, int id) : etype(ENT_CHAR), spawn(-1), model(newstring(m)), chatpos(-1), lastchat(-1)
	{
		interact = approach = death = name = icon = description = NULL;
		subtype = NULL;
		switch(etype)
		{
			case ENT_CHAR:
				subtype = new rpgchar(); break;
			case ENT_ITEM:
				subtype = new rpgitem(); break;
			case ENT_SPELL:
				subtype = new rpgspell(); break;
			case ENT_OBJECT:
				subtype = new rpgobject(); break;
		}

		temp.id = id;
	}
	~rpgent()
	{
		cleansubtypes();
		dialogue.deletecontentsp();
		effects.deletecontentsp();
		if(model) delete[] model;
		if(name) delete[] name;
		if(description) delete[] description;
		if(icon) delete[] icon;
		if(interact) delete [] interact;
		if(approach) delete [] approach;
		if(death) delete [] death;
	}
};

/* struct rpgai : rpgent
{
	int state,
		friendliness;
}; */

enum
{
	S_JUMP = 0, S_LAND, S_RIFLE, S_TELEPORT, S_SPLASH1, S_SPLASH2, S_CG,
	S_RLFIRE, S_RUMBLE, S_JUMPPAD, S_WEAPLOAD, S_ITEMAMMO, S_ITEMHEALTH,
	S_ITEMARMOUR, S_ITEMPUP, S_ITEMSPAWN,  S_NOAMMO, S_PUPOUT,
	S_PAIN1, S_PAIN2, S_PAIN3, S_PAIN4, S_PAIN5, S_PAIN6,
	S_DIE1, S_DIE2,
	S_FLAUNCH, S_FEXPLODE,
	S_SG, S_PUNCH1,
	S_GRUNT1, S_GRUNT2, S_RLHIT,
	S_PAINO,
	S_PAINR, S_DEATHR,
	S_PAINE, S_DEATHE,
	S_PAINS, S_DEATHS,
	S_PAINB, S_DEATHB,
	S_PAINP, S_PIGGR2,
	S_PAINH, S_DEATHH,
	S_PAIND, S_DEATHD,
	S_PIGR1, S_ICEBALL, S_SLIMEBALL, S_PISTOL,

	S_V_BASECAP, S_V_BASELOST,
	S_V_FIGHT,
	S_V_BOOST, S_V_BOOST10,
	S_V_QUAD, S_V_QUAD10,
	S_V_RESPAWNPOINT,

	S_FLAGPICKUP,
	S_FLAGDROP,
	S_FLAGRETURN,
	S_FLAGSCORE,
	S_FLAGRESET,

	S_BURN,
	S_CHAINSAW_ATTACK,
	S_CHAINSAW_IDLE,

	S_HIT
};

namespace entities
{
	extern vector<extentity *> ents;
	extern void spawnfroment(int i);
	extern void startmap();
	extern void teleport(rpgent &d, int dest);
	extern void renderentities(bool mainpass);
	extern void update(rpgent *d);
};

namespace game
{
	extern bool transfer;
	extern rpgent *player1, *hover;
	extern vector<rpgent *> rpgobjs;
	extern int lastid;
	extern int rpgobjdist;
	extern dynent *intersectclosest(const vec &from, const vec &to, rpgent *at, float &bestdist, float maxdist = 1e16f);
	extern bool intersect(dynent *d, const vec &from, const vec &to, float &dist);
	extern const char *geteffecticon(status &s);
	extern const char *geteffectdescription(status &s);
	extern int entcolour(rpgent *d);
};

namespace rpgobj
{
	extern rpgent *selected;
	extern vector <dynlightcache> lights;
	extern vector<projeffect *> effects;
	extern void preload();
	extern int getident(rpgent *d);
	extern int rpgobjupdatedist;
	extern void update();
	extern void adddynlights();
	extern int calcexp(int level);
	extern void renumerate();
};

namespace gui
{
	extern bool guisopen();
	extern void render();
	extern void clearguis();
}

#endif
