/*
 * frend
 * frend.cpp
 * by David Nordlund, 1998-1999
 *
 * frend is short for front-end, and basically it abstracts the toolkit, and
 * provides a simple way for making front end programs.
 * (this particular version uses gtk+)
 * Feel free to use and modify frend to your liking.
 */

#include <iostream>
#include <list>
#include <set>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#ifdef FR_GNOME
# include <gnome.h>
#else
# include <gtk/gtk.h>
#endif
#include "frend.h"

using namespace std;

char *fr_AppName;
GtkTooltips *fr_Tooltips;

#ifdef FR_GNOME
GnomeUIInfo fr_GnomeUIseparator = GNOMEUIINFO_SEPARATOR;
GnomeUIInfo fr_GnomeUIend = GNOMEUIINFO_END;
#endif
void			fr_exit(GtkObject*O, int*exitcode);
void			fr_exit_event(GtkObject*O, GdkEvent*Event, int*exitcode);

/* ############################### fr_ArgList ############################# */
/// Create an ArgList of initial size s
fr_ArgList::fr_ArgList(int s) {
   ArgAlloc(s);
}

/// Create an ArgList from the string args
fr_ArgList::fr_ArgList(char*args) {
   static char*delim = " \n\r\t\v";
   char *NewArg, *argd;
   int s = 2;
   for(int i=0; args[i]; i++)
     if(args[i]<=' ')
       s++;
   ArgAlloc(s);
   argd = new char[strlen(args) + 1];
   strcpy(argd, args);
   NewArg = strtok(argd, delim);
   while(NewArg) {
      *this << NewArg;
      NewArg = strtok(0, delim);
   };
   delete[] argd;
}

/// Create an ArgList from a url-encoded string
fr_ArgList::fr_ArgList(char*s, bool strict) {
   int c, ai = 0, i, p=1;
   char a[128], hex[5];
	
   if ((!s)||(!s[0])) return;
   for(i=0; s[i]; i++)
     if((s[i]=='&')||(s[i]==' '))
       p++;
   ArgAlloc(p);
   memset(a, 0, sizeof(a));
   for(i=0; s[i];)
     switch(s[i]) {
      case '%':
	memset(hex, 0, sizeof(hex));
	strncpy(hex, &(s[i+1]), 2);
	sscanf(hex, " %X ", &c);
	a[ai++] = c;
	i+=3;
	break;
      case ' ':
      case '&':
	*this << a;
	ai = 0;
	memset(a, 0, sizeof(a));
	i++;
	break;
      case '+':
	a[ai++] = ' ';
	break;
      default:
	a[ai++] = s[i++];
     };
   *this << a;
}

/// Create an ArgList from an array of strings
fr_ArgList::fr_ArgList(int argc, char*argv[]) {
   ArgAlloc(argc);
   for(int a=0; a<argc; a++)
     *this << argv[a];
}

/// Allocate memory for s entries in an ArgList
void fr_ArgList::ArgAlloc(int s) {
   casesensitivity = fr_CaseSensitive;
   s++; //be slightly generous;
   argv = new char*[sizeof(char*)*s];
   marked = new bool[sizeof(bool)*s];
   sensitive = new fr_CaseSensitivity[sizeof(fr_CaseSensitivity)*s];
   size = --s;
   countargs = countchars = 0;
   argv[0] = 0;
}

/// Copy the ArgList elements into bigger arrays and delete the old ones
void fr_ArgList::Grow() {
   size *= 2;
   char **newargs = new char*[size];
   bool *newmarks = new bool[size];
   fr_CaseSensitivity *newsense = new fr_CaseSensitivity[size];
   
   memcpy(newargs, argv, (countargs+1)*sizeof(char*));
   memcpy(newmarks, marked, (countargs+1)*sizeof(bool));
   memcpy(newsense, sensitive, (countargs+1)*sizeof(fr_CaseSensitivity));
   
   delete[] argv;
   delete[] marked;
   delete[] sensitive;
   
   argv = newargs;
   marked = newmarks;
   sensitive = newsense;
}

/// Free up memory that was allocated
fr_ArgList::~fr_ArgList() {
   for(int i=0; i<countargs; i++)
     delete[] argv[i];
   delete[] argv;
   delete[] marked;
   delete[] sensitive;
}

/// Set the case sensitivity of forthcoming Args
fr_ArgList& fr_ArgList::operator<< (const fr_CaseSensitivity cs) {
   casesensitivity = cs;
   return *this;
}

/// Add an argument to the list
fr_ArgList& fr_ArgList::operator<< (const char*arg) {
   if (!arg) return *this;
   if (size-countargs<1) Grow();
   int s = strlen(arg) + 1;
   char *a = new char[s];
   strcpy(a, arg);
   argv[countargs] = a;
   sensitive[countargs] = casesensitivity;
   argv[++countargs] = 0;
   countchars += s;
   return *this;
}

/// @return a pointer to the Nth entry in an ArgList
char* fr_ArgList::operator[] (int n) {
   if( (n < 0) || (n >= countargs) )
     return (char*)0;
   return argv[n];
}

/**
 * Convert the array of entries to a single string
 * @return a pointer to the static string
 */
char*fr_ArgList::GetString() {
   static char ArgString[256];
   strcpy(ArgString, "");
   for(int i=0; argv[i]; i++) {
      strcat(ArgString, argv[i]);
      if(argv[i+1])
	strcat(ArgString, " ");
   };
   return ArgString;
}

/// Convert an fr_ArgList into a url-encoded string
char*fr_ArgList::GetEncoded(bool strict) {
   char *a, hex[5];
   static char s[256];
	
   if(countargs<1)
     return (char*)0;
   int si = 0;
   memset(s, 0, sizeof(s));
   for(int i=0; i<countargs; i++) {
      a = argv[i];
      for(int c=0; (a)&&(a[c]); c++) {
	 if((isalnum(a[c]))
	    ||((!strict)&&(strchr("_-/.", a[c]))))
	   s[si++] = a[c];
	 else if(a[c]==' ')
	   s[si++] = '+';
	 else {
	    sprintf(hex, "%%%.2X", a[c]);
	    strcat(s, hex);
	    si+=strlen(hex);
	 };
      };
      s[si++] = strict?'&':' ';
      if(si>=1020) break;
   };
   s[--si] = 0;
   return s;
}

/**
 * Check to see if S is an entry in this ArgList
 * skip over marked entries
 * if a match is found, mark it
 * @return the index of the match, -1 for no match
 */
int fr_ArgList::MatchArg(char*S, fr_CaseSensitivity cs) {
   if((!S)||(S[0]==0))
     return -2;
   for(int a=0, match=0; a<countargs && argv[a]; a++) {
      if (marked[a]) continue;
      if(cs==fr_CaseInsensitive)
	match = !strcasecmp(S, argv[a]);
      else
	match = !strcmp(S, argv[a]);
      if(match) {
	 marked[a] = true;
	 return a;
      };
   };
   return -1;
}

/**
 * For every entry in this ArgList,
 * try to Match it in ArgList L.
 * If it is there, mark the entry that matched it here
 * @return the index of the match in this ArgList, -1 for no match
 */
int fr_ArgList::MatchArgs(fr_ArgList& L) {
   for(int a=0,m=0; a<countargs; a++) {
      // if (marked[a]) continue;
      m = L.MatchArg(argv[a], sensitive[a]);
      if (m<0) continue;
      marked[a] = true;
      return m;
   };
   return -1;
}

/// Unmark any marked entries in this ArgList
void fr_ArgList::ClearMarks() {
   memset(marked, 0, size);
}

/// Mark the Nth entry of this ArgList
void fr_ArgList::Mark(int n) {
   if ( (n>=0) && (n<countargs) )
     marked[n] = true;
}

/// Check if entry n is marked
bool fr_ArgList::IsMarked(int n) {
   if( (n>=0) && (n<countargs) )
     return marked[n];
   return false;
}

/// @return a pointer to the first-marked or first entry in this ArgList
char*fr_ArgList::GetPreferredArg() {
   for(int a=0; a<countargs; a++)
     if (marked[a]) return argv[a];
   return argv[0];
}

/* ############################## fr_Image ################################ */
fr_Image::fr_Image(fr_Element*parent, char**xpm) {
   GtkStyle *Style = parent?
     gtk_widget_get_style(parent->Window)
       :gtk_widget_get_default_style();
   xpixmap = gdk_pixmap_create_from_xpm_d(parent->Window->window,
					  &xbitmap, &(Style->bg[0]), xpm);
}

#ifdef FR_GNOME
fr_Image::fr_Image(fr_Element*, const char*GnomePic) {
   gnome_stock_pixmap_gdk(GnomePic, (char*)0, &xpixmap, &xbitmap);
}
#endif


/* ############################## fr_Style ################################ */
fr_Style::fr_Style(fr_Element*parent, int fg_rgb, int bg_rgb, fr_Image*I) {
   GdkColor fg_clr, bg_clr;
   fg_clr.red   = ((fg_rgb >> 16) & 0xFF) * 0xFF;
   fg_clr.green = ((fg_rgb >>  8) & 0xFF) * 0xFF;
   fg_clr.blue  = (fg_rgb & 0xFF) * 0xFF;
   fg_clr.pixel = 0;			 
   bg_clr.red   = ((bg_rgb >> 16) & 0xFF) * 0xFF;
   bg_clr.green = ((bg_rgb >>  8) & 0xFF) * 0xFF;
   bg_clr.blue  = (bg_rgb & 0xFF) * 0xFF;
   bg_clr.pixel = 0;
   GdkColormap *cmap = gdk_window_get_colormap(parent->Window->window);
   gdk_color_alloc(cmap, &fg_clr);
   gdk_color_alloc(cmap, &bg_clr);
   GtkWidget *g = parent->getGUI();
   if(g)
     Style = gtk_style_copy(gtk_widget_get_style(g));
   else
     Style = gtk_style_copy(gtk_widget_get_default_style());
   Style->fg[GTK_STATE_NORMAL] = fg_clr;
   Style->fg[GTK_STATE_SELECTED] = fg_clr;
   Style->fg[GTK_STATE_ACTIVE] = fg_clr;
   Style->bg[GTK_STATE_NORMAL] = bg_clr;
   Style->bg[GTK_STATE_SELECTED] = bg_clr;
   Style->bg[GTK_STATE_ACTIVE] = bg_clr;
   Style->bg[GTK_STATE_INSENSITIVE] = bg_clr;
   if(I)
     Style->bg_pixmap[GTK_STATE_NORMAL] = I->xpixmap;
}

/* ############################## fr_Element ############################## */
/// Create an empty element
fr_Element::fr_Element() {
   Name = "";
   GUI = Element = (GtkWidget*)0;
   Listeners = 0;
}

/// Create an empty element with a parent
fr_Element::fr_Element(fr_Element* parent, char*name):
Parent(parent), Name(name)
{
   Element = GUI = (GtkWidget*)0;
   if (Parent) Window = Parent->Window;
   Listeners = 0;
}

/// Create a custom element from a gtk widget
fr_Element::fr_Element(fr_Element*parent, GtkWidget*custom):
Parent(parent)
{
   Name = "";
   Element = GUI = custom;
   if (Parent) Window = Parent->Window;
   SetVisibility(true);
   Listeners = 0;
}

/// @return a pointer to the gtk widget representing this element
GtkWidget*fr_Element::getGUI() {
   if (GUI) return GUI;
   return Element;
}

/// Create the tool-tip for this element
void fr_Element::CreateTooltip(char*T) {   
   if ((!T)||(T[0]==0)) return;
   if(!fr_Tooltips)
     fr_Tooltips = gtk_tooltips_new();
   gtk_tooltips_set_tip(fr_Tooltips, Element, T, 0);
}

/// Assign a name to this element
void fr_Element::SetName(char*name) {
   Name = name;
}

/// Assign a tool-tip to this element
void fr_Element::SetTooltip(char*T) {
   CreateTooltip(T);
}

/// Enable/Disable the ability to use this element
void fr_Element::SetEditable(bool s) {
   if (Element) gtk_widget_set_sensitive(Element, s);
   if (GUI)     gtk_widget_set_sensitive(GUI,     s);
}

/// Suggest a size for this element to take up
void fr_Element::SetSize(int x, int y) {
   GtkWidget *W = getGUI();
   if (!W) return;
   gtk_widget_set_usize(W, x, y);
}

/// Show / Hide this element
void fr_Element::SetVisibility(bool s) {
   if (!Element) g_error("Whoah, I can't show nothing!");
   if (!GUI) GUI = Element;
   if (s) {
      gtk_widget_show(Element);
      gtk_widget_show(GUI);
   } else {
      gtk_widget_hide(Element);
      gtk_widget_hide(GUI);
   };
}

/// Focus input on this element
void fr_Element::GrabFocus() {
   gtk_widget_grab_focus(Element);
   if(GTK_WIDGET_CAN_DEFAULT(Element))
     gtk_widget_grab_default(Element);
}

/// If this element is a container, give it some contents
void fr_Element::Contain(fr_Element*Contents) {
   gtk_container_add(GTK_CONTAINER(Element), Contents->getGUI());
}

/// Add a dettachable handle to this element
void fr_Element::AddHandle() {
   GtkWidget *Handle, *gui;
   
   if (!(gui = getGUI())) return;
   Handle=gtk_handle_box_new();
   gtk_widget_show(Handle);
   gtk_container_add(GTK_CONTAINER(Handle), gui);
   GUI = Handle;
}

/// Add a label to the left of this element from string L
void fr_Element::AddLabel(char*L) {
   GtkWidget *lbl = gtk_label_new(L);
   gtk_widget_show(lbl);
   if(!getGUI()) {
      GUI = Element = lbl;
      return;
   };
   GtkWidget *box = gtk_hbox_new(false, 1);
   gtk_box_pack_start(GTK_BOX(box), lbl, false, false, 1);
   gtk_box_pack_start(GTK_BOX(box), getGUI(), true, true, 1);
   gtk_widget_show(box);
   GUI = box;
}

/// Add a label to the left of this element based on Label L
void fr_Element::AddLabel(fr_Label*L) {
   GtkWidget *box = gtk_hbox_new(false, 1);
   gtk_box_pack_start(GTK_BOX(box), L->getGUI(), false, false, 1);
   if(getGUI())
     gtk_box_pack_start(GTK_BOX(box), getGUI(), true, true, 1);
   gtk_widget_show(box);
   GUI = box;
}

void fr_Element::ApplyStyle(fr_Style* S) {
   if(S) {
      //gtk_widget_set_style(Element, S->Style);
      gtk_widget_set_style(GUI,     S->Style);
   } else {
      //gtk_widget_restore_default_style(Element);
      gtk_widget_restore_default_style(GUI);
   }
}
void fr_Element::ResetStyle() {
   ApplyStyle(0);
}

/**
 * Set up the event handler for this element
 * If this element receives an event,
 * notify the EventHandler in the parent.
 */
void fr_Element::AddListener(fr_Listener*L) {
   if (!L) return;
   if (!Listeners)
     Listeners = new list<fr_Listener*> ();
   Listeners->push_back(L);
}

/// Remove an event listener for this object
void fr_Element::RemoveListener(fr_Listener*L) {
   if ((!L) || (!Listeners)) return;
   list<fr_Listener*>::iterator i;
   for(i=Listeners->begin(); i!=Listeners->end(); i++)
     if(*i == L)
       Listeners->erase(i);
   if(Listeners->empty()) {
      delete Listeners;
      Listeners = 0;
   }
}

/* ############################# fr_Event ################################# */
fr_Event::fr_Event(fr_Element*E, int t, int ia, void*arg):
Element(E), Type(t), intArg(ia), Arg(arg)
{
    if(!(E->Listeners)) return;
    list<fr_Listener*>::iterator i;
    for(i=E->Listeners->begin(); i!=E->Listeners->end(); i++)
      (*i)->EventOccurred(this);
}

bool fr_Event::Is(fr_Element* E, int t) {
  if(Element == E) {
    if(t<0) return true;
    else return Type==t;
  }
  return false;
}

/* ############################# fr_Listener ############################## */
void fr_Listener::Click(GtkObject*, fr_Element*E) {
   fr_Event(E, fr_Click);
}
void fr_Listener::Changed(GtkObject*, fr_Element*E) {
   fr_Event(E, fr_Changed);
}
void fr_Listener::MenuClick(GtkObject*, fr_Element*E) {
   fr_Event(E, fr_MenuClick);
}


/* ############################# fr_Separator ############################# */
/// Create a new bar (just a divider)
fr_Separator::fr_Separator(fr_Element*parent, fr_Direction Dir):
fr_Element(parent)
{
   if(Dir==fr_Vertical)
     GUI = Element = gtk_vseparator_new();
   else
     GUI = Element = gtk_hseparator_new();
   SetVisibility(true);
}

/* ############################# fr_ScrollWin ############################# */
fr_ScrollWin::fr_ScrollWin(fr_Element*parent, fr_Element*child):
fr_Box(parent)
{
   Element = child->getGUI();
   GUI = gtk_scrolled_window_new(0, 0);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(GUI),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
   gtk_container_add(GTK_CONTAINER(GUI), Element);
   SetVisibility(true);
}

fr_ScrollWin::fr_ScrollWin(fr_Element*parent, GtkWidget*child):
fr_Box(parent)
{
   Element = child;
   GUI = gtk_scrolled_window_new(0, 0);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(GUI),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
   gtk_container_add(GTK_CONTAINER(GUI), Element);
   SetVisibility(true);
}

/* ############################# fr_Box	################################## */
/// Create a new box
fr_Box::fr_Box(fr_Element*parent, char*name):
fr_Element(parent, name)
{
}

/// Set the table size for this box
void fr_Box::SetGridSize(int x, int y, bool SameSize) {
   GUI = Element = gtk_table_new(y, x, SameSize);
   SizeX = x;
   SizeY = y;
   PosX = PosY = 0;
   SetPadding(2, 2);
   SetStretch(Normal, Normal);
   fr_Element::SetVisibility(true);
}

/// Set the padding to be put around elements added to this box
void fr_Box::SetPadding(int x, int y) {
   PadX = x;
   PadY = y;
}

/// Specify the stretching mechanism to be used with elements added to this box
void fr_Box::SetStretch(fr_StretchMode x, fr_StretchMode y) {
   StretchX = x;
   StretchY = y;
   ax = ay = GTK_FILL;
   if     (StretchX==Shrink) ax = GTK_SHRINK;
   else if(StretchX == Grow) ax = GTK_EXPAND;
   if     (StretchY==Shrink) ay = GTK_SHRINK;
   else if(StretchY == Grow) ay = GTK_EXPAND;
}

/// put padding space around this box
void fr_Box::Pad() {
   if (!getGUI()) return;
   gtk_container_border_width(GTK_CONTAINER(getGUI()), PadX);
}

/**
 * Put a frame aroud this box.
 * If the box has been named (fr_Element::SetName), the frame will
 * have the boxes namenear the upper left corner
 */
void fr_Box::Frame() {
   if (!getGUI()) return;
   GtkWidget *F = gtk_frame_new(Name);
   gtk_container_add(GTK_CONTAINER(F), getGUI());
   gtk_widget_show(F);
   GUI = F;
}

/// Add a 3D border around this box, Beveled Out
void fr_Box::AddBorderBevelOut() {
   if (!GUI) return;
   GtkWidget *B = gtk_frame_new(0);
   gtk_container_add(GTK_CONTAINER(B), GUI);
   gtk_frame_set_shadow_type(GTK_FRAME(B), GTK_SHADOW_OUT);
   GUI = B;
}

/// Add a 3D border around this box, Beveled In
void fr_Box::AddBorderBevelIn() {
   if (!GUI) return;
   GtkWidget *B = gtk_frame_new(0);
   gtk_container_add(GTK_CONTAINER(B), GUI);
   gtk_frame_set_shadow_type(GTK_FRAME(B), GTK_SHADOW_IN);
   GUI = B;
}

/**
 * Add an element to this box.
 * It will be placed horizontally between x1 and x2
 * It will be placed vertically between y1 and y2
 */
void fr_Box::Pack(fr_Element& E, int x1, int y1, int x2, int y2) {
   GtkWidget *G = E.getGUI();
   if(!G)
     g_error("Packing a non-existant Element into table!");
   if((StretchX == Normal)&&(StretchY == Normal)) {
      gtk_table_attach_defaults(GTK_TABLE(Element), G,
				x1, x2, y1, y2);
   } else {
      gtk_table_attach(GTK_TABLE(Element), G,
		       x1, x2, y1, y2, ax, ay, PadX, PadY);
   };
   PosX = x2;
   PosY = y1;
   if(PosX >= SizeX) {
      PosX = 0;
      PosY++;
   };
}

/**
 * Add an element to this box.
 * It will start in the next table cell after the last element added
 * (or the first cell if it is the first item)
 * and will take up dx cells across and dy cells down
 */
void fr_Box::Pack(fr_Element& E, int dx, int dy) {
   Pack(E, PosX, PosY, PosX+dx, PosY+dy);
}

/* ############################# fr_Label ################################ */
/// Create a label based on text, a pixmap or both
fr_Label::fr_Label(fr_Element*parent, char*label, char**XPM, bool Both):
fr_Element(parent, label)
{
   if(XPM) {
      xpixmap = XPM;
      GtkStyle *Style = gtk_widget_get_default_style();
      GdkBitmap *mask;
      GdkPixmap *Pixmap = gdk_pixmap_create_from_xpm_d(Window->window, &mask,
						       &Style->bg[GTK_STATE_NORMAL], (char**)XPM);
      GUI = Element = gtk_pixmap_new(Pixmap, mask);
      gtk_widget_show(Element);
      IsPixmap = true;
      if(Both) {
	 GUI = gtk_hbox_new(false, 0);
	 gtk_box_pack_start(GTK_BOX(GUI), Element, false, false, 1);
	 Element = gtk_label_new(label);
	 gtk_widget_show(Element);
	 gtk_box_pack_start(GTK_BOX(GUI), Element, false, false, 1);
	 Element = GUI;
      };
   } else {
      xpixmap = (char**)0;
      GUI = Element = gtk_label_new(Name);
      IsPixmap = false;
   };
   SetVisibility(true);
}

/// Assign new text to this label
void fr_Label::SetLabel(char*label) {
   Name = label;
   if(!IsPixmap)
#if ((GTK_MAJOR_VERSION==1)&&(GTK_MINOR_VERSION<2))
     gtk_label_set(GTK_LABEL(Element), Name);
#else
   gtk_label_set_text(GTK_LABEL(Element), Name);
#endif
}

/* ############################# fr_Button ################################ */
/// Create a new button with a string for a label
fr_Button::fr_Button(fr_Element*parent, char*label):
fr_Element(parent, label)
{
#ifdef FR_GNOME
   Element = (GtkWidget*)0;
   if(strlen(label)<21) {
      char BtnStr[30];
      sprintf(BtnStr, "Button_%s", label);
      if(strstr(GNOME_STOCK_BUTTON_OK  GNOME_STOCK_BUTTON_CANCEL
		GNOME_STOCK_BUTTON_YES GNOME_STOCK_BUTTON_NO
		GNOME_STOCK_BUTTON_CLOSE
		GNOME_STOCK_BUTTON_APPLY
		GNOME_STOCK_BUTTON_HELP
		GNOME_STOCK_BUTTON_PREV GNOME_STOCK_BUTTON_NEXT
		GNOME_STOCK_BUTTON_UP GNOME_STOCK_BUTTON_DOWN
		GNOME_STOCK_BUTTON_FONT,
		BtnStr))
	GUI = Element = gnome_stock_or_ordinary_button(BtnStr);
   };
   if(!Element)
     GUI = Element = gnome_stock_or_ordinary_button(label);
#else
   GUI = Element = gtk_button_new_with_label(label);
#endif
   SetVisibility(true);
}

/// Create a new button with a label from an fr_Label
fr_Button::fr_Button(fr_Element*parent, fr_Label*label):
fr_Element(parent, label->Name)
{
   Element = gtk_button_new();
   Contain(label);
   SetVisibility(true);
}

fr_Button::fr_Button(fr_Element*parent, char*label, fr_Image& I):
fr_Element(parent, label)
{
   GUI = gtk_hbox_new(false, false);
   gtk_widget_show(GUI);
   GtkWidget *l = gtk_label_new(Name);
   gtk_widget_show(l);
   GtkWidget *i = gtk_pixmap_new(I.xpixmap, I.xbitmap);
   gtk_widget_show(i);
   gtk_box_pack_start(GTK_BOX(GUI), i, false, true, 1);
   gtk_box_pack_start(GTK_BOX(GUI), l, false, true, 1);
   Element = gtk_button_new();
   gtk_container_add(GTK_CONTAINER(Element), GUI);
   GUI = Element;
   SetVisibility(true);
}

void fr_Button::AddListener(fr_Listener*L) {
   if(!Listeners)
     gtk_signal_connect(GTK_OBJECT(Element), "clicked",
			GTK_SIGNAL_FUNC(fr_Listener::Click), this);
   fr_Element::AddListener(L);
}

/* ############################# fr_ButtonBox ############################ */
fr_ButtonBox::fr_ButtonBox(fr_Element*parent, fr_Direction dir):
fr_Element(parent)
{
   Element = GUI = (dir == fr_Vertical)?
     gtk_vbutton_box_new():gtk_hbutton_box_new();
   gtk_button_box_set_layout(GTK_BUTTON_BOX(Element), GTK_BUTTONBOX_END);
   gtk_button_box_set_spacing(GTK_BUTTON_BOX(Element), 5);
   SetVisibility(true);
}

/// Add a button to the button box
void fr_ButtonBox::AddButton(fr_Button& btn, bool dflt=false) {
   GtkWidget *b = btn.getGUI();
   GTK_WIDGET_SET_FLAGS(b, GTK_CAN_DEFAULT);
   if(dflt)
     gtk_widget_grab_default(b);
   gtk_box_pack_start(GTK_BOX(Element), b, true, true, 0);
}

/* ############################# fr_Option ############################### */
/// Create a new option
fr_Option::fr_Option(fr_Element*parent, char*name):
fr_Element(parent, name)
{
   KeyStroke = (char*)0;
   Enabled = true;
}

/**
 * Associate a keystroke with this option
 * (this is not an accelerator for the frend-based GUI)
 */
void fr_Option::SetKeyStroke(char*K) {
   KeyStroke = K;
}

/// Assign a tool-tip to this option
void fr_Option::SetTooltip(char*T) {
   char ToolStr[256], S[80];
   if(T) strcpy(ToolStr, T);
   else  strcpy(ToolStr, "");

   char *arg = Args.GetPreferredArg();   
   if(arg) {
      sprintf(S, "\noption: %s", arg);
      strcat(ToolStr, S);
   };
   if(KeyStroke) {
      sprintf(S, "\nkeystroke: %s", KeyStroke);
      strcat(ToolStr, S);
   };
   CreateTooltip(ToolStr);   
}

/// Enable/Disable the ability to use this option
void fr_Option::SetEditable(bool s) {
   if(!Enabled) s=false;
   if(Element) gtk_widget_set_sensitive(Element, s);
   if(GUI)     gtk_widget_set_sensitive(GUI, s);
}

/// If enable is false, set the option to its default value and disable it
void fr_Option::EnableIf(bool cond) {
   if((Enabled)&&(!cond))
     SetToDefault();
   Enabled = cond;
   SetEditable(cond);
}

/* ############################# fr_Toggle ############################### */
/// Create a new Toggle element
fr_Toggle::fr_Toggle(fr_Element*parent, char*name):
fr_Option(parent, name)
{
   DefaultState = false;
}

/// Specify the default state of this toggle item
void fr_Toggle::SetDefaultState(bool s) {
   DefaultState = s;
}

/// Assign a tool-tip to this toggle element
void fr_Toggle::SetTooltip(char*T) {
   char ToolStr[256], S[80];
   
   if(T) strcpy(ToolStr, T);
   else  strcpy(ToolStr, "");

   char *arg = Args.GetPreferredArg();   
   if(arg) {
      sprintf(S, "\noption(on): %s", arg);
      strcat(ToolStr, S);
   };
   arg = NotArgs.GetPreferredArg();   
   if(arg) {
      sprintf(S, "\noption(off): %s", arg);
      strcat(ToolStr, S);
   };
   sprintf(S, "\nDefault: %s", DefaultState?"on":"off");
   strcat(ToolStr, S);
   if(KeyStroke) {
      sprintf(S, "\nkeystroke: %s", KeyStroke);
      strcat(ToolStr, S);
   };
   CreateTooltip(ToolStr);   
}

/**
 * Given an ArgList, see if any args belong to this toggle element,
 * if they do, they will be marked, and the toggle state set accordingly
 */
void fr_Toggle::SiftArgs(fr_ArgList& L) {
   if(Args.MatchArgs(L)>=0)    SetState(true);
   if(NotArgs.MatchArgs(L)>=0) SetState(false);
}

/// Set the state of this toggle element
void fr_Toggle::SetState(bool s) {
   if(Enabled) gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(Element), s);
}

/// @return the state of this toggle element
bool fr_Toggle::GetState() {
   return GTK_TOGGLE_BUTTON(Element)->active;
}

/// Set the state of this toggle element to its default state
void fr_Toggle::SetToDefault() {
   SetState(DefaultState);
}

/// @return true if this toggle element is in its default state
bool fr_Toggle::IsDefault() {
   return (GetState() == DefaultState);
}

/**
 * @return a command-line argument that reflects the state of this element
 * @return 0 if it is in its default state
 */
char*fr_Toggle::GetActiveArg() {
   if (IsDefault()) return (char*)0;
   return DefaultState?
     NotArgs.GetPreferredArg() : Args.GetPreferredArg();
}

/**
 * Add to an ArgList a command-line argument that reflects the current state
 * of this toggle element.
 * Nothing is added if it is in its default state
 */
void fr_Toggle::CompileArgs(fr_ArgList& L) {
   L << GetActiveArg();
}

void fr_Toggle::AddListener(fr_Listener*L) {
   if(!Listeners)
     gtk_signal_connect(GTK_OBJECT(Element), "toggled",
			GTK_SIGNAL_FUNC(fr_Listener::Click), this);
   fr_Element::AddListener(L);
}

/* ############################# fr_Checkbox ############################## */
/// Create a new checkbox, with a label and a default state
fr_Checkbox::fr_Checkbox(fr_Element*parent, char*label, bool s):
fr_Toggle(parent, label)
{
   GUI = Element = gtk_check_button_new_with_label(label);
   SetDefaultState(s);
   SetVisibility(true);
}

/* ############################# fr_ToggleInGroup ######################### */
fr_ToggleInGroup::fr_ToggleInGroup(fr_GroupHolder*parent, char*name):
fr_Toggle(parent, name)
{
   Next = 0;
}

fr_ToggleInGroup::~fr_ToggleInGroup() {
   if (Next) delete Next;
}

/* ############################# fr_GroupHolder ########################### */
/// Create a new GroupHolder
fr_GroupHolder::fr_GroupHolder(fr_Element*parent, char*name):
fr_Option(parent, name)
{
   First = Last = 0;
   DefaultIndex = GroupSize = 0;
   Group = 0;
}

fr_GroupHolder::~fr_GroupHolder() {
   delete First;
}

/// Activate element i in this group
void fr_GroupHolder::SelectIndex(int i) {
   fr_ToggleInGroup *P = First;
   for(int x=0;  P;  x++, P=P->Next)
     P->SetState(x==i);
}

/// @return a pointer to the selected element in this group
fr_ToggleInGroup*fr_GroupHolder::GetSelected() {
   fr_ToggleInGroup *P;
   for(P=First; P; P=P->Next)
     if(P->GetState())
       return P;
   return (fr_ToggleInGroup*)0;
}

/// @return the index of the selected element in this group
int fr_GroupHolder::GetIndex() {
   fr_ToggleInGroup *P = First;
   for(int i=0;  P;  i++, P=P->Next)
     if(P->GetState())
       return i;
   return -1;
}

/// @return true if the selected element is the default element
bool fr_GroupHolder::IsDefault() {
   return (GetIndex() == DefaultIndex);
}

/// Select the default element
void fr_GroupHolder::SetToDefault() {
   SelectIndex(DefaultIndex);
}

/// Given an ArgList, scan for args belonging to elements in this group
void fr_GroupHolder::SiftArgs(fr_ArgList& L) {
   fr_ToggleInGroup *P;
   
   for(P=First; P; P=P->Next)
     P->SiftArgs(L);
}

/// Add to L the an arg reflecting the current state of the selected element
void fr_GroupHolder::CompileArgs(fr_ArgList& L) {
   fr_ToggleInGroup *P;
   if(!(P = GetSelected()))
     return;
   P->CompileArgs(L);
}

/* ############################# fr_RadioButton ########################### */
/// Create a new radio button with a label
fr_RadioButton::fr_RadioButton(fr_RadioGroup*parent, char*label):
fr_ToggleInGroup(parent, label)
{
   GUI = Element = gtk_radio_button_new_with_label(parent->Group, label);
   SetVisibility(true);
   parent->AddItem(this);
}

/* ############################# fr_RadioGroup ############################ */
/// Create a group for adding radio buttons of a single group
fr_RadioGroup::fr_RadioGroup(fr_Element*parent, char*label, bool box):
fr_GroupHolder(parent, label)
{
   Group = 0;
   if(box) {
      Element = gtk_hbox_new(false, 3);
      GUI = gtk_frame_new(label);
      gtk_container_add(GTK_CONTAINER(GUI), Element);
      SetVisibility(true);
   } else
     Element = GUI = (GtkWidget*)0;
}

/// Add a radio button to this group
void fr_RadioGroup::AddItem(fr_RadioButton*NewItem) {
   if (!NewItem) return;
   else GroupSize++;
   if (!First) First = NewItem;
   else Last->Next = NewItem;
   Last = NewItem;
   if(Element)
     gtk_box_pack_start(GTK_BOX(Element), Last->GUI, true, false, 3);
   Group = gtk_radio_button_group(GTK_RADIO_BUTTON(Last->Element));
}

/* ############################# fr_MenuItem ############################## */
/// Create a Menu item with a label
fr_MenuItem::fr_MenuItem(fr_PulldownMenu*parent, char*label):
fr_ToggleInGroup(parent, label)
{
   GUI = Element = gtk_radio_menu_item_new_with_label(parent->Group, label);
   gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(Element), true);
   parent->Group = gtk_radio_menu_item_group(GTK_RADIO_MENU_ITEM(Element));
   SetVisibility(true);
   parent->AddItem(this);
}

/// @return true if this menu option is selected
bool fr_MenuItem::GetState() {
   return GTK_CHECK_MENU_ITEM(Element)->active;
}

/// Scan ArgList L for args that match this menu item
void fr_MenuItem::SiftArgs(fr_ArgList& L) {
   if(Args.MatchArgs(L)>=0)
     ((fr_PulldownMenu*)Parent)->SelectItem(this);
}

/* ############################ fr_PulldownMenu ########################### */
/// Create a new pull-down menu with a title (label)
fr_PulldownMenu::fr_PulldownMenu(fr_Element*parent, char*label):
fr_GroupHolder(parent, label)
{
   GUI = Element = gtk_option_menu_new();
   Menu = gtk_menu_new();
   Group = 0;
   gtk_option_menu_set_menu(GTK_OPTION_MENU(GUI), Menu);
   gtk_widget_show(GUI);
   GtkWidget *W = gtk_menu_item_new_with_label(label);
   gtk_widget_show(W);
   gtk_menu_append(GTK_MENU(Menu), W);
   W = gtk_menu_item_new();
   gtk_widget_set_sensitive(W, false);
   gtk_widget_show(W);
   gtk_menu_append(GTK_MENU(Menu), W);
   gtk_option_menu_set_history(GTK_OPTION_MENU(GUI), 0);
   GroupSize = 0;
   SetVisibility(true);
}

/// Add a MenuItem to this pull down menu
void fr_PulldownMenu::AddItem(fr_MenuItem*NewItem) {
   if (!NewItem) return;
   GroupSize++;
   if (!First) First = NewItem;
   else Last->Next = NewItem;
   Last = NewItem;
   gtk_menu_append(GTK_MENU(Menu), NewItem->GUI);
}

/// Select by index a MenuItem in this pull-down menu
void fr_PulldownMenu::SelectIndex(int i) {
   fr_ToggleInGroup *P = First;
   for(int x=0; P; x++,P=P->Next)
     if(i==x) {
	SelectItem(P);
	break;
     };
}

/// Select by matching a pointer a menu item in this pull-down menu
void fr_PulldownMenu::SelectItem(fr_ToggleInGroup*I) {
   if(!Enabled) return;
   fr_ToggleInGroup *P = First;
   for(int i=0; P; i++,P=P->Next)
     if(P==I) {
	gtk_check_menu_item_set_state(GTK_CHECK_MENU_ITEM(P->Element), true);
	gtk_option_menu_set_history(GTK_OPTION_MENU(GUI), i+2);
	break;
     };
}

/// Select the default item in this pull-down menu
void fr_PulldownMenu::SetToDefault() {
   SelectItem(First);
   gtk_option_menu_set_history(GTK_OPTION_MENU(GUI), 0);
}

void fr_PulldownMenu::ApplyStyle(fr_Style*S) {
   fr_Element::ApplyStyle(S);
   if(S)
     gtk_widget_set_style(Menu, S->Style);
   else
     gtk_widget_restore_default_style(Menu);
}

void fr_PulldownMenu::AddListener(fr_Listener*L) {
   if(!Listeners)
     gtk_signal_connect(GTK_OBJECT(Element), "value_changed",
			GTK_SIGNAL_FUNC(fr_Listener::Click), this);
   fr_Element::AddListener(L);
}

/* ############################# fr_Adjustment ############################ */
/**
 * Create an adjustment element with a
 * Default Value, minimum value, maximum value, and number of
 * decimal places to use
 */
fr_Adjustment::fr_Adjustment(fr_Element*parent, char*name,
			     float DV, float min, float max, int DP):
fr_Option(parent, name)
{
   DefaultValue = DV;
   DecimalPlaces = DP;
   Minimum = min;
   Maximum = max;
   sprintf(Format, "%%.%df", DecimalPlaces);
   Step = 1;
   for(int setStep=0; setStep<DecimalPlaces; setStep++)
     Step /= 10;
   Adj = gtk_adjustment_new(DV, min, max, Step, Step, 1);
}

/// Set the value of this adjustment
void fr_Adjustment::SetValue(float v) {
   if (Enabled) gtk_adjustment_set_value(GTK_ADJUSTMENT(Adj), v);   
}

/// @return the value of this adjustment
float fr_Adjustment::GetValue() {
   return GTK_ADJUSTMENT(Adj)->value;
}

/// @return in a string the value of this adjustment
char*fr_Adjustment::GetFormattedValue() {
   sprintf(Printed, Format, GetValue());
   return Printed;
}

/// Reset the default value of this adjustment
void fr_Adjustment::ResetDefault(float DV) {
   bool WasDefault = IsDefault();
   bool WasEnabled = Enabled;
   Enabled = true;
   DefaultValue = DV;
   if (WasDefault) SetToDefault();
   Enabled = WasEnabled;
}

/// Set the value of this adjustment to its default value
void fr_Adjustment::SetToDefault() {
   SetValue(DefaultValue);
}

/// @return true if this adjustment is at its default value
bool fr_Adjustment::IsDefault() {
   if(ABS(GetValue() - DefaultValue) < Step)
     return true;
   return false;
}

/// Assign a tool-tip for this adjustment
void fr_Adjustment::SetTooltip(char*T) {
   char ToolStr[256], S[80], S2[80];
   
   if(T) strcpy(ToolStr, T);
   else  strcpy(ToolStr, "");
   
   char *arg = Args.GetPreferredArg();   
   if(arg) {
      sprintf(S, "\noption: %s <num>", arg);
      strcat(ToolStr, S);
   };
   if(Format) {
      sprintf(S2, "\nDefault: %s ", Format);
      sprintf(S, S2, DefaultValue);
      strcat(ToolStr, S);
      sprintf(S2, "\nRange: [ %s, %s ]", Format, Format);
      sprintf(S, S2, Minimum, Maximum);
      strcat(ToolStr, S);
   };
   if(KeyStroke) {
      sprintf(S, "\nkeystroke: %s", KeyStroke);
      strcat(ToolStr, S);
   };
   CreateTooltip(ToolStr);   
}

/// Scan ArgList L for args that belong to this asjustment
void fr_Adjustment::SiftArgs(fr_ArgList& L) {
   int m = Args.MatchArgs(L) + 1;
   if (m<1) return;
   char *S = L[m];
   if(S) {
      L.Mark(m);
      SetValue(atof(S));
   } else
     cerr << "Missing numeric value after \"" << L[m-1] << '"' << endl;
}

/// Put in ArgList L args that reflect the current value of this adjustment
void fr_Adjustment::CompileArgs(fr_ArgList& L) {
   if (IsDefault()) return;
   L << Args.GetPreferredArg() << GetFormattedValue();
}

void fr_Adjustment::AddListener(fr_Listener*L) {
   if(!Listeners)
     gtk_signal_connect(GTK_OBJECT(Adj), "value_changed",
			GTK_SIGNAL_FUNC(fr_Listener::Changed), this);
   fr_Element::AddListener(L);
}

/* ############################# fr_Slider ################################ */
/**
 * Create a new slider with a label, direction, default value, min, max,
 * number of decimal places to use, and whether or not to display a value
 * next to the slider
 */
fr_Slider::fr_Slider(fr_Element*parent, char*label, fr_Direction dir,
		     float DV, float min, float max, int DP, bool ShowDigits):
fr_Adjustment(parent, label, DV, min, max+1, DP)
{
   Maximum--;
   if(dir==fr_Horizontal)
     GUI = Element = gtk_hscale_new(GTK_ADJUSTMENT(Adj));
   else
     GUI = Element = gtk_vscale_new(GTK_ADJUSTMENT(Adj));
   gtk_scale_set_digits(GTK_SCALE(Element), DP);
   gtk_scale_set_draw_value(GTK_SCALE(Element), ShowDigits);
   AddLabel(label);
   SetVisibility(true);
}

/* ############################# fr_NumEntry ############################## */
/**
 * Create a new text entry area for a numeric value, with a default value,
 * min & max values, and number of decimal places to use.
 */
fr_NumEntry::fr_NumEntry(fr_Element*parent, char*label,
			 float DV, float min, float max, int DP):
fr_Adjustment(parent, label, DV, min, max, DP)
{
   GUI = Element = gtk_spin_button_new(GTK_ADJUSTMENT(Adj), 0, DP);
   AddLabel(label);
   SetVisibility(true);
}

/* ############################# fr_Text ################################## */
/**
 * Create a one line text box, with a label, default string,
 * Case sensitive value for matching, and maximum length for value
 */
fr_Text::fr_Text(fr_Element*parent, char*label,
		 char*DefaultStr, bool Case, int MaxLen):
fr_Option(parent, label)
{
   GUI = Element = gtk_entry_new_with_max_length(MaxLen);
   DefaultText = DefaultStr;
   CaseSensitive = Case; 
   if((label)&&(label[0])) {
      Name = label;
      AddLabel(label);
   };
   SetVisibility(true);
}

/// Set the string in this text element
void fr_Text::SetText(char*S) {
   if (Enabled) gtk_entry_set_text(GTK_ENTRY(Element), S);
}

/// @return the text for this text entry
char*fr_Text::GetText() {
   return gtk_entry_get_text(GTK_ENTRY(Element));
}

/// Select the text in this text entry
void fr_Text::SelectText() {
   gtk_entry_select_region(GTK_ENTRY(Element), 0, strlen(GetText()));
}

/// Set the text field to its default value
void fr_Text::SetToDefault() {
   gtk_entry_set_text(GTK_ENTRY(Element), DefaultText);
}

/// @return true if the current text matches the default text
bool fr_Text::IsDefault() {
   if(CaseSensitive)
     return (strcmp(GetText(), DefaultText) == 0);
   return (strcasecmp(GetText(), DefaultText) == 0);
}

/// Scan ArgList L for args that belong to this text entry
void fr_Text::SiftArgs(fr_ArgList& L) {
   int m = Args.MatchArgs(L) + 1;
   if (m<1) return;
   char *S = L[m];
   if(S) {
      L.Mark(m);
      SetText(S);
   } else
     cerr << "Missing text data after \"" << L[m-1] << '"' << endl;
}

/// Put in ArgList L args that reflect the value of this text entry
void fr_Text::CompileArgs(fr_ArgList& L) {
   char *S = GetText();
   if((!S)||(!S[0])) return;
   L << Args.GetPreferredArg() << S;
}

void fr_Text::AddListener(fr_Listener*L) {
   if(!Listeners) {
      gtk_signal_connect(GTK_OBJECT(Element), "activate",
			 GTK_SIGNAL_FUNC(fr_Listener::Click), this);
      gtk_signal_connect(GTK_OBJECT(Element), "changed",
			 GTK_SIGNAL_FUNC(fr_Listener::Changed), this);
   }
   fr_Element::AddListener(L);
}

/* ############################# fr_DataTable ############################# */
fr_DataTable::fr_DataTable(fr_Element*parent, int cols, bool sngl):
fr_Element(parent),
Rows(0), Columns(cols)
{
   Element = gtk_clist_new(Columns);
   GUI = gtk_scrolled_window_new(0, 0);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(GUI),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);
   gtk_container_add(GTK_CONTAINER(GUI), Element);
   SetSingle(sngl);
   gtk_clist_set_shadow_type(GTK_CLIST(Element), GTK_SHADOW_IN);
   SetVisibility(true);
}

void fr_DataTable::SetHeaders(char**Headers) {
   for(int c=0; c<Columns && Headers[0]; c++)
     gtk_clist_set_column_title(GTK_CLIST(Element), c, Headers[c]);
   gtk_clist_column_titles_show(GTK_CLIST(Element));
}

void fr_DataTable::SetSortColumn(int col) {
   if((col<0)||(col>=Columns)) return;
   SortColumn = col;
}

void fr_DataTable::SetSortOnInsert(bool s, bool ascend) {
   gtk_clist_set_sort_type(GTK_CLIST(Element),
			   ascend?GTK_SORT_ASCENDING:GTK_SORT_DESCENDING);
   gtk_clist_set_auto_sort(GTK_CLIST(Element), s);
}

int fr_DataTable::AddRow(char**RowData, int row) {
   Rows++;
   int r;
   if(row>=0)
     r = gtk_clist_insert(GTK_CLIST(Element), row, RowData);
   else
     r = gtk_clist_append(GTK_CLIST(Element), RowData);
   return r;
}

void fr_DataTable::RemoveRow(int row) {
   gtk_clist_remove(GTK_CLIST(Element), row);
   Rows--;
}

void fr_DataTable::AssociateData(int row, void*data) {
   gtk_clist_set_row_data(GTK_CLIST(Element), row, data);
}

void*fr_DataTable::AssociatedData(int row) {
   return gtk_clist_get_row_data(GTK_CLIST(Element), row);
}

void fr_DataTable::Deselect(int row) {
   gtk_clist_unselect_row(GTK_CLIST(Element), row, 0);
}

void fr_DataTable::DeselectAll() {
#if ((GTK_MAJOR_VERSION > 1) || (GTK_MINOR_VERSION > 1))
   gtk_clist_unselect_all(GTK_CLIST(Element));
#else
   for(int i=0; i<Rows; i++)
     gtk_clist_unselect_row(GTK_CLIST(clist), i, 0);
#endif
}

void fr_DataTable::Select(int row) {
   gtk_clist_select_row(GTK_CLIST(Element), row, 0);
}

char*fr_DataTable::GetCell(int row, int col) {
   char *t;
   gtk_clist_get_text(GTK_CLIST(Element), row, col, &t);
   return t;
}

fr_Image* fr_DataTable::GetCellPic(int row, int col) {
   GdkPixmap *p;
   GdkBitmap *b;
   gtk_clist_get_pixmap(GTK_CLIST(Element), row, col, &p, &b);
   if(!p) return (fr_Image*)0;
   
   set<fr_Image*>::iterator i, e = ImageCollection.end();
   for(i=ImageCollection.begin(); i!=e; i++)
     if((*i)->xpixmap == p)
       return *i;
   return (fr_Image*)0;
}
 
void fr_DataTable::SetCell(int row, int col, char*txt,
			   fr_Image*I, bool showboth) {
   if(!I)
     gtk_clist_set_text(GTK_CLIST(Element), row, col, txt);
   else if(!showboth)
     gtk_clist_set_pixmap(GTK_CLIST(Element), row, col,
			  I->xpixmap, I->xbitmap);
   else
     gtk_clist_set_pixtext(GTK_CLIST(Element), row, col, txt, 3,
			   I->xpixmap, I->xbitmap);
   if(I)
     ImageCollection.insert(I);
}

void fr_DataTable::AllowReorder(bool s) {
   gtk_clist_set_reorderable(GTK_CLIST(Element), s);
}

void fr_DataTable::SetRowHeight(int h) {
   gtk_clist_set_row_height(GTK_CLIST(Element), h);
}

void fr_DataTable::SetColumnWidth(int col, int w) {
   if((col<0)||(col>=Columns)) return;
   if(w<0)
     gtk_clist_set_column_auto_resize(GTK_CLIST(Element), col, true);
   else
     gtk_clist_set_column_width(GTK_CLIST(Element), col, w);
}

void fr_DataTable::SetColumnAlign(int col, Alignment a) {
   switch(a) {
    case Left:
      gtk_clist_set_column_justification(GTK_CLIST(Element), col,
						  GTK_JUSTIFY_LEFT);
      return;
    case Right:
      gtk_clist_set_column_justification(GTK_CLIST(Element), col,
						  GTK_JUSTIFY_RIGHT);
      return;
    case Center:
      gtk_clist_set_column_justification(GTK_CLIST(Element), col,
						  GTK_JUSTIFY_CENTER);
      return;      
   }
}

void fr_DataTable::Freeze(bool f) {
   if(f) gtk_clist_freeze(GTK_CLIST(Element));
   else  gtk_clist_thaw(GTK_CLIST(Element));
}

int fr_DataTable::CountSelectedRows() {
   return g_list_length(GTK_CLIST(Element)->selection);
}

int fr_DataTable::GetSelected(int nth) {
   int *rp = (int*)g_list_nth(GTK_CLIST(Element)->selection, nth);
   if (rp) return *rp;
   return -1;
}

bool fr_DataTable::IsRowSelected(int row) {
   int *r;
   for(int i=CountSelectedRows(); i>0; --i) {
      r = (int*)g_list_nth(GTK_CLIST(Element)->selection, i);
      if((*r)==row)
	return true;
   }
   return false;
}

void fr_DataTable::ShowRow(int row) {
    gtk_clist_moveto(GTK_CLIST(Element), row, 0, 0.0, 0.0);
}

void fr_DataTable::RemoveAll() {
   gtk_clist_clear(GTK_CLIST(Element));
}

void fr_DataTable::ReplaceRow(char**RowData, int row) {
   for(int c=Columns; c; --c)
     gtk_clist_set_text(GTK_CLIST(Element), row, c, RowData[c]);
}

void fr_DataTable::SetSingle(bool s) {
   gtk_clist_set_selection_mode(GTK_CLIST(Element),
				s ? GTK_SELECTION_SINGLE
				  : GTK_SELECTION_EXTENDED);
}

void fr_DataTable::Sort(bool ascending) {
#if ((GTK_MAJOR_VERSION > 1) || (GTK_MINOR_VERSION > 1))
   gtk_clist_set_sort_column(GTK_CLIST(Element), SortColumn);
   gtk_clist_set_sort_type(GTK_CLIST(Element), ascending?
			   GTK_SORT_ASCENDING:GTK_SORT_DESCENDING);
   gtk_clist_sort(GTK_CLIST(Element));
#endif
}

void fr_DataTable::AddListener(fr_Listener*L) {
   if(!Listeners) {
      gtk_signal_connect(GTK_OBJECT(Element), "select_row",
			 GTK_SIGNAL_FUNC(EventRowSelected), this);
      gtk_signal_connect(GTK_OBJECT(Element), "unselect_row",
			 GTK_SIGNAL_FUNC(EventRowUnselected), this);
      gtk_signal_connect(GTK_OBJECT(Element), "click_column",
			 GTK_SIGNAL_FUNC(EventColSelected), this);
   }
   fr_Element::AddListener(L);
}

void fr_DataTable::EventRowSelected(GtkWidget*W, int row, int col,
				    void*event, fr_DataTable*D) {
   fr_Event(D, fr_Select, row); 
   if((event)&&(((GdkEventButton*)event)->type==GDK_2BUTTON_PRESS))
     fr_Event(D, fr_DoubleClick, row);
}

void fr_DataTable::EventRowUnselected(GtkWidget*W, int row, int col,
				      void*, fr_DataTable*D) {
   fr_Event(D, fr_Unselect, row);
}

void fr_DataTable::EventColSelected(GtkWidget*W, int col, fr_DataTable*D) {
   fr_Event(D, fr_Click, col);
}

/* ############################# fr_File ################################## */
/// Create a new file selection box with a browse button
fr_File::fr_File(fr_Element*parent, char*title, char*ENVvar, char*Path,
		 bool dironly, char**XPM):
fr_Option(parent, title),
BrowseButton(this, "..."),
OK(this, "Ok"),
Cancel(this, "Cancel"),
DirOnly(dironly)
{
   Element = gtk_entry_new();
   gtk_widget_show(Element);

   DefaultPath = g_strdup(Path);
   DefaultEnv = g_strdup(ENVvar);
   xpixmap = XPM;
   if(title)
     BrowseStr = g_strdup_printf("%s", title);
   else BrowseStr = title;
#ifdef FR_GNOME
   gtk_widget_destroy(BrowseButton.getGUI());
   GtkWidget *BrowsePixmap = gnome_stock_pixmap_widget(Window,
						       GNOME_STOCK_MENU_OPEN);
   gtk_widget_show(BrowsePixmap);
   BrowseButton.GUI = BrowseButton.Element = gtk_button_new();
   gtk_container_add(GTK_CONTAINER(BrowseButton.GUI), BrowsePixmap);
   BrowseButton.SetVisibility(true);
#endif
   BrowseButton.SetTooltip(BrowseStr);
   BrowseButton.AddListener(this);

   FileSelect = 0;
	
   OK.AddListener(this);
   Cancel.AddListener(this);

   GUI = gtk_table_new(1, 5, false);
   gtk_table_attach_defaults(GTK_TABLE(GUI), Element, 0,4,0,1);
   gtk_table_attach(GTK_TABLE(GUI), BrowseButton.getGUI(), 4,5,0,1, GTK_SHRINK, GTK_SHRINK, 3, 2);
   gtk_widget_show(GUI);
   
   SetVisibility(true);
}

fr_File::~fr_File() {
   g_free(DefaultPath);
   g_free(DefaultEnv);
   if(BrowseStr) g_free(BrowseStr);
}

/**
 * Get the default path to use in the file text entry.
 * If the DefaultEnv environtment variable is set, it will use that,
 * if not, it will use DefaultPath
 */
char*fr_File::GetDefaultPath() {
   char *E;
   if((DefaultEnv)&&(DefaultEnv[0]!=0))
     E = getenv(DefaultEnv);
   else
     E = (char*)0;
   if(E) return E;
   return DefaultPath;
}

/// @return true if the file selected is the default path
bool fr_File::IsDefault() {
   return (strcmp(GetFileName(), DefaultPath)==0);
}

/// Set the file selected to the default path
void fr_File::SetToDefault() {
   SetFileName(DefaultPath);
}

/// Bring up a file browser for navigating a file system to find a file
void fr_File::FilePopup() {
   GdkPixmap *Pixmap;
   GdkBitmap *mask;

   if ((FileSelect)||(!Enabled)) return;
   FileSelect = gtk_file_selection_new(BrowseStr?BrowseStr:"Browse");
   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(FileSelect)->ok_button),
		      "clicked", GTK_SIGNAL_FUNC(fr_Listener::Click), &OK);
   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(FileSelect)->cancel_button),
		      "clicked", GTK_SIGNAL_FUNC(fr_Listener::Click), &Cancel);
   gtk_signal_connect(GTK_OBJECT(FileSelect),
		      "destroy", GTK_SIGNAL_FUNC(fr_Listener::Click), &Cancel);
   char *FN = GetFileName();
   if ((!FN)||(FN[0] == 0)) FN = GetDefaultPath();
   gtk_file_selection_set_filename(GTK_FILE_SELECTION(FileSelect), FN);
   gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(FileSelect));
   SetEditable(false);	//disable editing of Element & BrowseButton

   GtkWidget *pixwidget;
#ifdef FR_GNOME
   if(!DirOnly || !xpixmap) {
      pixwidget = gnome_stock_pixmap_widget(Window, GNOME_STOCK_MENU_OPEN);
      gtk_widget_show(pixwidget);
      gtk_clist_set_column_widget(GTK_CLIST(GTK_FILE_SELECTION(FileSelect)->dir_list),
				  0, pixwidget);
   }
#endif
   Pixmap = 0;
   GtkStyle *Style = gtk_widget_get_default_style();
   if(xpixmap)
     Pixmap = gdk_pixmap_create_from_xpm_d(Window->window, &mask,
					   &Style->bg[GTK_STATE_NORMAL],
					   (char**)xpixmap);
#ifdef FR_GNOME
   else if(!DirOnly)
     gnome_stock_pixmap_gdk(GNOME_STOCK_MENU_NEW, 0, &Pixmap, &mask);
#endif
   if(Pixmap) {
      pixwidget = gtk_pixmap_new(Pixmap, mask);
      gtk_widget_show(pixwidget);
      if(DirOnly)
	gtk_clist_set_column_widget(GTK_CLIST(GTK_FILE_SELECTION(FileSelect)->dir_list),
				    0, pixwidget);
      else
	gtk_clist_set_column_widget(GTK_CLIST(GTK_FILE_SELECTION(FileSelect)->file_list),
				    0, pixwidget);
   };
   if(DirOnly)
     gtk_widget_set_sensitive(GTK_WIDGET(GTK_CLIST(GTK_FILE_SELECTION(FileSelect)->file_list)),
			      false);
   gtk_widget_show(FileSelect);
   if (Pixmap) gdk_window_set_icon(FileSelect->window, 0, Pixmap, mask);
}

/// Callback for when the OK button in ::PopUp is pressed
void fr_File::FileOK() {
   SetFileName(gtk_file_selection_get_filename(GTK_FILE_SELECTION(FileSelect)));
   gtk_widget_grab_focus(Element);
   gtk_widget_destroy(FileSelect);
   FileSelect=0;
   SetEditable(true);
}

/// Callback for when the Cancel button in ::Popup is pressed
void fr_File::FileCanceled() {
   gtk_widget_destroy(FileSelect);
   FileSelect=0;
   SetEditable(true);
}

/// Set the file name to use
void fr_File::SetFileName(char*filename) {
   if(Enabled) gtk_entry_set_text(GTK_ENTRY(Element), filename);
}

/// @return the selected file name
char*fr_File::GetFileName() {
   return gtk_entry_get_text(GTK_ENTRY(Element));
}

/// Scan ArgList L for args that belong to this file
void fr_File::SiftArgs(fr_ArgList& L) {
   int m = Args.MatchArgs(L) + 1;
   if(m<1) return;
   char *S = L[m];
   if(S) {
      L.Mark(m);
      SetFileName(S);
   } else
     cerr << "Missing filename after \"" << L[m-1] << '"' << endl;
}

/// Add to ArgList L args that reflect the selected file
void fr_File::CompileArgs(fr_ArgList& L) {
   if(IsDefault()) return;
   L << Args.GetPreferredArg() << GetFileName();
}

/// Handle events from the ::Popup file browser
void fr_File::EventOccurred(fr_Event*e) {
   if     (e->Is(BrowseButton)) FilePopup();
   else if(e->Is(OK))		FileOK();
   else if(e->Is(Cancel)) 	FileCanceled();
}

/* ############################# fr_Notebook ############################## */
/// Create a new notebook
fr_Notebook::fr_Notebook(fr_Element*parent, char*name):
fr_Box(parent, name)
{
   GUI = Element = gtk_notebook_new();
   SetVisibility(true);
}

/// Add a page to this notebook
void fr_Notebook::AddPage(fr_Notepage& P) {
   gtk_container_border_width(GTK_CONTAINER(P.getGUI()), (PadX+PadY)/2);
   gtk_notebook_append_page(GTK_NOTEBOOK(Element), P.getGUI(), P.Label->getGUI());
}

/**
 * if b is true, the notetabs will be scrollable if there are too many
 * to fit across the top of the notebook, otherwise the notebook is
 * expanded to accomodate the tabs
 */
void fr_Notebook::SetScrollable(bool b) {
   gtk_notebook_set_scrollable(GTK_NOTEBOOK(Element), b);
}

/* ############################# fr_Notepage ############################## */
/// Create a new Notepage
fr_Notepage::fr_Notepage(fr_Element*parent, char*TabLabel):
fr_Box(parent, TabLabel)
{
   if(TabLabel) Label = new fr_Label(this, TabLabel);
}

fr_Notepage::~fr_Notepage() {
   if(Label) delete Label;
}

// ############################# fr_Window ################################ //
void fr_Window::WindowInit() {
   MenuBar = gtk_menu_bar_new();
   MenuHandle = gtk_handle_box_new();
   gtk_container_add(GTK_CONTAINER(MenuHandle), MenuBar);
   Vbox = gtk_vbox_new(0, 0);
   gtk_widget_show(Vbox);
   gtk_box_pack_start(GTK_BOX(Vbox), MenuHandle, false, false, 0);
}

/// Create a new window
fr_Window::fr_Window(fr_Element*parent, char*Title, char**XPM, char*IconLabel):
fr_Box(parent, Title)
{
   WindowInit();
   Window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gtk_widget_realize(Window);
   gtk_window_set_policy(GTK_WINDOW(Window), false, true, false);
   gtk_container_add(GTK_CONTAINER(Window), Vbox);
   SetName(Title);
   if (XPM) SetIcon(IconLabel?IconLabel:Title, XPM);
}

fr_Window::~fr_Window() {
   if(Window) gtk_widget_destroy(Window);
}

/// Set the table size for this window
void fr_Window::SetGridSize(int x, int y, bool SameSize) {
   fr_Box::SetGridSize(x, y, SameSize);
   gtk_box_pack_start(GTK_BOX(Vbox), GUI, false, false, 0);
}

/**
 * Name this window.
 * The name will also be used by the window manager
 */
void fr_Window::SetName(char*name) {
   fr_Element::SetName(name);
   gtk_window_set_title(GTK_WINDOW(Window), name);
}

/**
 * Assign an icon and icon Label to this window
 * Some window managers use the icon, others ignore it
 */
void fr_Window::SetIcon(char*IconLabel, char**XPM) {
   GdkPixmap *Pixmap;
   GdkBitmap *mask;
   if(IconLabel)
     gdk_window_set_icon_name(Window->window, IconLabel);
   if(XPM) {
      GtkStyle *Style = gtk_widget_get_default_style();
      Pixmap = gdk_pixmap_create_from_xpm_d(Window->window, &mask,
					    &Style->bg[GTK_STATE_NORMAL], (char**)XPM);
      gdk_window_set_icon(Window->window, 0, Pixmap, mask);
   };
}

/// Add a menu bar to the top of this window
void fr_Window::AddMenu(char*label, fr_MenuBarItem Items[]) {
   GtkWidget *Menu, *MenuItem, *MenuLabel;
   
   Menu = gtk_menu_new();
   for(int i=0; Items[i].Label; i++) {
      if(strcmp(Items[i].Label, "-")) {
	 if(Items[i].Icon > 0) {
	    MenuItem = gtk_menu_item_new();
	    GtkWidget *hbox = gtk_hbox_new(false, false);
	    GtkWidget *pix = gtk_pixmap_new(Items[i].Icon->xpixmap,
					    Items[i].Icon->xbitmap);
	    gtk_widget_show(pix);
	    gtk_box_pack_start(GTK_BOX(hbox), pix, false, false, 1);
	    GtkWidget *mlbl = gtk_label_new(Items[i].Label);
	    gtk_widget_show(mlbl);
	    gtk_box_pack_start(GTK_BOX(hbox), mlbl, false, false, 1);
	    gtk_widget_show(hbox);
	    gtk_container_add(GTK_CONTAINER(MenuItem), hbox);
	 } else
	      MenuItem = gtk_menu_item_new_with_label(Items[i].Label);
      } else {
	 MenuItem = gtk_menu_item_new();
	 gtk_widget_set_sensitive(MenuItem, false);
      };
      gtk_widget_show(MenuItem);
      gtk_menu_append(GTK_MENU(Menu), MenuItem);
      if(Items[i].Element)
	gtk_signal_connect(GTK_OBJECT(MenuItem), "activate",
			   GTK_SIGNAL_FUNC(fr_Listener::MenuClick),
			   Items[i].Element);
   };
   MenuItem = gtk_menu_item_new();
   MenuLabel = gtk_label_new(label);
   gtk_widget_show(MenuLabel);
   gtk_container_add(GTK_CONTAINER(MenuItem), MenuLabel);
   gtk_widget_show(MenuItem);
   gtk_menu_item_set_submenu(GTK_MENU_ITEM(MenuItem), Menu);
   if(strcmp(label, "Help")==0)
     gtk_menu_item_right_justify(GTK_MENU_ITEM(MenuItem));
   gtk_menu_bar_append(GTK_MENU_BAR(MenuBar), MenuItem);
   gtk_widget_show(MenuBar);
   gtk_widget_show(MenuHandle);
}

void fr_Window::ApplyStyle(fr_Style*S) {
   //fr_Element::ApplyStyle(S);
   if(S) {
      gtk_widget_set_style(Window, S->Style);
      //gtk_widget_set_style(Vbox, S->Style);
   } else {
      gtk_widget_restore_default_style(Window);
      //gtk_widget_restore_default_style(Vbox);
   }
}

/// Show/Hide this window
void fr_Window::SetVisibility(bool s) {
   fr_Element::SetVisibility(s);
   gtk_window_set_policy(GTK_WINDOW(Window), false, true, true);
   if(s) {
      gtk_widget_show(Window);
      gdk_window_raise(Window->window);
   } else
     gtk_widget_hide(Window);
}

void fr_Window::AddListener(fr_Listener*L) {
   if(!Listeners) {
      gtk_signal_connect(GTK_OBJECT(Window), "destroy",
			 GTK_SIGNAL_FUNC(EventDestroy), this);
      gtk_signal_connect(GTK_OBJECT(Window), "delete_event",
			 GTK_SIGNAL_FUNC(EventDelete), this);
   };
   fr_Element::AddListener(L);
}

void fr_Window::EventDestroy(GtkObject*, fr_Window*W) {
   fr_Event(W, fr_Destroy);
}

void fr_Window::EventDelete(GtkObject*, GdkEvent*, fr_Window*W) {
   fr_Event(W, fr_Destroy);
}


/* ############################# fr_About ################################# */
/// Create an about box for this application
fr_About::fr_About(fr_Element*parent, char*ver, char*copyright,
		   const char*authors[], char*comments, char*logo):
fr_Element(parent, "About"),
Version(ver), Copyright(copyright), Comments(comments), Logo(logo),
Authors(authors)
{
}

/// Show the About dialog
void  fr_About::Splash() {
#ifdef FR_GNOME
   Window = GUI = Element = gnome_about_new(Parent->GetName(), Version,
					    Copyright, Authors,
					    Comments, Logo);
#else
#define NewLabel(l, j, b) \
     w = gtk_label_new(l);\
   gtk_widget_show(w);\
   gtk_label_set_justify(GTK_LABEL(w), j);\
   gtk_box_pack_start(GTK_BOX(b), w, false, true, 3);
	
   char NameVersion[80];
   GtkWidget *w, *f, *box1, *box2;
   
   Window = gtk_dialog_new();
   gtk_window_set_title(GTK_WINDOW(Window), "About");
   gtk_signal_connect(GTK_OBJECT(Window), "destroy",
		      GTK_SIGNAL_FUNC(gtk_widget_destroy), Window);
   gtk_signal_connect(GTK_OBJECT(Window), "delete_event",
		      GTK_SIGNAL_FUNC(gtk_widget_destroy), Window);
   gtk_widget_realize(Window);
	
   GUI = GTK_DIALOG(Window)->vbox;
   f = gtk_frame_new(0); gtk_widget_show(f);
   gtk_frame_set_shadow_type(GTK_FRAME(f), GTK_SHADOW_IN);
   gtk_box_pack_start(GTK_BOX(GUI), f, true, true, 3);
   box1 = gtk_vbox_new(0, 0); gtk_widget_show(box1);
   gtk_container_add(GTK_CONTAINER(f), box1);
   f = gtk_frame_new(0); gtk_widget_show(f);
   gtk_frame_set_shadow_type(GTK_FRAME(f), GTK_SHADOW_IN);
   gtk_box_pack_start(GTK_BOX(GUI), f, true, true, 3);
   box2 = gtk_vbox_new(0, 0); gtk_widget_show(box2);
   gtk_container_add(GTK_CONTAINER(f), box2);

   sprintf(NameVersion, "%s %s", Parent->GetName(), Version);
   NewLabel(NameVersion, GTK_JUSTIFY_CENTER, box1);
   NewLabel(Copyright, GTK_JUSTIFY_LEFT, box2);
   NewLabel((Authors[1]?"Authors:":"Author:"), GTK_JUSTIFY_LEFT, box2);
   for(int i=0; Authors[i]; i++) {
      NewLabel(Authors[i], GTK_JUSTIFY_RIGHT, box2);
   };
   NewLabel(Comments, GTK_JUSTIFY_LEFT, box2);

   Element = gtk_button_new_with_label("Close");
   gtk_signal_connect_object(GTK_OBJECT(Element), "clicked",
			     GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(Window));
   gtk_widget_show(Element);
   GTK_WIDGET_SET_FLAGS(Element, GTK_CAN_DEFAULT);
   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Window)->action_area), Element,
		      true, true, 2);

   gtk_container_border_width(GTK_CONTAINER(GUI), 10);
   gtk_widget_show(Window);
   gtk_widget_grab_focus(Element);
   gtk_widget_grab_default(Element);
#undef NewLabel
#endif
   SetVisibility(true);
}

/* ############################# fr_MainProgram ########################### */
/**
 * Set up the main program.
 * This class should be inherited, and used in main().
 */
fr_MainProgram::fr_MainProgram(char*name, char**XPM):
fr_Window((fr_Element*)0, name, XPM)
{
   Executable = (char*)0;
}

/// Set the path/name of the executable this front end is for.
void fr_MainProgram::SetExecutable(char*exe) {
   Executable = exe;
}

/**
 * exit the main loop of the interface
 * (this is usually followed by fr_exec())
 */
void fr_MainProgram::Run() {
   gtk_main_quit();
}

/// exit the front end program without doing anything
void fr_MainProgram::Exit(int ExitCode) {
   gtk_exit(ExitCode);
}


/* ############################# Misc. Functions ########################## */
/// Give a message to the user of this program.
void fr_Mesg(const char*Message) {
   GtkWidget *Window;
#ifdef FR_GNOME
   Window = gnome_message_box_new(Message, GNOME_MESSAGE_BOX_GENERIC,
				  GNOME_STOCK_BUTTON_CLOSE, 0);
   gnome_dialog_set_close(GNOME_DIALOG(Window), true);
#else
   GtkWidget *Label, *Button;
   Window = gtk_dialog_new();
   gtk_window_set_title(GTK_WINDOW(Window), "Message");
   gtk_container_border_width(GTK_CONTAINER(GTK_DIALOG(Window)->vbox),
			      10);
   Label = gtk_label_new(Message);
   gtk_widget_show(Label);
   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Window)->vbox), Label,
		      true, true, 0);
   Button = gtk_button_new_with_label("Close");
   GTK_WIDGET_SET_FLAGS(Button, GTK_CAN_DEFAULT);
   gtk_box_pack_start(GTK_BOX(GTK_DIALOG(Window)->action_area), Button,
		      true, true, 0);
   gtk_signal_connect_object(GTK_OBJECT(Button), "clicked",
			     GTK_SIGNAL_FUNC(gtk_widget_destroy),
			     GTK_OBJECT(Window));
   gtk_widget_show(Button);
#endif
   gtk_signal_connect(GTK_OBJECT(Window), "destroy",
		      GTK_SIGNAL_FUNC(gtk_widget_destroy), Window);
   gtk_signal_connect(GTK_OBJECT(Window), "delete_event",
		      GTK_SIGNAL_FUNC(gtk_widget_destroy), Window);
#ifndef FR_GNOME
   gtk_widget_grab_focus(Button);
   gtk_widget_grab_default(Button);
#endif
   gtk_widget_show(Window);
}

/// @return true if a file or directory specified by Path exists
bool fr_Exists(char*Path) {
   struct stat StatBuf; 
   if((!Path)||(Path[0]==0)) return 0;
   return (stat(Path, &StatBuf)==0);
}

/// @return true if Dir is an existing directory
bool fr_DirExists(char*Dir) {
   struct stat StatBuf;
   if((!Dir)||(Dir[0]==0)) return 0;
   if(stat(Dir, &StatBuf))
     return false;
   return (StatBuf.st_mode & S_IFDIR);
}

/// @return true if Filename is an existing file
bool fr_FileExists(char*Filename) {
   struct stat StatBuf; 
   if((!Filename)||(Filename[0]==0)) return false;
   if(stat(Filename, &StatBuf))
     return false;
   return (StatBuf.st_mode & S_IFREG);
}

/// return a pointer to a string with the name of the Home Directory
char*fr_HomeDir() {
   return g_get_home_dir();
}

/// do any unfinished GUI business
void fr_Flush() {
   while(gtk_events_pending())
     gtk_main_iteration();
}

void fr_exit(GtkObject*O, int*exitcode) {
   gtk_exit(*exitcode);
}

void fr_exit_event(GtkObject*O, GdkEvent*event, int*exitcode) {
   gtk_exit(*exitcode);
}

/// Initialize the GUI for this application
fr_ArgList& fr_init(char*AppName, int*argc, char**argv[]) {
   char *RCfile, *Home;
   fr_ArgList *ExecArgs;
	
   if(!AppName)
     AppName = g_basename(*(argv[0]));
   fr_AppName = AppName;
   Home = g_get_home_dir();
   RCfile = g_strdup_printf("%s/.%s.gtkrc",
			    Home?Home:".", AppName?AppName:"frend");
#ifdef FR_GNOME
   gnome_init(AppName, 0, 1, *argv);
#else
   gtk_init(argc, argv);
#endif
   gtk_rc_parse(RCfile);
   if(*argc > 1) ExecArgs = new fr_ArgList((*argc) - 1, (*argv)+1);
   else          ExecArgs = new fr_ArgList();

   g_free(RCfile);
   return *ExecArgs;
}

/// Everything is set up, hand control over to the GUI
fr_ArgList& fr_main(fr_MainProgram*Program, fr_ArgList& CmdArgs) {
   Program->SetToDefaults();
   Program->SiftArgs(CmdArgs);

   gtk_main();

   fr_ArgList *ExecArgs = new fr_ArgList();
   *ExecArgs << Program->GetExecutable();
   ExecArgs->Mark(0);
   Program->CompileArgs(*ExecArgs);
	
   return *ExecArgs;
}

/**
 * using the args collected from the main program,
 * execute the thing that this is a front-end for
 */
int fr_exec(fr_ArgList& ExecArgs) {
   char **Args;

   Args = ExecArgs.GetArgs();

   if(Args[0]) {
      cout << "execvp(\"" << Args[0] << "\", {";
      for(int i=1; Args[i]; i++)
	cout << '"' << Args[i] << '"' << Args[i+1]?", ":"";
      cout << "});" << endl;
      return execvp(Args[0], Args);
   } else
     cerr << "frend: No executable has been specified!; aborting" << endl;
   
   cerr << "frend: " << '"' << Args[0] << '"';
   cerr << "isn't in the PATH; aborting." << endl;
   return -1;
}
