/* ------------------------------------------------------------------------
 * client.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-19 by Steve Houston.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000, 2001 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <iostream>
#include <stdlib.h>
#include <time.h>

// -- 3Dwm Includes
#include <Nobel++/Nobel.hh>
#include <Nobel/Shape.hh>
#include <Nobel/Image.hh>
#include <Nobel/Mapper.hh>
#include <Nobel/Desktop.hh>
#include <Nobel/NodeKit.hh>
#include <Nobel/Appearance.hh>
#include <Nobel/GeometryKit.hh>
#include <Nobel/TextureKit.hh>
#include <Nobel/TriangleGeometry.hh>
#include <Nobel/TextureGenerator.hh>

using namespace Nobel;

// -- Local Function Prototypes

static void usage(const char *binary, const char *msg);
static void createTexture(const char* filename, const char* port, 
			  const char* passwd, Node_ptr root, 
			  NodeKit_ptr node_kit, GeometryKit_ptr geo_kit,
			  TextureKit_ptr tex_kit);

// -- Code Segment

int main(int argc, char *argv[]) {
    
    try {
	
	// Get the filename from the command line
        if (argc < 4)
            usage(argv[0], " <server> <port> <password>");

	// Initialize the connection
	Nobelxx::Client client("VNC Client", argc, argv);
        
	// Resolve the necessary kits
	NodeKit_var node_kit =
	    client.resolve<NodeKit>(Nobelxx::name<NodeKit>());
	GeometryKit_var geo_kit =
	    client.resolve<GeometryKit>(Nobelxx::name<GeometryKit>());
	TextureKit_var tex_kit = 
	    client.resolve<TextureKit>(Nobelxx::name<TextureKit>());

	// And here's the scene graph root (local to the client)
	Node_var root = client.root();
	
	// Build the windowing system texture
	createTexture(argv[1], argv[2], argv[3], root, node_kit, 
		      geo_kit, tex_kit);
    }
    catch (const CORBA::Exception &e) {

	// The ORB threw an exception
	std::cerr << "Uncaught CORBA exception: " << e << std::endl;
	return EXIT_FAILURE;
    }
    catch (...) {
	
	// Another exception was raised
	std::cerr << "Exception thrown. Exiting." << std::endl;
	return EXIT_FAILURE;
    }
    
    return EXIT_SUCCESS;
}

void usage(const char *binary, const char *msg)
{
    std::cerr << "USAGE: " << binary << msg << std::endl;
    throw 0;
}

void createTexture(const char *server, const char *port, const char* passwd,
		   Node_ptr root, NodeKit_ptr node_kit,
		   GeometryKit_ptr geo_kit, TextureKit_ptr tex_kit)
{
    // Create the triangle shape for the 
    Shape_var shape = node_kit->createShape();
    TriangleGeometry_var geo = geo_kit->createTriangleGeometry();

    // Connect to the VNC desktop
    Desktop_var desktop = node_kit->createVNCDesktop(server, atoi(port),
						     passwd);
    DesktopController_var controller = desktop->createController(shape);

    // Create a planar mapper to use in the controller
    Mapper_var mapper = tex_kit->createPlanarMapper();
    mapper->scale(Nobelxx::Vertex3D(0.2, 0.2, 1.0));
    controller->setMapper(mapper);
    
    // Retrieve and link the texture generator to our procedural texture
    TextureGenerator_var texGen = desktop->getTextureGenerator();
    ProceduralTexture_var tex = tex_kit->createProceduralTexture();
    tex->setTextureGenerator(texGen);
    
    // Link this texture to the appearance object
    Appearance_var appear = geo_kit->createAppearance();
    appear->setTexture(tex);
    
    // Set the appearance into the shape
    shape->setAppearance(appear);
    
    // Initialise the vertex list
    Vertex3D vertex;
    geo->setVertexNumber(4);

    #define XSCALE 10.0f
    #define YSCALE 10.0f

    vertex.x = -0.5f * XSCALE;
    vertex.y = -0.5f * YSCALE;
    vertex.z =  0.0f;
    geo->setVertex(0, vertex);

    vertex.x =  0.5f * XSCALE;
    vertex.y = -0.5f * YSCALE;
    vertex.z =  0.0f;
    geo->setVertex(1, vertex);

    vertex.x =  0.5f * XSCALE;
    vertex.y =  0.5f * YSCALE;
    vertex.z =  0.0f;
    geo->setVertex(2, vertex);

    vertex.x = -0.5f * XSCALE;
    vertex.y =  0.5f * YSCALE;
    vertex.z =  0.0f;
    geo->setVertex(3, vertex);

    // Initialise the tex coordinate list
    TexCoord tc;
    geo->setTexCoordNumber(4);

    tc.u = 0.0f;
    tc.v = 1.0f;
    geo->setTexCoord(0, tc);

    tc.u = 1.0f;
    tc.v = 1.0f;
    geo->setTexCoord(1, tc);

    tc.u = 1.0f;
    tc.v = 0.0f;
    geo->setTexCoord(2, tc);

    tc.u = 0.0f;
    tc.v = 0.0f;
    geo->setTexCoord(3, tc);

    // Set the indices
    geo->setFaceNumber(2);
    Triangle face, texCoordFace;

    face.a = 0;
    face.b = 1;
    face.c = 3;
    texCoordFace.a = 0;
    texCoordFace.b = 1;
    texCoordFace.c = 3;
    geo->setVertexIndex(0, face);
    geo->setTexCoordIndex(0, texCoordFace);

    face.a = 1;
    face.b = 2;
    face.c = 3;
    texCoordFace.a = 1;
    texCoordFace.b = 2;
    texCoordFace.c = 3;
    geo->setVertexIndex(1, face);
    geo->setTexCoordIndex(1, texCoordFace);

    // Add geometry to shape and shape to root
    shape->addGeometry(geo);
    //root->insert(shape);
    root->insert(controller);
}
