; UBDD Library
; Copyright (C) 2008-2011 Warren Hunt and Bob Boyer
; Significantly revised in 2008 by Jared Davis and Sol Swords.
; Now maintained by Centaur Technology.
;
; Contact:
;   Centaur Technology Formal Verification Group
;   7600-C N. Capital of Texas Highway, Suite 300, Austin, TX 78731, USA.
;   http://www.centtech.com/
;
; This program is free software; you can redistribute it and/or modify it under
; the terms of the GNU General Public License as published by the Free Software
; Foundation; either version 2 of the License, or (at your option) any later
; version.  This program is distributed in the hope that it will be useful but
; WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
; more details.  You should have received a copy of the GNU General Public
; License along with this program; if not, write to the Free Software
; Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.

; core.lisp - fundamental operations, e.g., normp, eval-bdd, q-not, and q-ite.

(in-package "ACL2")
(include-book "../misc/memory-mgmt-logic")
(include-book "misc/computed-hint-rewrite" :dir :system)
(include-book "tools/rulesets" :dir :system)
(include-book "cutil/defsection" :dir :system)

(defdoc ubdds
  ":Doc-Section ubdds
   BDD-like operations implemented with hons.~/

   UBDDs provide a concise and efficient way to represent binary
   decision diagrams.

   The valid UBDDs are a subset of purely-Boolean cons-trees, i.e.,
   trees where all the tips are ~c[t] or ~c[nil].  Unlike most BDD
   packages, the internal nodes are unlabelled, and the variable order
   is determined by the depth from the root; thus, the name UBDD for
   'Unlabeled BDD'.  Also, we do not reduce internal nodes unless both
   outgoing edges point to the same constant.

   For example, here are all of the valid one-variable UBDDs:

     - ~c[t] or ~c[nil]   for \"var is irrelevant to this decision\", or

     - ~c[(cons t nil)]   for \"if ~c[var] then ~c[t], else ~c[nil]\", or

     - ~c[(cons nil t)]   for \"if ~c[var] then ~c[nil], else ~c[t].

   But ~c[(cons t t)] and ~c[(cons nil nil)] are not permitted; they
   are reduced to ~c[t] and ~c[nil], respectively.  The valid UBDDs
   are recognized with the function ~il[normp], and means a cons-tree
   in T and NIL without any subtree that is '(t . t) or '(nil . nil).

   By 'cons-tree in T and NIL' we mean the intersection of all sets S
   such that:

      1.  T is a member of S,
      2.  NIL is a member of S, and
      3.  for all x and y, if x and y are in S, then (CONS x y) is in S.

   The fancy 'intersection' operation above may seem obtuse, but it is
   a standard mathematical device whereby we can speak very precisely,
   and whereby we get only the 'finitely generated' cons-trees.  Most
   textbooks would add some extra, unnecessary, warm and fuzzy rule
   such as 'the only cons-trees are those generated by rules 1, 2, and
   3.'  Such a rule is an atrocity of both English and mathematics.

   Legend:  Rules of thumb for naming of functions.

    A function whose name begins q-  returns an ubdd.
    A function whose name beqins qv- returns a list of ubdds.
    A function whose name begins qs- is a set operation on ubdds.

   If, especially in the name of a function, we write 'bdd', we mean
   'ubdd'.

   This file defines a number of UBDD-related operations.  BDDs
   generally have three characteristics: each path from the root to a
   tip encounters each variable at most once, every path encounters
   variables in pre-specified order, and no pair of outgoing edges
   point to the same node (reduced).  Our UBDD definition differs from
   ROBDDs in several ways:  we 'reduce' internal nodes unless both
   outgoing edges of a UBDD node point to the same constant, UBDDs
   'nodes' do not include a variable name, the 'variable' associated
   with a ITE node corresponds to its depth from the UBDD root.  As a
   result, we do not need to store the variable name in the UBDD
   nodes.  We do reduce our UBDDs in the sense that constant values
   terminate any path.

   The QCAR, QCDR, QCONS, QNOT, and Q-ITE provide the complete
   definition of our UBDD system.  The order of the UBDD variables is
   implicit -- there are no names, just the depth from the root.  Thus,
   a UBDD with only one variable can be either the reduced values T or
   NIL, or it can be, (HONS T NIL), or (HONS NIL T); (HONS T T) and
   (HONS NIL NIL) are not permitted, but reduced to T and NIL,
   respectively (see the definition of NORMP below).

   ~/

   For these definitions to serve as an effective BDD system, we
   memoize the functions ~il[q-not] and ~il[q-ite].")

;;; THE CORE UBDD OPERATIONS

(defun ubddp (x)

  ":Doc-Section ubdds

   Recognizer for well-formed UBDDs~/~/ ~c[(UBDDP x)] is our
   recognizer for well-formed UBDDs.  All of the tips should be
   booleans, and no internal node should have the form ~c[(t . t)] or
   ~c[(nil . nil)].

   Theorem.  (EQUAL (UBDDP x) (NORMP x))

   (NORMP x) returns T or NIL according to whether X is a well-formed
   ubdd, i.e., a rooted, binary tree, in T and NIL, with no node equal
   to '(T . T) or '(NIL . NIL).

   To say it another way, just for repetition, (UBDDP x) returns T or
   NIL according to whether x is a finitely generated CONS tree (a)
   whose tips are always T and/or NIL and (b) which has no subtree
   that is (T . T) or (NIL . NIL).  T and NIL are UBDDPs.  Neither (T
   . T) nor (NIL . NIL) is a UBDDP.  (T . NIL) and (NIL . T) are
   UDBDDs, as is ((NIL . T) . T), ((NIL . T) . (T . NIL)) and so
   forth.

   For efficiency, nonatomic UBDDs are usually represented in ACL2 as
   hash-conses, i.e., HONSPs, 'under-the-hood' or 'behind the scenes'.
   However, this is not expressible as a requirement in the ACL2
   logic, and it is certainly not a requirement in any sense.

   Theorem.  (UBDDP (CONS T NIL)).

   Theorem.  (UBDDP (HONS T NIL)).

   Though some call UBDDPs with the name 'NORMP', UBDDP is better.
   Calling something 'normal' in mathematics is ridiculous, as John
   Kelley so well points out in his 'General Topology', because so
   many weird things have already been defined as 'normal'
   somethingsorother."

  (declare (xargs :guard t))
  (mbe :logic (if (atom x)
                  (booleanp x)
                (and (ubddp (car x))
                     (ubddp (cdr x))
                     (if (atom (car x))
                         (not (equal (car x) (cdr x)))
                       t)))
       :exec (cond ((eq x t) t)
                   ((eq x nil) t)
                   ((atom x) nil)
                   (t (let ((a (car x))
                            (d (cdr x)))
                        (cond ((eq a t)
                               (cond ((eq d nil) t)
                                     ((eq d t) nil)
                                     (t (ubddp d))))
                              ((eq a nil)
                               (cond ((eq d nil) nil)
                                     ((eq d t) t)
                                     (t (ubddp d))))
                              (t (and (ubddp a) (ubddp d)))))))))

(defund normp (x)

  ":Doc-Section ubdds

   Recognizer for well-formed UBDDs~/~/ ~c[(normp x)] is our
   recognizer for well-formed UBDDs.  All of the tips should be
   booleans, and no internal node should have the form ~c[(t . t)] or
   ~c[(nil . nil)].

   (NORMP x) returns T or NIL according to whether X is a well-formed
   ubdd, i.e., a rooted, binary tree, in T and NIL, with no node equal
   to '(T . T) or '(NIL . NIL).

   Theorem.  (EQUAL (UBDDP x) (NORMP x))

   To say it another way, just for repetition, (UBDDP x) returns T or
   NIL according to whether x is a finitely generated CONS tree (a)
   whose tips are always T and/or NIL and (b) which has no subtree
   that is (T . T) or (NIL . NIL).  T and NIL are UBDDPs.  Neither (T
   . T) nor (NIL . NIL) is a UBDDP.  (T . NIL) and (NIL . T) are
   UDBDDs, as is ((NIL . T) . T), ((NIL . T) . (T . NIL)) and so
   forth.

   For efficiency, nonatomic UBDDs are usually represented in ACL2 as
   hash-conses, i.e., HONSPs, 'under-the-hood' or 'behind the scenes'.
   However, this is not expressible as a requirement in the ACL2
   logic, and it is certainly not a requirement in any sense.

   Theorem.  (UBDDP (CONS T NIL)).

   Theorem.  (UBDDP (HONS T NIL)).

   Though some call UBDDPs with the name 'NORMP', UBDDP is better.
   Calling something 'normal' in mathematics is ridiculous, as John
   Kelley so well points out in his 'General Topology', because so
   many weird things have already been defined as 'normal'
   somethingsorother."


  (declare (xargs :guard t))
  (mbe :logic (if (atom x)
                  (booleanp x)
                (and (normp (car x))
                     (normp (cdr x))
                     (if (atom (car x))
                         (not (equal (car x) (cdr x)))
                       t)))
       :exec (cond ((eq x t) t)
                   ((eq x nil) t)
                   ((atom x) nil)
                   (t (let ((a (car x))
                            (d (cdr x)))
                        (cond ((eq a t)
                               (cond ((eq d nil) t)
                                     ((eq d t) nil)
                                     (t (normp d))))
                              ((eq a nil)
                               (cond ((eq d nil) nil)
                                     ((eq d t) t)
                                     (t (normp d))))
                              (t (and (normp a) (normp d)))))))))

(memoize 'normp :condition '(consp x))

(defn ubdd-listp (l)
  (if (atom l)
      (null l)
    (and (normp (car l))
         (ubdd-listp (cdr l)))))

(defund eval-bdd (x values)

  ":Doc-Section ubdds
   Evaluates a UBDD object w.r.t. a list of binary values.~/

   General form:
   ~bv[]
     (eval-bdd x values)
   ~ev[]
   Typically ~c[x] should be a valid ~il[normp] UBDD and ~c[values]
   should be a ~c[boolean-listp].~/

   If ~c[x] is an atom, then ~c[x] is its own 'value' as in ~il[qcar]
   and ~il[qcdr].  Otherwise, we use the ~c[(car values)] to guide us
   further through ~c[x].

   We also pretend that ~c[values] contains an infinite list of ~c[nil]s
   at the end.  So if we run out of ~c[values], we follow the false
   branches of ~c[x] all the way to its tips.

   (EVAL-BDD x values) is the 'value' of X with respect to VALUES.

   X is normally a CONS tree of Booleans and VALUES is normally a
   TRUE-LISTP of Booleans, i.e., Ts and NILs.  (Typically, X is T,
   NIL, or a 'HONSP' NORMP.)  Of course, since EVAL-BDD's guard is T,
   it can be given any two ACL2 objects as arguments.

   If X is an atom, then X is its own 'value'; otherwise, we use the
   CAR and CDR of VALUES, say A and D, to guide us further through X.
   (If VALUES is an atom, we use NIL for both A and D.)  If A is NIL
   the answer is the value of (CDR X) with respect to D; otherwise the
   answer is the value of (CAR X) with respect to D.

   One can think of the VALUES argument to EVAL-BDD as having its last
   atom replaced with an infinite list of NILs."

  (declare (xargs :guard t))
  (if (atom x)
      (if x t nil) ;; Fixes result to a Boolean.
    (if (atom values)
        (eval-bdd (cdr x) nil)
      (if (car values)
          (eval-bdd (car x) (cdr values))
        (eval-bdd (cdr x) (cdr values))))))

(defn qcar-fn (x)
  (if (consp x) (car x) x))

(defmacro qcar (x)
  ":Doc-Section ubdds
   The ``true branch'' of a UBDD.~/~/
   For a compound UBDD, i.e., ~c[(cons x y)], we simply return ~c[x].  But for
   the atomic UBDDs ~c[t] and ~c[nil], this is the identity function.

   Hence, ~c[t] represents an infinite tree of ~c[t]s, and ~c[nil] represents an
   infinite tree of ~c[nil]s."
  `(qcar-fn ,x))

(defn qcdr-fn (x)
  (if (consp x) (cdr x) x))

(defmacro qcdr (x)
  ":Doc-Section ubdds
   The ``false branch'' of a UBDD.~/~/
   For a compound UBDD, i.e., ~c[(cons x y)], we simply return ~c[y].  But for
   atomic the UBDDs ~c[t] and ~c[nil], this is the identity function.

   Hence, ~c[t] represents an infinite tree of ~c[t]s, and ~c[nil] represents an
   infinite tree of ~c[nil]s."
  `(qcdr-fn ,x))

(defn qcons-fn (x y)
  ":Doc-Section ubdds
   Constructor for UBDDs.~/~/
   ~c[(qcons-fn x y)] constructs a UBDD whose true-branch is ~c[x] and whose
   false-branch is ~c[y].  It handles any collapsing which needs to occur."
  (if (if (eq x t)
          (eq y t)
        (and (eq x nil) (eq y nil)))
      x
    (hons x y)))

(defmacro qcons (x y)
  `(qcons-fn ,x ,y))

(defund q-not (x)
  (declare (xargs :guard t))
  ":Doc-Section ubdds
   Negate a UBDD.~/~/
   ~c[(q-not x)] builds a new UBDD representing ~c[(not x)].
   That is, the following is a theorem:
   ~bv[]
      (equal (eval-bdd (q-not x) values)
             (not (eval-bdd x values)))
   ~ev[]"
  (if (atom x)
      (if x nil t)
    (hons (q-not (car x))
          (q-not (cdr x)))))

(memoize 'q-not :condition
         '(and (consp x)
               (or (consp (car x))
                   (consp (cdr x)))))

(defund q-ite-fn (x y z)

  ":Doc-Section ubdds
   Perform a UBDD-based if-then-else.~/~/
   ~c[(q-ite-fn x y z)] builds a new UBDD representing ~c[(if x y z)].
   That is, the following is a theorem:
   ~bv[]
     (equal (eval-bdd (q-ite-fn x y z) values)
            (if (eval-bdd x values)
                (eval-bdd y values)
              (eval-bdd z values)))
   ~ev[]
   Normally you will want to use q-ite instead, which is a macro which is
   logically the same as q-ite-fn, but expands into a lazy call of q-ite-fn
   which can avoid the need to evaluate y and z."

  (declare (xargs :guard t))
  (cond ((null x) z)
        ((atom x) y)
        (t (let ((y (if (hons-equal x y) t y))
                 (z (if (hons-equal x z) nil z)))
             (cond ((hons-equal y z) y)
                   ((and (eq y t) (eq z nil)) x)
                   ;; Calling Q-NOT is meant to improve efficiency.
                   ((and (eq y nil) (eq z t)) (q-not x))
                   (t (qcons (q-ite-fn (car x) (qcar y) (qcar z))
                             (q-ite-fn (cdr x) (qcdr y) (qcdr z)))))))))

#||
(memoize 'q-ite-fn :condition
         '(and (consp x)
               (consp (car x))
               (consp (cdr x))
               (or (and (consp y) (consp (car y)) (consp (cdr y)))
                   (and (consp z) (consp (car z)) (consp (cdr z))))))
||#

; The above condition doesn't memoize Q-ITE-FN calls that could have
; an enormous test structure, but with the test structure being
; (HONS <atom> <huge-structure>) or (HONS <huge-structure> <atom>).

(memoize 'q-ite-fn :condition  ;; Changed to this on 9 July 2009
         '(and (consp x)
               (or (consp (car x))
                   (consp (cdr x)))))

(defmacro q-ite (x y z)

  ":Doc-Section ubdds
   An optimized way to call ~c[q-ite-fn].  (macro) ~/

   ~c[(q-ite x y z)] is a macro that is logically equal to Q-ITE-FN.
   But in the :exec world, special measures are taken so that we can
   sometimes avoid evaluating y or z.~/

   At a first approximation, ~c[(q-ite x y z)] expands into

   ~bv[]
     (let ((<x> x))
       (cond ((null <x>) z)
             ((atom <x>) y)
             (t
              (let* ((<y> y)
                     (<z> z))
                (q-ite-fn <x> <y> <z>)))))
   ~ev[]

   And the advantage of this over a raw call of Q-ITE-FN is that when
   the expression for x evaluates to an atom, then we can avoid
   evaluating y or z.  Hence, when y or z is expensive to
   evaluate (e.g., perhaps they are terms involving calls to
   UBDD-building operations like Q-NOT), then this can save some time.

   Of course, if neither y nor z is expensive to evaluate, then the
   above just adds some minor overhead to each call of Q-ITE-FN.
   Sometimes we can avoid this overhead by recognizing that Y and Z
   are cheap to evaluate.  In particular, we claim that constants and
   variable symbols are always cheap to evaluate, and so if both Y and
   Z are either constants or variables, then we do not bother to
   introduce the LET statement above and instead we just lay down a
   raw call of Q-ITE-FN.

   (Q-ITE x y z) expects three UBDDs, which are to be interpreted at
   the same level.  Informally speaking. Q-ITE returns a single UBDD,
   also at the same level, that is 'equivalent' to (IF x y z).  The two
   theorems Q-ITE-CORRECT and NORMP-Q-ITE express formally what Q-ITE
   returns."

  (cond ((and (or (quotep y) (atom y))
              (or (quotep z) (atom z)))
         `(q-ite-fn ,x ,y ,z))
        (t
         `(mbe :logic (q-ite-fn ,x ,y ,z)
               :exec (let ((q-ite-x-do-not-use-elsewhere ,x))
                       (cond
                        ((null q-ite-x-do-not-use-elsewhere) ,z)
                        ((atom q-ite-x-do-not-use-elsewhere) ,y)
                        (t
                         (let* ((q-ite-y-do-not-use-elsewhere ,y)
                                (q-ite-z-do-not-use-elsewhere ,z))
                           (prog2$
                            (last-chance-wash-memory)
                            (q-ite-fn q-ite-x-do-not-use-elsewhere
                                      q-ite-y-do-not-use-elsewhere
                                      q-ite-z-do-not-use-elsewhere))))))))))


;;; REASONING ABOUT THE CORE OPERATIONS




(defthm equal-of-booleans-rewrite    ; !!! Why is this rule here?
  (implies (and (booleanp x)
                (booleanp y))
           (equal (equal x y)
                  (iff x y)))
  :rule-classes ((:rewrite :backchain-limit-lst 1)))


(defsection eval-bdd

  (local (in-theory (enable eval-bdd)))

  (defthm eval-bdd-when-not-consp
    (implies (not (consp x))
             (equal (eval-bdd x values)
                    (if x t nil))))

  (defthm eval-bdd-of-nil
    (equal (eval-bdd nil values)
           nil))

  (defthm eval-bdd-of-t
    (equal (eval-bdd t values)
           t))

  (defthm eval-bdd-when-non-consp-values
    ;; !!! Could probably be better expressed as a congruence
    ;; rule for a LIST-EQUIV operation.
    (implies (and (syntaxp (not (equal vals *nil*)))
                  (atom vals))
             (equal (eval-bdd x vals)
                    (eval-bdd x nil)))))


(defsection equal-by-eval-bdds

  (defdoc equal-by-eval-bdds
    ":Doc-Section ubdds
     A strategy for proving two ~il[normp]s are equal.~/
     Suppose we know that ~c[x] and ~c[y] are ~c[normp] objects, and we are
     trying to prove they are equal.  The ~c[equal-by-eval-bdds] approach
     involves trying to establish this equality by proving the equivalent
     fact:
     ~bv[]
       (forall (values)

               ;; This looks formal, but it is all crap because there does
               ;; not exist a function symbol 'forall' of two arguments in
               ;; ACL2.  !!! Probably Jared can make this prose perfect

               (equal (eval-bdd x values)
                      (eval-bdd y values)))
     ~ev[]~/
     We use a computed hint to automatically invoke this strategy when we
     can determine (with rewriting) that ~c[x] and ~c[y] are ~c[normp]s.
     Actually, we can even assume that values has a certain length and is
     purely boolean, but usually these are not needed.

     Generally speaking, if you want to prove that (foo ...) and (bar ...)
     are equal BDDs, you should begin by proving the rules:

     ~bv[]
       (equal (eval-bdd (foo ...) values)
              (... expression 1 ...))

       (equal (eval-bdd (bar ...) values)
              (... expression 2 ...))
     ~ev[]

     Then ACL2 will be able to prove (equal (foo ...) (bar ...)) by just
     arguing that expression 1 and expression 2 are the same.")

  (defund max-depth (x)
    (declare (xargs :guard t))
    (if (atom x)
        0
      (1+ (max (max-depth (car x))
               (max-depth (cdr x))))))

  ;; This is like the trick in the osets book. We want to write the rule:
  ;;
  ;;    (normp x)
  ;;    (normp y)
  ;;    (forall (values)
  ;;            (equal (eval-bdd x values)
  ;;                   (eval-bdd y values)))
  ;;     ---->
  ;;    (equal x y)
  ;;
  ;; But this is not possible to do with a regular rewrite rule due to
  ;; the nested quantifier.  So we instead we begin by expressing our
  ;; hypotheses as a constraint about uninterpreted function symbols.

  (encapsulate
   (((bdd-hyps) => *)
    ((bdd-lhs)  => *)
    ((bdd-rhs)  => *))

   (local (defun bdd-hyps () nil))
   (local (defun bdd-lhs  () nil))
   (local (defun bdd-rhs  () nil))

   (defthmd bdd-equivalence-constraint
     (implies (and (bdd-hyps)
                   (equal (len arbitrary-values)
                          (max (max-depth (bdd-lhs))
                               (max-depth (bdd-rhs))))
                   (boolean-listp arbitrary-values)
                   (normp (bdd-lhs))
                   (normp (bdd-rhs)))
              (equal (eval-bdd (bdd-lhs) arbitrary-values)
                     (eval-bdd (bdd-rhs) arbitrary-values)))))


  ;; Now we use this constraint to prove a generic theorem that says (bdd-lhs)
  ;; and (bdd-rhs) are equal when the (bdd-hyps) hold.  This takes a bit of
  ;; work because of the weaker constraint we chose above.

  (local (defund find-conflicting-values (x y)
           (declare (xargs :measure (+ (acl2-count x) (acl2-count y))))
           (if (or (consp x)
                   (consp y))
               ;; At least one of them is a cons.  We descend both trees and try
               ;; to discover a path that will break their equality.
               (mv-let (car-successp car-path)
                 (find-conflicting-values (qcar x) (qcar y))
                 (if car-successp
                     ;; We took the "true" branch.
                     (mv t (cons t car-path))
                   (mv-let (cdr-successp cdr-path)
                     (find-conflicting-values (qcdr x) (qcdr y))
                     (if cdr-successp
                         (mv t (cons nil cdr-path))
                       (mv nil nil)))))
             ;; Otherwise, both are atoms.  If they are equal, then we have failed to
             ;; find a conflicting path.  But if they are not equal, then this path
             ;; violates their success.
             (mv (not (equal x y)) nil))))

  (local (defthm mv-nth-1
           (equal (mv-nth 1 x)
                  (cadr x))
           :hints(("Goal" :in-theory (enable mv-nth)))))

  (local (defthm lemma1
           (implies (and (normp x)
                         (normp y)
                         (car (find-conflicting-values x y)))
                    (equal (equal (eval-bdd x (cadr (find-conflicting-values x y)))
                                  (eval-bdd y (cadr (find-conflicting-values x y))))
                           nil))
           :hints(("Goal" :in-theory (enable (:induction find-conflicting-values)
                                             eval-bdd normp)
                   :induct (find-conflicting-values x y)
                   :expand ((find-conflicting-values x y))))))

  (local (defthm lemma2
           (implies (and (normp x)
                         (normp y))
                    (equal (car (find-conflicting-values x y))
                           (not (equal x y))))
           :hints(("Goal" :in-theory (enable (:induction find-conflicting-values)
                                              normp)
                   :induct (find-conflicting-values x y)
                   :expand ((find-conflicting-values x y))))))

  (local (defthm lemma3
           (<= (len (cadr (find-conflicting-values x y)))
               (max (max-depth x) (max-depth y)))
           :hints(("Goal" :in-theory (enable (:induction find-conflicting-values)
                                              normp)
                   :induct (find-conflicting-values x y)
                   :expand ((find-conflicting-values x y)
                            (max-depth x) (max-depth y))))
           :rule-classes (:linear :rewrite)))

  (local (defthm lemma4
           (boolean-listp (cadr (find-conflicting-values x y)))
           :hints(("Goal" :in-theory (enable find-conflicting-values)))))

  (local (defun extend-to-len (lst n)
           (declare (xargs :guard t))
           (if (zp (nfix n))
               lst
             (if (atom lst)
                 (cons nil (extend-to-len lst (1- n)))
               (cons (car lst) (extend-to-len (cdr lst) (1- n)))))))

  (local (defthm extend-to-len-of-cons
           (equal (extend-to-len (cons a b) n)
                  (cons a (extend-to-len b (1- n))))))

  (local (defun eval-extend-induct (x lst n)
           (if (zp (nfix n))
               (cons lst x)
             (if (atom lst)
                 (list (eval-extend-induct (car x) lst (1- n))
                       (eval-extend-induct (cdr x) lst (1- n)))
               (list (eval-extend-induct (car x) (cdr lst) (1- n))
                     (eval-extend-induct (cdr x) (cdr lst) (1- n)))))))

  (local (defthm eval-bdd-of-extend
           (equal (eval-bdd x (extend-to-len lst n))
                  (eval-bdd x lst))
           :hints (("goal" :in-theory (enable eval-bdd)
                    :induct (eval-extend-induct x lst n)))))

  (local (defthm len-of-extend
           (equal (len (extend-to-len lst n))
                  (max (len lst) (nfix n)))))

  (local (defthm boolean-listp-of-extend
           (implies (boolean-listp lst)
                    (boolean-listp (extend-to-len lst n)))))

  ;; Finally, here is the theorem we are going to functionally instantiate.

  (defthm equal-by-eval-bdds
    (implies (and (bdd-hyps)
                  (normp (bdd-lhs))
                  (normp (bdd-rhs)))
             (equal (equal (bdd-lhs) (bdd-rhs))
                    t))
    :hints(("Goal"
            :use ((:instance bdd-equivalence-constraint
                             (arbitrary-values (extend-to-len
                                                (cadr (find-conflicting-values
                                                       (bdd-lhs) (bdd-rhs)))
                                                (max (max-depth (bdd-lhs))
                                                     (max-depth (bdd-rhs))))))))))


  ;; We set up a computed hint which directs ACL2 to try to functionally
  ;; instantiate the above theorem when it is trying to prove equalities
  ;; between normp things.  As a help for this heuristic, we set up a table of
  ;; "bdd functions" --- things that are supposed to produce BDDs.  When we are
  ;; trying to prove such BDD functions are equal, we'll give the hint.

  ;; BOZO we may want a more flexible notion here that allows us to pattern
  ;; match against (MV-NTH 1 (FOO ...)) and so on, so that a particular return
  ;; value is a BDD we can handle it.  Consider using the untranslate-patterns
  ;; book's rewriter pattern-matcher in term-is-bdd.

  (defmacro bdd-fns ()
    '(cdr (assoc 'bdd-fns (table-alist 'bdd-fns world))))

  (defmacro set-bdd-fns (val)
    `(table bdd-fns 'bdd-fns ,val))

  (defmacro add-bdd-fn (val)
    `(set-bdd-fns (cons ',val (bdd-fns))))

  (defmacro add-bdd-fns (val)
    `(set-bdd-fns (append ',val (bdd-fns))))

  (set-bdd-fns nil)



  ;; What follows is nasty ACL2 magic.  You probably don't want to try to
  ;; understand it unless you need to.

  (defun term-is-bdd (term hyps whole hist pspv world ctx state)
    ;; This function decides whether we think that TERM is a normp.  If it is a
    ;; variable, we ask the rewriter if it's known to be a normp.  If it's a
    ;; quoted constant, we see if its value is a normp.  And if it's a function
    ;; call, we see if it is one of the functions we explicitly mentioned in
    ;; bdd-fns.
    (declare (xargs :mode :program :stobjs state))
    (cond ((atom term)
           (er-let* ((term-and-ttree
                      (computed-hint-rewrite
                       `(normp ,term) hyps t whole hist pspv world ctx state)))
                    (value (equal (car term-and-ttree) *t*))))
          ((eq (car term) 'quote)
           ;; !!! Note to Sol: Jared changed this to normp from booleanp.
           (value (normp (cadr term))))
          (t
           (value (member-eq (car term) (bdd-fns))))))

  (defun equal-by-eval-bdds-hint-fn (id clause hist pspv ctx stable world state)
    (declare (xargs :mode :program :stobjs state)
             (ignorable id))
    (if (not stable)
        ;; Don't suggest anything until the goal is stable.  This gives other
        ;; rules a chance to fire.
        (value nil)
      (let* ((rcnst (access prove-spec-var pspv :rewrite-constant))
             (ens   (access rewrite-constant rcnst :current-enabled-structure)))
        (if (not (enabled-numep (fnume '(:rewrite equal-by-eval-bdds) world) ens))
            ;; Don't suggest anything if equal-by-eval-bdds is
            ;; disabled.  This gives the user an easy way to turn off
            ;; the strategy.
            (value nil)
          (let ((conclusion (car (last clause))))
            (if (not (and (consp conclusion)
                          (or (eq (car conclusion) 'equal)
                              (eq (car conclusion) 'not))))
                (value nil)
              ;; We look for conclusions of the form (equal lhs rhs)
              ;; or (not lhs).  We think of (not lhs) as the same as
              ;; (equal lhs nil).
              (let ((lhs  (second conclusion))
                    (rhs  (or (third conclusion) *nil*))
                    (hyps (dumb-negate-lit-lst (butlast clause 1))))
                ;; We only want to apply the hint if we believe lhs
                ;; and rhs are normp objects.
                (mv-let (erp lhs-okp state)
                  (term-is-bdd lhs hyps clause hist pspv world ctx state)
                  (if (or erp (not lhs-okp))
                      (value nil)
                    (mv-let (erp rhs-okp state)
                      (term-is-bdd rhs hyps clause hist pspv world ctx state)
                      (if (or erp (not rhs-okp))
                          (value nil)
                        ;; We think LHS and RHS are normp's.  Go ahead
                        ;; and give the hint.
                        (let ((hint `(:use (:functional-instance
                                            equal-by-eval-bdds
                                            (bdd-hyps (lambda () (and ,@hyps)))
                                            (bdd-lhs  (lambda () ,lhs))
                                            (bdd-rhs  (lambda () ,rhs))))))
                          (prog2$
                           ;; And tell the user what's happening.
                           (cw "~|~%We now appeal to ~s3 in an ~
                                attempt to show that ~x0 and ~x1 ~
                                are equal because all of their ~
                                evaluations under ~s2 are the ~
                                same.  (You can disable ~s3 to ~
                                avoid this.  See :doc ~s3 for more ~
                                details.)~|~%"
                               (untranslate lhs nil world)
                               (untranslate rhs nil world)
                               'eval-bdd
                               'equal-by-eval-bdds)
                           (value hint))))))))))))))

  (defmacro equal-by-eval-bdds-hint ()
    `(equal-by-eval-bdds-hint-fn id clause hist pspv ctx
                                 stable-under-simplificationp world state))

  (add-default-hints!
   '((equal-by-eval-bdds-hint)))



  ;; Experimental "heavy" hint.  Sometimes it is difficult to make use of
  ;; equalities between BDDs when carrying out the pick-a-point proof.
  ;; QS-SUBSET-MODE is one way to approach this.  And the HEAVY hint below is
  ;; another.  The idea is to replace other equalities between BDDs with what
  ;; they mean in terms of the arbitrary-values which have just been picked.

  (defun lit-find-equality (lit)
    (case-match lit
      (('not ('normp . &) . &)
       (mv nil nil nil))
      (('normp . &)
       (mv nil nil nil))
      (('not ('equal a b . &) . &)
       (mv 'ineq a b))
      (('not a . &)
       (mv 'eq a *nil*))
      (('equal a b . &)
       (mv 'eq a b))
      (& (mv 'ineq lit *nil*))))

  (defun collect-normp-eq-and-ineqs (clause whole hyps hist pspv ctx world state)
    (declare (xargs :mode :program :stobjs state))
    (if (atom clause)
        (mv nil nil nil state)
      (mv-let (eqs ineqs remhyps state)
        (collect-normp-eq-and-ineqs (cdr clause) whole hyps hist pspv ctx world
                                    state)
        (let ((conclusion (car clause)))
          (mv-let (sense lhs rhs)
            (lit-find-equality conclusion)
            (mv-let (erp lhs-okp state)
              (term-is-bdd lhs hyps whole hist pspv world ctx state)
             (if (or erp (not lhs-okp))
                 (mv eqs ineqs (cons (dumb-negate-lit conclusion) remhyps) state)
               (mv-let (erp rhs-okp state)
                 (term-is-bdd rhs hyps whole hist pspv world ctx state)
                 (if (or erp (not rhs-okp))
                     (mv eqs ineqs (cons (dumb-negate-lit conclusion) remhyps) state)
                   (case sense
                     (eq    (mv (cons (cons lhs rhs) eqs) ineqs
                                (cons (dumb-negate-lit conclusion) remhyps)
                                state))
                     (ineq  (mv eqs (cons (cons lhs rhs) ineqs) remhyps state))
                     (t     (mv eqs ineqs
                                (cons (dumb-negate-lit conclusion) remhyps)
                                state))))))))))))


  (defun collect-eq-hyps (eq-hyps acc)
    (declare (xargs :mode :program))
    (if (atom eq-hyps)
        acc
      (collect-eq-hyps
       (cdr eq-hyps)
       (cons `(equal (eval-bdd ,(caar eq-hyps) arbitrary-values)
                     (eval-bdd ,(cdar eq-hyps) arbitrary-values))
             acc))))

  (defun collect-hints (eqs hyps)
    (declare (xargs :mode :program))
    (if (atom eqs)
        nil
      (cons `(:use (:functional-instance equal-by-eval-bdds
                                         (bdd-hyps (lambda () (and . ,hyps)))
                                         (bdd-lhs  (lambda () ,(caar eqs)))
                                         (bdd-rhs  (lambda () ,(cdar eqs)))))
            (collect-hints (cdr eqs) hyps))))

  (defun equal-by-eval-bdds-hint-heavy-fn (id clause hist pspv ctx stable world state)
    (declare (xargs :mode :program :stobjs state)
             (ignorable id))
    (if (not stable)
        ;; Don't suggest anything until the goal is stable.  This
        ;; gives other rules a chance to fire.
        (value nil)
      (let* ((rcnst (access prove-spec-var pspv :rewrite-constant))
             (ens   (access rewrite-constant rcnst :current-enabled-structure)))
        (if (not (enabled-numep (fnume '(:rewrite equal-by-eval-bdds) world) ens))
            ;; Don't suggest anything if equal-by-eval-bdds is
            ;; disabled.  This gives the user an easy way to turn off
            ;; the strategy.
            (value nil)
          (let ((hyps (dumb-negate-lit-lst clause)))
            (mv-let (eqs ineqs new-hyps state)
              (collect-normp-eq-and-ineqs clause clause hyps
                                          hist pspv ctx world state)
              (if (not eqs)
                  (value nil)
                (let* ((new-hyps (collect-eq-hyps ineqs new-hyps))
                       (hints (collect-hints eqs new-hyps)))
                  (prog2$ nil ;; (cw "hyps: ~x0~%hints: ~x1~%" hyps hints)
                          (value (list :or hints)))))))))))

  (defmacro equal-by-eval-bdds-hint-heavy ()
    `(equal-by-eval-bdds-hint-heavy-fn
      id clause hist pspv ctx stable-under-simplificationp world state))


  (defun equal-by-eval-bdds-inst-fn (lhs rhs concl add-hyps
                                         instantiate-hyps clause hist pspv ctx
                                         world state)
    (declare (xargs :mode :program :stobjs state))
    (let ((clhyps (dumb-negate-lit-lst clause)))
      (mv-let (cleqs clineqs cl-new-hyps state)
        (collect-normp-eq-and-ineqs
         clause clause clhyps hist pspv ctx world state)
        (mv-let (lhs rhs state)
          (cond ((and lhs rhs)
                 (mv (car lhs) (car rhs) state))
                (concl
                 (er-let* ((concl (translate (car concl) t t t
                                             'equal-by-eval-bdds-inst world state)))
                   (mv-let (sense lhs rhs)
                     (lit-find-equality (car concl))
                     (declare (ignore sense))
                     (mv lhs rhs state))))
                (t (mv (caar (last cleqs)) (cdar (last cleqs)) state)))
          (let* ((hyps
                  (if (and instantiate-hyps (car instantiate-hyps))
                      (collect-eq-hyps clineqs cl-new-hyps)
                    clhyps))
                 (hyps
                  (if add-hyps
                      (append (car add-hyps) hyps)
                    hyps)))
            (let ((res `(:use (:functional-instance
                           equal-by-eval-bdds
                           (bdd-hyps (lambda () (and . ,hyps)))
                           (bdd-lhs  (lambda () ,lhs))
                           (bdd-rhs  (lambda () ,rhs))))))
              (prog2$ (cw "res: ~x0~%" res)
                      (value res))))))))

  (defmacro equal-by-eval-bdds-inst (&key (lhs 'nil lhsp)
                                          (rhs 'nil rhsp)
                                          (concl 'nil conclp)
                                          (add-hyps 'nil add-hypsp)
                                          (instantiate-hyps 'nil instantiate-hypsp))
    `(equal-by-eval-bdds-inst-fn ,(and lhsp `(list ,lhs))
                                 ,(and rhsp `(list ,rhs))
                                 ,(and conclp `(list ,concl))
                                 ,(and add-hypsp `(list ,add-hyps))
                                 ,(and instantiate-hypsp `(list ,instantiate-hyps))
                                 clause hist pspv ctx world state)))


(defsection q-not

  (add-bdd-fn q-not)

  (local (in-theory (enable q-not)))

  (defthm consp-of-q-not
    (equal (consp (q-not x))
           (consp x))
    :hints(("Goal" :in-theory (enable q-not))))

  (local (defthm lemma
           (implies (and (not (equal a t))
                         (normp a))
                    (iff (q-not a)
                         t))
           :hints(("Goal" :in-theory (enable normp)))))

  (local (defthm lemma2
           (implies (and a
                         (normp a))
                    (equal (equal t (q-not a))
                           nil))
           :hints(("Goal" :in-theory (enable normp)))))

  (defthm normp-of-q-not
    (implies (force (normp x))
             (equal (normp (q-not x))
                    t))
    :hints(("Goal" :in-theory (enable normp))))

  (defthm eval-bdd-of-q-not
    (equal (eval-bdd (q-not x) values)
           (not (eval-bdd x values)))
    :hints(("Goal" :in-theory (enable eval-bdd)))))




(defsection q-ite

  (add-bdd-fn q-ite-fn)
  ;; !!! Jared, Bob and Warren are now both lost.
  ;; What does the next line mean when Q-ITE is already defined?
  (add-macro-alias q-ite q-ite-fn)

  (local (in-theory (disable (:type-prescription q-ite-fn)
                             equal-of-booleans-rewrite
                             eval-bdd-when-not-consp
                             eval-bdd-when-non-consp-values)))

  (local (defun q-ite-induct (x y z)
           (declare (ignorable y z))
           (cond ((null x) nil)
                 ((atom x) nil)
                 (t (let ((y (if (equal x y) t y))
                          (z (if (equal x z) nil z)))
                      (list (q-ite-induct (car x) (qcar y) (qcar z))
                            (q-ite-induct (cdr x) (qcdr y) (qcdr z))))))))

  (defthm normp-of-q-ite
    (implies (and (force (normp x))
                  (force (normp y))
                  (force (normp z)))
             (equal (normp (q-ite x y z))
                    t))
    :hints(("Goal"
            :in-theory (e/d (normp) (default-car default-cdr))
            :induct (q-ite-induct x y z)
            :expand ((:free (y z) (q-ite-fn x y z))
                      (:free (y z) (q-ite-fn nil y z))
                      (:free (y z) (q-ite-fn t y z))))))

  (local (defun q-ite-induct-vals (x y z vals)
           (declare (ignorable y z))
           (cond ((null x) nil)
                 ((atom x) nil)
                 (t (let ((y (if (equal x y) t y))
                          (z (if (equal x z) nil z)))
                      (cond ((car vals)
                             (q-ite-induct-vals (car x) (qcar y) (qcar z) (cdr vals)))
                            (t
                             (q-ite-induct-vals (cdr x) (qcdr y) (qcdr z) (cdr vals)))))))))

  (with-output
   :off prove
   (defthm eval-bdd-of-q-ite
     (equal (eval-bdd (q-ite x y z) values)
            (if (eval-bdd x values)
                (eval-bdd y values)
              (eval-bdd z values)))
     :hints(("Goal"
             :in-theory (enable eval-bdd)
             :induct (q-ite-induct-vals x y z values)
             :expand ((:free (y z) (q-ite-fn x y z))
                      (:free (y z) (q-ite-fn nil y z))
                      (:free (y z) (q-ite-fn t y z))))))))


;; One reasoning strategy is to rewrite all BDD-building functions
;; into calls of q-ite.

(defthm canonicalize-q-not
  (implies (force (normp x))
           (equal (q-not x)
                  (q-ite x nil t)))
  :rule-classes :definition)

(def-ruleset canonicalize-to-q-ite '(canonicalize-q-not))


;; Once we get everything into a q-ite form, we'll often want to apply
;; some of the simple reductions you would expect.  The order of these
;; rules is important --- to avoid loops, you want the T and NIL cases
;; to come last.

;; !!! I think we should maybe be able to get some more of these
;; hypothesis free if we change q-ite-fn around a bit so that it
;; coerces atoms into booleans.  Would performance be okay?

(defthm |(q-ite x (q-ite y nil t) z)|
  (implies (and (syntaxp (not (equal z ''t))) ;; Prevents loops (see next rule)
                (force (normp x))
                (force (normp y))
                (force (normp z)))
           (equal (q-ite x (q-ite y nil t) z)
                  (q-ite y
                         (q-ite x nil z)
                         (q-ite x t z)))))

(defthm |(q-ite x (q-ite y nil t) t)|
  ;; ACL2's loop-stopper works.
  (implies (and (force (normp x))
                (force (normp y))
                (force (normp z)))
           (equal (q-ite x (q-ite y nil t) t)
                  (q-ite y (q-ite x nil t) t))))

(defthm |(q-ite (q-ite a b c) x y)|
  (implies (and (force (normp a))
                (force (normp b))
                (force (normp c))
                (force (normp x))
                (force (normp y)))
           (equal (q-ite (q-ite a b c) x y)
                  (q-ite a
                         (q-ite b x y)
                         (q-ite c x y)))))

(defthm |(q-ite x y y)|
  (equal (q-ite x y y)
         y)
  :hints(("Goal" :in-theory (enable q-ite))))

(defthm |(q-ite x x y)|
  (implies (and (force (normp x))
                (force (normp y)))
           (equal (q-ite x x y)
                  (q-ite x t y))))

(defthm |(q-ite x y x)|
  (equal (q-ite x y x)
         (q-ite x y nil))
  :hints(("Goal" :in-theory (e/d (q-ite) ((force))))))

(defthm |(q-ite x t nil)|
  (implies (force (normp x))
           (equal (q-ite x t nil)
                  x)))

(defthm |(q-ite non-nil y z)|
  (implies (and (atom x)
                x)
           (equal (q-ite x y z)
                  y))
  :hints(("Goal" :in-theory (enable q-ite))))

(defthm |(q-ite t x y)|
  (equal (q-ite t x y)
         x)
  :hints(("Goal" :in-theory (enable q-ite))))

(defthm |(q-ite nil x y)|
  (equal (q-ite nil x y)
         y)
  :hints(("Goal" :in-theory (enable q-ite))))


(defthm eval-bdd-of-non-consp-cheap
  (implies (not (consp x))
           (equal (eval-bdd x vals)
                  (if x t nil)))
  :rule-classes ((:rewrite :backchain-limit-lst 0)))


(defthm normp-compound-recognizer
  (implies (normp x)
           (or (consp x)
               (booleanp x)))
  :hints(("Goal" :in-theory (enable normp)))
  :rule-classes :compound-recognizer)
