#include <qvbox.h>
#include <qlabel.h>
#include <qsplitter.h>
#include <qtimer.h>
#include <qwidgetstack.h>

#include <kpushbutton.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kaction.h>
#include <kactionclasses.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kparts/part.h>
#include <ktrader.h>
#include <klibloader.h>
#include <kstatusbar.h>

#include <apt-pkg/packagemanager.h>
#include <apt-front/manager.h>
#include <apt-front/init.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/cache/component/state.h>
#include <apt-front/cache/component/history.h>
#include <apt-front/predicate/factory.h>

#include <adept/acqprogresswidget.h>
#include <adept/progress.h>
#include <adept/utils.h>

#include "app.h"

using namespace aptFront;
using namespace aptFront::cache;
using namespace adept;

void WaitForLister::waiting()
{
    kdDebug() << "WaitForLister::waiting()" << endl;
    if (app->m_list->searchView()->lister()->busy())
        QTimer::singleShot( 100, this, SLOT( waiting() ) );
    else {
        QTimer::singleShot( 0, app, slot );
        deleteLater();
    }
}

TestApp::TestApp()
{
    m_all = new QVBox( this );
    m_stack = new QWidgetStack( m_all );

    m_stack->addWidget( m_loading = new QLabel( i18n( "Loading, please wait..." ), m_stack ) );
    m_loading->setAlignment( Qt::AlignHCenter | Qt::AlignVCenter );

    m_buttons = new QHBox( m_all );
    QLabel *space = new QLabel( m_buttons ); // spacing
    m_next = new KPushButton( i18n( "Next" ), m_buttons );
    m_quit = new KPushButton( i18n( "Quit" ), m_buttons );
    m_next->setEnabled( false );
    m_quit->setEnabled( false );

    m_buttons->setSpacing( 2 );
    m_buttons->setMargin( 2 );
    QSizePolicy buttons( QSizePolicy::Preferred, QSizePolicy::Fixed, false );
    m_buttons->setSizePolicy( buttons );
    m_next->setSizePolicy( buttons );
    m_quit->setSizePolicy( buttons );
    space->setSizePolicy( QSizePolicy(
                               QSizePolicy::Expanding,
                               QSizePolicy::Fixed, false ) );

    setStandardToolBarMenuEnabled( false );
    createStandardStatusBarAction();
    setupActions();
    setupGUI( Keys|StatusBar|Save|Create );
    delete toolBar();

    Application::setStatusBar( statusBar() );

    QTimer::singleShot(
        0, this,
        SLOT( delayed() ) );

    setCentralWidget( m_all );
    setMinimumSize( 400, 300 );
}

void TestApp::delayed()
{
    initialize();
    observeComponent< component::State >();

    m_stack->addWidget( m_list = new adept::Browser( m_stack ) );
    m_stack->addWidget( m_bye = new QLabel( i18n( "Update Complete" ),
                                            m_stack ) );
    m_stack->addWidget( m_start = new QLabel( i18n( "Welcome to Adept Updater" ),
                                              m_stack ) );
    m_bye->setAlignment( Qt::AlignHCenter | Qt::AlignVCenter );
    m_start->setAlignment( Qt::AlignHCenter | Qt::AlignVCenter );

    m_list->searchView()->setUpgradeMode();

    m_stack->addWidget( m_progress = new adept::AcqProgressWidget( m_stack ) );
    m_stack->addWidget( m_commitProgress = new CommitProgress( m_stack ) );

    m_stack->raiseWidget( m_start );

    statusBar()->clear();
    notifyPostChange(0);
    start();
}

void TestApp::setupActions()
{
    KStdAction::quit( kapp, SLOT( quit() ), actionCollection() );
    m_undo = KStdAction::undo( this, SLOT( undo() ), actionCollection() );
    m_redo = KStdAction::redo( this, SLOT( redo() ), actionCollection() );
    setHistoryEnabled( false );
    createStandardStatusBarAction();
}

void TestApp::setHistoryEnabled( bool e ) {
    if (e && history() ) {
        m_undo->setEnabled( history()->canUndo() );
        m_redo->setEnabled( history()->canRedo() );
    } else {
        m_undo->setEnabled( false );
        m_redo->setEnabled( false );
    }
}

void TestApp::notifyPreChange( component::Base *b )
{
    Application::notifyPreChange( b );
    checkpoint();
}

void TestApp::notifyPostChange( component::Base *b )
{
    Application::notifyPostChange( b );
    m_list->searchView()->lister()->scheduleRebuild(); // rebuild on change...
}

void TestApp::setNext( QString s, const char *slot, QString q, const char *quits ) {
    disconnect( m_next, SIGNAL( clicked() ), 0, 0 );
    m_next->setText( s );
    m_next->setEnabled( slot );
    if ( slot )
        connect( m_next, SIGNAL( clicked() ), this, slot );

    disconnect( m_quit, SIGNAL( clicked() ), 0, 0 );
    m_quit->setEnabled( quits );
    m_quit->setText( q );
    if ( quits )
        connect( m_quit, SIGNAL( clicked() ), this, quits );
}

void TestApp::disableNext() {
    setNext( i18n( "Next" ), 0 );
}

void TestApp::disableButtons() {
    disableNext();
    m_quit->setEnabled( false );
}

void TestApp::start() {
    disableNext();
    setNext( i18n( "Fetch List of Updates" ), SLOT( update() ),
             i18n( "Skip Fetching List" ), SLOT( postUpdate() ) );
}

void TestApp::update() {
    kdDebug() << "TestApp::update" << endl;
    disableButtons();

    aptFront::Manager m;
    m.setProgressCallback( m_progress->callback() );
    m.setUpdateInterval( 100000 );
    kdDebug() << "manager set up" << endl;
    try {
        m_stack->raiseWidget( m_progress );
        m.update();
    } catch ( exception::OperationCancelled ) {
    } catch ( ... ) {
        KMessageBox::sorry(
            this, i18n( "There was an error downloading updates. " ),
            i18n( "Could not fetch updates" ) );
    }
    postUpdate();
}

void TestApp::postUpdate() {
    cache::Global::get().state().distUpgrade();
    if ( cache::Global::get().state().upgradableCount() > 0 ) {
        m_list->searchView()->lister()->scheduleRebuild();
        m_stack->raiseWidget( m_list );
        setHistoryEnabled( true );
        setNext( i18n( "Apply Updates" ), SLOT( commit() ) );
    } else {
        m_bye->setText( i18n( "Nothing to Update" ) );
        m_stack->raiseWidget( m_bye );
        disableNext();
        // setNext( i18n( "Nothing to do, Quit" ), SLOT( close() ) );
        m_quit->setText( i18n( "Quit" ) );
        m_quit->setEnabled( true );
    }
}

void TestApp::commit() {
    kdDebug() << "TestApp::commit" << endl;
    disableButtons();
    setHistoryEnabled( false );
    if (m_list->searchView()->lister()->busy()) {
        new WaitForLister( this, SLOT( commit() ) );
        return;
    }

    aptFront::Manager m;
    m.setProgressCallback( m_progress->callback() );
    m.setUpdateInterval( 100000 );
    try {
        m_stack->raiseWidget( m_progress );
        m.download();
        m_stack->raiseWidget( m_commitProgress );
        m.commit();
    } catch ( exception::OperationCancelled ) {
    } catch ( ... ) {
        KMessageBox::sorry(
            this, i18n( "There was an error commiting changes. "
                        "Possibly there was a problem downloading some "
                        "packages or the commit would break packages. " ),
            i18n( "Could not commit changes" ) );
    }

    m_stack->raiseWidget( m_bye );
    disableNext();
    m_quit->setText( i18n( "Quit" ) );
    m_quit->setEnabled( true );
    notifyPostChange( 0 );
}

#include "app.moc"
