# Controlling expression of selection-statements and iteration-statements.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/c/syntax"
require "adlint/c/object"
require "adlint/c/mediator"
require "adlint/c/expr"
require "adlint/c/conv"

module AdLint #:nodoc:
module C #:nodoc:

  # == DESCRIPTION
  # === Class structure
  #  ControllingExpression
  #  |
  #  +-> ValueDomainManipulator
  #      | <-- ValueDomainNarrower
  #      | <-- ValueDomainWidener
  #      |
  #      +-> ValueDomainNarrowing
  #            <-- ValueComparison
  #            <-- LogicalAnd
  #            <-- LogicalOr
  #            <-- StrictObjectDerivation
  #            <-- DelayedObjectDerivation
  class ControllingExpression
    def initialize(interpreter, branch)
      @interpreter = interpreter
      @branch = branch
      @manipulators = []
    end

    def ensure_true_by_narrowing(expression = nil)
      if expression
        new_manip = ValueDomainNarrower.new(@interpreter, expression)
        begin
          if @branch.implicit_condition?
            @interpreter._quiet = true
          end
          new_manip.prepare!
        ensure
          if @branch.implicit_condition?
            @interpreter._quiet = false
          end
        end
      else
        new_manip =
          WithoutExpressionValueDomainNarrower.new(@interpreter, @branch.group)
      end

      @manipulators.push(new_manip)
      new_manip
    end

    def ensure_true_by_widening(expression = nil)
      if expression
        new_manip = ValueDomainWidener.new(@interpreter, expression)
        begin
          if @branch.implicit_condition?
            @interpreter._quiet = true
          end
          new_manip.prepare!
        ensure
          if @branch.implicit_condition?
            @interpreter._quiet = false
          end
        end
      else
        new_manip =
          WithoutExpressionValueDomainWidener.new(@interpreter, @branch.group)
      end

      @manipulators.push(new_manip)
      new_manip
    end

    def undo(path_terminated)
      @manipulators.each { |manip| manip.rollback! } if path_terminated
    end

    def affected_variables
      @manipulators.map { |manip| manip.affected_variables }.flatten.uniq
    end

    def save_affected_variables
      @manipulators.each { |manip| manip.save! }
    end

    def restore_affected_variables
      @manipulators.each { |manip| manip.restore! }
    end
  end

  class ValueDomainManipulator < SyntaxTreeVisitor
    include InterpreterMediator
    include NotifierMediator
    include Conversion
    include ExpressionUtil
    include MonitorUtil

    def initialize(interpreter, expression)
      @interpreter = interpreter
      @expression = expression
      @affected_variables = []
      @narrowing = nil
      @value_memory = nil
    end

    attr_reader :interpreter
    attr_reader :affected_variables

    def prepare!
      if @expression
        @narrowing = @expression.accept(self)
        @narrowing.execute!
      end
    end

    def commit!
      if @narrowing
        @narrowing.ensure_result_equal_to(ScalarValue.of_true)
      end

      commit_changes(@narrowing)

      if @narrowing
        @affected_variables = @narrowing.narrowed_values.each_key.to_a
        @narrowing = nil
      end
    end

    def rollback!
      # NOTE: Rollback narrowed version to cut out the value-domain to enter
      #       the current branch.
      @affected_variables.each { |variable| variable.value.rollback! }
    end

    def save!
      @value_memory = {}
      @affected_variables.each do |variable|
        @value_memory[variable] = variable.value.to_single_value.dup
      end
    end

    def restore!
      if @value_memory
        @value_memory.each do |variable, saved_value|
          variable.assign!(saved_value)
        end
        @value_memory = nil
      end
    end

    def self.def_strict_object_derivation(method_name)
      class_eval <<-EOS
        define_method("#{method_name}") do |*args|
          StrictObjectDerivation.new(self, args.first)
        end
      EOS
    end
    private_class_method :def_strict_object_derivation

    def_strict_object_derivation :visit_error_expression
    def_strict_object_derivation :visit_object_specifier
    def_strict_object_derivation :visit_constant_specifier
    def_strict_object_derivation :visit_string_literal_specifier
    def_strict_object_derivation :visit_null_constant_specifier

    def visit_array_subscript_expression(node)
      checkpoint(node.location)

      base_narrowing = node.expression.accept(self)
      subscript_narrowing = node.array_subscript.accept(self)

      DelayedObjectDerivation.new(self, node,
                                  base_narrowing, subscript_narrowing) do
        checkpoint(node.location)

        base_narrowing.execute!
        subscript_narrowing.execute!

        execute_array_subscript_expression(node, base_narrowing.result,
                                           subscript_narrowing.result)
      end
    end

    def visit_function_call_expression(node)
      checkpoint(node.location)

      object_narrowing = node.expression.accept(self)
      arg_narrowings = node.argument_expressions.map { |expr|
        expr.accept(self)
      }

      DelayedObjectDerivation.new(self, node,
                                  object_narrowing, *arg_narrowings) do
        checkpoint(node.location)

        object_narrowing.execute!
        args = arg_narrowings.map { |narrowing|
          narrowing.execute!; [narrowing.result, narrowing.node]
        }

        execute_function_call_expression(node, object_narrowing.result, args)
      end
    end

    def visit_member_access_by_value_expression(node)
      checkpoint(node.location)

      object_narrowing = node.expression.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!

        execute_member_access_by_value_expression(
          node, object_narrowing.result)
      end
    end

    def visit_member_access_by_pointer_expression(node)
      checkpoint(node.location)

      object_narrowing = node.expression.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!

        execute_member_access_by_pointer_expression(
          node, object_narrowing.result)
      end
    end

    def_strict_object_derivation :visit_bit_access_by_value_expression
    def_strict_object_derivation :visit_bit_access_by_pointer_expression

    def visit_postfix_increment_expression(node)
      checkpoint(node.location)

      object_narrowing = node.operand.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!
        execute_postfix_increment_expression(node, object_narrowing.result)
      end
    end

    def visit_postfix_decrement_expression(node)
      checkpoint(node.location)

      object_narrowing = node.operand.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!
        execute_postfix_decrement_expression(node, object_narrowing.result)
      end
    end

    def_strict_object_derivation :visit_compound_literal_expression

    def visit_prefix_increment_expression(node)
      checkpoint(node.location)

      object_narrowing = node.operand.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!
        execute_prefix_increment_expression(node, object_narrowing.result)
      end
    end

    def visit_prefix_decrement_expression(node)
      checkpoint(node.location)

      object_narrowing = node.operand.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!
        execute_prefix_decrement_expression(node, object_narrowing.result)
      end
    end

    def_strict_object_derivation :visit_address_expression

    def visit_indirection_expression(node)
      checkpoint(node.location)

      object_narrowing = node.operand.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!

        execute_indirection_expression(node, object_narrowing.result)
      end
    end

    def visit_unary_arithmetic_expression(node)
      checkpoint(node.location)

      object_narrowing = node.operand.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!

        execute_unary_arithmetic_expression(node, object_narrowing.result)
      end
    end

    def_strict_object_derivation :visit_sizeof_expression
    def_strict_object_derivation :visit_sizeof_type_expression
    def_strict_object_derivation :visit_alignof_expression
    def_strict_object_derivation :visit_alignof_type_expression

    def visit_cast_expression(node)
      checkpoint(node.location)

      object_narrowing = node.operand.accept(self)

      DelayedObjectDerivation.new(self, node, object_narrowing) do
        checkpoint(node.location)

        object_narrowing.execute!

        execute_cast_expression(node, object_narrowing.result)
      end
    end

    def visit_multiplicative_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      DelayedObjectDerivation.new(self, node, lhs_narrowing, rhs_narrowing) do
        checkpoint(node.location)

        lhs_narrowing.execute!
        rhs_narrowing.execute!

        execute_multiplicative_expression(
          node, lhs_narrowing.result, rhs_narrowing.result)
      end
    end

    def visit_additive_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      DelayedObjectDerivation.new(self, node, lhs_narrowing, rhs_narrowing) do
        checkpoint(node.location)

        lhs_narrowing.execute!
        rhs_narrowing.execute!

        execute_additive_expression(
          node, lhs_narrowing.result, rhs_narrowing.result)
      end
    end

    def visit_shift_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      DelayedObjectDerivation.new(self, node, lhs_narrowing, rhs_narrowing) do
        checkpoint(node.location)

        lhs_narrowing.execute!
        rhs_narrowing.execute!

        execute_shift_expression(
          node, lhs_narrowing.result, rhs_narrowing.result)
      end
    end

    def visit_relational_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      ValueComparison.new(self, node, lhs_narrowing, rhs_narrowing)
    end

    def visit_equality_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      ValueComparison.new(self, node, lhs_narrowing, rhs_narrowing)
    end

    def visit_and_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      DelayedObjectDerivation.new(self, node, lhs_narrowing, rhs_narrowing) do
        checkpoint(node.location)

        lhs_narrowing.execute!
        rhs_narrowing.execute!

        execute_and_expression(node,
                               lhs_narrowing.result, rhs_narrowing.result)
      end
    end

    def visit_exclusive_or_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      DelayedObjectDerivation.new(self, node, lhs_narrowing, rhs_narrowing) do
        checkpoint(node.location)

        lhs_narrowing.execute!
        rhs_narrowing.execute!

        execute_exclusive_or_expression(
          node, lhs_narrowing.result, rhs_narrowing.result)
      end
    end

    def visit_inclusive_or_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      DelayedObjectDerivation.new(self, node, lhs_narrowing, rhs_narrowing) do
        checkpoint(node.location)

        lhs_narrowing.execute!
        rhs_narrowing.execute!

        execute_inclusive_or_expression(
          node, lhs_narrowing.result, rhs_narrowing.result)
      end
    end

    def visit_logical_and_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      LogicalAnd.new(self, node, lhs_narrowing, rhs_narrowing)
    end

    def visit_logical_or_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      LogicalOr.new(self, node, lhs_narrowing, rhs_narrowing)
    end

    def_strict_object_derivation :visit_conditional_expression

    def visit_simple_assignment_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      DelayedObjectDerivation.new(self, node, lhs_narrowing, rhs_narrowing) do
        checkpoint(node.location)

        lhs_narrowing.execute!
        rhs_narrowing.execute!

        execute_simple_assignment_expression(
          node, lhs_narrowing.result, rhs_narrowing.result)
      end
    end

    def visit_compound_assignment_expression(node)
      checkpoint(node.location)

      lhs_narrowing = node.lhs_operand.accept(self)
      rhs_narrowing = node.rhs_operand.accept(self)

      DelayedObjectDerivation.new(self, node, lhs_narrowing, rhs_narrowing) do
        checkpoint(node.location)

        lhs_narrowing.execute!
        rhs_narrowing.execute!

        execute_compound_assignment_expression(
          node, lhs_narrowing.result, rhs_narrowing.result)
      end
    end

    def visit_comma_separated_expression(node)
      checkpoint(node.location)

      object_narrowings = node.expressions.map { |expr| expr.accept(self) }

      DelayedObjectDerivation.new(self, node, *object_narrowings) do
        checkpoint(node.location)

        object_narrowings.map { |narrowing|
          narrowing.execute!
          narrowing.result
        }.last
      end
    end

    private
    def commit_changes(narrowing)
      subclass_responsibility
    end
  end

  class ValueDomainNarrower < ValueDomainManipulator
    private
    def commit_changes(narrowing)
      narrowing.narrowed_values.each do |variable, narrowed_value|
        variable.narrow_value_domain!(:==, narrowed_value)
      end
    end
  end

  class ValueDomainWidener < ValueDomainManipulator
    private
    def commit_changes(narrowing)
      narrowing.narrowed_values.each do |variable, narrowed_value|
        variable.widen_value_domain!(:==, narrowed_value)
      end
    end
  end

  class WithoutExpressionValueDomainNarrower < ValueDomainManipulator
    def initialize(interpreter, branch_group)
      super(interpreter, nil)
      @branch_group = branch_group
    end

    def prepare!
      raise TypeError, "no preparation without expression."
    end

    private
    def commit_changes(narrowing)
      @branch_group.all_controlling_variables.each do |variable|
        variable.narrow_value_domain!(:==, variable.type.arbitrary_value)
      end
      true
    end
  end

  class WithoutExpressionValueDomainWidener < ValueDomainManipulator
    def initialize(interpreter, branch_group)
      super(interpreter, nil)
      @branch_group = branch_group
    end

    def prepare!
      raise TypeError, "no preparation without expression."
    end

    private
    def commit_changes(narrowing)
      @branch_group.all_controlling_variables.each do |variable|
        variable.widen_value_domain!(:==, variable.type.arbitrary_value)
      end
      true
    end
  end

  class ValueDomainNarrowing
    include InterpreterMediator
    include NotifierMediator
    include Conversion

    def initialize(value_domain_manipulator, node, *children)
      @manipulator = value_domain_manipulator
      @node = node
      @children = children
      @original_values = {}
      @narrowed_values = {}
      @result = nil
    end

    attr_reader :node
    attr_reader :narrowed_values
    attr_reader :result

    def load_original_values!(narrowing)
      @original_values = narrowing.narrowed_values
      @children.each { |child| child.load_original_values!(narrowing) }
    end

    def execute!
      @result = do_narrowing
      @children.each do |narrowing|
        @narrowed_values = narrowing.narrowed_values.merge(@narrowed_values)
      end
    ensure
      if sequence_point = node.subsequent_sequence_point
        notify_sequence_point_reached(sequence_point)
      end
    end

    def ensure_result_equal_to(value)
      if @result.variable? && @result.designated_by_lvalue?
        if @result.value.scalar? && value.scalar?
          ensure_relation(@result, :==, value)
        end
      end
    end

    protected
    attr_reader :original_values

    private
    def do_narrowing
      subclass_responsibility
    end

    def do_logical_arithmetic_conversion(node, lhs_variable, rhs_variable)
      lhs_result, rhs_result =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_result
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_result)
      end

      unless rhs_variable == rhs_result
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_result)
      end

      return lhs_result, rhs_result
    end

    def ensure_relation(variable, operator_symbol, value)
      target_value = save_original_value(variable).dup
      target_value.narrow_domain!(operator_symbol, value)
      update_narrowed_value(variable, target_value)
    end

    def save_original_value(variable)
      @original_values[variable.to_named_variable] ||= variable.value.dup
    end

    def original_value_of(variable)
      @original_values[variable.to_named_variable]
    end

    def update_narrowed_value(variable, new_value)
      @narrowed_values[variable.to_named_variable] = new_value
    end

    def narrowing_merge!(lhs_narrowing, rhs_narrowing)
      lhs_values = lhs_narrowing.narrowed_values
      rhs_values = rhs_narrowing.narrowed_values

      @narrowed_values = lhs_values.merge(rhs_values) { |key, lhs_val, rhs_val|
        result = lhs_val.dup
        result.narrow_domain!(:==, rhs_val)
        result
      }
    end

    def widening_merge!(lhs_narrowing, rhs_narrowing)
      lhs_values = lhs_narrowing.narrowed_values
      rhs_values = rhs_narrowing.narrowed_values

      @narrowed_values = lhs_values.merge(rhs_values) { |key, lhs_val, rhs_val|
        result = lhs_val.dup
        result.widen_domain!(:==, rhs_val)
        result
      }
    end

    def interpreter
      # NOTE: This is private attr_reader for InterpreterMediator.
      #       This attribute is read via a normal method to suppress
      #       `private attribute?' warning.
      @manipulator.interpreter
    end
  end

  class ValueComparison < ValueDomainNarrowing
    def initialize(value_domain_manip, node, lhs_narrowing, rhs_narrowing)
      super
      @operator_symbol = node.operator.type.to_sym
      @lhs_narrowing = lhs_narrowing
      @rhs_narrowing = rhs_narrowing
    end

    private
    def do_narrowing
      @lhs_narrowing.execute!
      lhs_variable = object_to_variable(@lhs_narrowing.result)

      @rhs_narrowing.execute!
      rhs_variable = object_to_variable(@rhs_narrowing.result)

      unless lhs_variable.type.scalar? && rhs_variable.type.scalar?
        return temporary_variable(int_type)
      end

      unless lhs_variable.value.scalar? && rhs_variable.value.scalar?
        return temporary_variable(int_type)
      end

      lhs_converted, rhs_converted =
        do_logical_arithmetic_conversion(@node, lhs_variable, rhs_variable)

      case @operator_symbol
      when :==
        result = temporary_variable(int_type,
                                    lhs_converted.value == rhs_converted.value)
      when :!=
        result = temporary_variable(int_type,
                                    lhs_converted.value != rhs_converted.value)
      when :<
        result = temporary_variable(int_type,
                                    lhs_converted.value < rhs_converted.value)
      when :>
        result = temporary_variable(int_type,
                                    lhs_converted.value > rhs_converted.value)
      when :<=
        result = temporary_variable(int_type,
                                    lhs_converted.value <= rhs_converted.value)
      when :>=
        result = temporary_variable(int_type,
                                    lhs_converted.value >= rhs_converted.value)
      else
        # NOTREACHED
      end

      case @operator_symbol
      when :==, :!=
        notify_equality_expr_evaled(@node,
                                    lhs_converted, rhs_converted, result)
      when :<, :>, :<=, :>=
        notify_relational_expr_evaled(@node,
                                      lhs_converted, rhs_converted, result)
      end

      case
      when lhs_converted.designated_by_lvalue?
        ensure_relation(lhs_converted, @operator_symbol, rhs_converted.value)
      when rhs_converted.designated_by_lvalue?
        ensure_relation(rhs_converted, @operator_symbol, lhs_converted.value)
      else
        # NOTE: Domain of the rvalue should not be narrowed.
      end

      result
    end
  end

  class LogicalAnd < ValueDomainNarrowing
    def initialize(value_domain_manip, node, lhs_narrowing, rhs_narrowing)
      super
      @lhs_narrowing = lhs_narrowing
      @rhs_narrowing = rhs_narrowing
    end

    private
    def do_narrowing
      @lhs_narrowing.execute!
      @lhs_narrowing.ensure_result_equal_to(ScalarValue.of_true)
      lhs_variable = object_to_variable(@lhs_narrowing.result)

      # TODO: Must look about the short-circuit evaluation.
      @rhs_narrowing.load_original_values!(@lhs_narrowing)
      @rhs_narrowing.execute!
      @rhs_narrowing.ensure_result_equal_to(ScalarValue.of_true)
      rhs_variable = object_to_variable(@rhs_narrowing.result)

      narrowing_merge!(@lhs_narrowing, @rhs_narrowing)

      unless lhs_variable.type.scalar? && rhs_variable.type.scalar?
        return temporary_variable(int_type)
      end

      unless lhs_variable.value.scalar? && rhs_variable.value.scalar?
        return temporary_variable(int_type)
      end

      lhs_converted, rhs_converted =
        do_logical_arithmetic_conversion(@node, lhs_variable, rhs_variable)

      result = temporary_variable(
        int_type, lhs_converted.value.logical_and(rhs_converted.value))
      notify_logical_and_expr_evaled(@node,
                                     lhs_converted, rhs_converted, result)
      result
    end
  end

  class LogicalOr < ValueDomainNarrowing
    def initialize(value_domain_manip, node, lhs_narrowing, rhs_narrowing)
      super
      @lhs_narrowing = lhs_narrowing
      @rhs_narrowing = rhs_narrowing
    end

    private
    def do_narrowing
      @lhs_narrowing.execute!
      @lhs_narrowing.ensure_result_equal_to(ScalarValue.of_true)
      lhs_variable = object_to_variable(@lhs_narrowing.result)

      # TODO: Must look about the short-circuit evaluation.
      # FIXME: Base value of the RHS narrowing should be updated to ensure that
      #        the LHS condition is false.
      @rhs_narrowing.execute!
      @rhs_narrowing.ensure_result_equal_to(ScalarValue.of_true)
      rhs_variable = object_to_variable(@rhs_narrowing.result)

      widening_merge!(@lhs_narrowing, @rhs_narrowing)

      unless lhs_variable.type.scalar? && rhs_variable.type.scalar?
        return temporary_variable(int_type)
      end

      unless lhs_variable.value.scalar? && rhs_variable.value.scalar?
        return temporary_variable(int_type)
      end

      lhs_converted, rhs_converted =
        do_logical_arithmetic_conversion(@node, lhs_variable, rhs_variable)

      result = temporary_variable(
        int_type, lhs_converted.value.logical_or(rhs_converted.value))
      notify_logical_or_expr_evaled(@node,
                                    lhs_converted, rhs_converted, result)
      result
    end
  end

  class StrictObjectDerivation < ValueDomainNarrowing
    def initialize(value_domain_manipulator, node)
      super(value_domain_manipulator, node)
      @object = interpret(node)
    end

    private
    def do_narrowing
      if @object.variable? && @object.named?
        if original_value = original_value_of(@object)
          @object = PhantomVariable.new(@object, original_value)
        end
      end
      @object
    end
  end

  class DelayedObjectDerivation < ValueDomainNarrowing
    def initialize(value_domain_manipulator, node, *children, &block)
      super(value_domain_manipulator, node, *children)
      @block = block
    end

    private
    def do_narrowing
      @block.call
    end
  end

  class PhantomVariable < AliasVariable
    def initialize(named_variable, phantom_value = nil)
      super(named_variable)

      @base_variable = named_variable
      @phantom_value = phantom_value ?
        phantom_value : named_variable.memory.read_modifiable.dup
    end

    def value
      @phantom_value
    end

    def assign!(value)
      @phantom_value = value
    end

    def to_named_variable
      @base_variable.to_named_variable
    end

    def pretty_print(pp)
      Summary.new(object_id, name, type, @phantom_value).pretty_print(pp)
    end

    Summary = Struct.new(:object_id, :name, :type, :value)

    private
    def modifiable_value
      @phantom_value
    end
  end

end
end
