/*
   Project: UL

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael  Johnston

   Created: 2005-05-31 16:30:43 +0200 by michael johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#include "ULOptionsViewController.h"

@implementation ULOptionsViewController

- (id) initWithOptionsController: (id) optController modelViewController: (id) mVC
{
	if((self = [super init]) != nil)
	{
		if([NSBundle loadNibNamed: @"Options" owner: self] == NO)
		{
			NSWarnLog(@"Problem loading interface");
			return nil;
		}

		optionsController = [optController retain];
		modelViewController = mVC;
		[tabView setIndentationPerLevel: [tabView indentationPerLevel]*2.5];
		//[tabView setHeaderView: nil];
		[window setDelegate: self];
		
		modelObjects = [[NSArray alloc] initWithObjects: 
					@"ULSystem",
					@"ULOptions",
					@"AdDataSet", 
					@"ULProcess",
					@"ULSimulation", 
					nil];

		displayTypes = [[NSArray alloc] initWithObjects: 
					@"ULOptions",
					@"ULProcess",
					nil];
	}

	return self;
}

- (void) dealloc
{
	[displayTypes release];
	[modelObjects release];
	[outlineDelegate release];
	[optionsController release];
	[super dealloc];
}

- (void) open: (id) sender
{
	[optionsController newModelObject];
	
	[outlineDelegate release];
	outlineDelegate = [[ULOutlineViewDelegate alloc] 
				initWithOptions: [[optionsController Options] Options]];
	[tabView setDelegate: outlineDelegate];
	[tabView setDataSource: outlineDelegate];
	[[[[tabView tableColumns] objectAtIndex: 0] 
		headerCell] setStringValue: @"Options"];
	
	[saveButton setTransparent: NO];
	[saveButton setAction: @selector(setAndSave:)];
	[setButton setTransparent: NO];
	[cancelButton setTitle: @"Cancel"];
	[tabView setDrawsGrid: YES];
	[tabView reloadData];
	[saveButton setEnabled: YES];
	[setButton setEnabled: NO];
	[window setTitle: @"Create New Options"];
	[window center];
	[window makeKeyAndOrderFront: self];
}

- (void) setAndSave: (id) sender
{
	id metadataViewController;

	[self close: sender];
	[self displayMetadataForModelObject: [optionsController Options]
		allowEditing: YES
		runModal: YES];

	NS_DURING
	{
		[optionsController saveModelObject];
	}
	NS_HANDLER
	{
		NSWarnLog(@"%@ %@", [localException userInfo], [localException reason]);
		[localException raise];
	}
	NS_ENDHANDLER	
}

- (void) set: (id) sender
{
	[self close: sender];
}

- (void) close: (id) sender
{
	if([NSApp modalWindow] != nil)
		[NSApp stopModal];

	[window close];
} 

- (void) newOptions: (id) sender
{
	[self open: sender];
}

- (void) display: (id) sender
{	
	ULPasteboard* pasteboard = [ULPasteboard appPasteboard];
	id options, modelObject;
	NSString* type;

	type = [pasteboard availableTypeFromArray: displayTypes]; 
	modelObject = [pasteboard objectForType: type];
	if([type isEqual: @"ULProcess"])
		options = [(ULProcess*)modelObject options];
	else 
		options = modelObject;

	[outlineDelegate release];
	outlineDelegate = [[ULOutlineViewDelegate alloc] 
				initWithOptions: [options Options]];
	[tabView setDelegate: outlineDelegate];
	[tabView setDataSource: outlineDelegate];
	[[[[tabView tableColumns] objectAtIndex: 0] 
		headerCell] setStringValue: @"Options"];

	//display setup

	[saveButton setTransparent: NO];
	[saveButton setAction: @selector(setAndSave:)];
	[setButton setTransparent: NO];
	[cancelButton setTitle: @"Cancel"];
	[tabView reloadData];
	[tabView setDrawsGrid: YES];
	[setButton setEnabled: YES];
	[saveButton setEnabled: NO];
	[window center];
	[window setTitle: [NSString stringWithFormat: 
		@"Edit Options - %@", [options valueForKey:@"name"]]];
	[window makeKeyAndOrderFront: self];
}

- (void) displayMetadataForModelObject: (id) modelObject allowEditing: (BOOL) value
{
	[self displayMetadataForModelObject: modelObject 
		allowEditing: value
		runModal: NO];
}

- (void) saveObjectMetadata: (id) sender
{
	if(![[currentModelObject database] isEqual: @"None"])
	{
		[[ULDatabaseInterface databaseInterface]
			updateMetadataForObject: currentModelObject
			inSchema: [currentModelObject schema]
			ofClient: [currentModelObject valueForVolatileMetadataKey: @"DatabaseClient"]];
	}		
	[self close: self];	
	[currentModelObject release];
}

- (void) displayMetadataForModelObject: (id) modelObject allowEditing: (BOOL) value runModal: (BOOL) flag
{
	[outlineDelegate release];
	outlineDelegate = [[ULOutlineViewDelegate alloc] 
				initWithProperties: [[modelObject dataDictionary] mutableCopy]
				allowEditing: value];
	[tabView setDelegate: outlineDelegate];
	[tabView setDataSource: outlineDelegate];

	//keep a ref to modelObject so we can save changes to its metadata
	currentModelObject = [modelObject retain];
	
	//set display for properties
	[setButton setTransparent: YES];
	[saveButton setAction: @selector(saveObjectMetadata:)];
	[saveButton setEnabled: YES];
	[tabView setDrawsGrid: NO];
	[tabView setUsesAlternatingRowBackgroundColors: NO];
	[tabView reloadData];
	//Start with everything expanded (General and Meta data)
	[tabView expandItem: nil expandChildren: YES];
	[[[[tabView tableColumns] objectAtIndex: 0] 
		headerCell] setStringValue: @"Properties"];
	[window setDelegate: self];
	[window setTitle: 
		[NSString stringWithFormat: @"Properties - %@  (%@)",
			 [modelObject valueForKey:@"name"], NSStringFromClass([modelObject class])]];
	[window center];
	[window makeKeyAndOrderFront: self];
	if(flag)
		[NSApp runModalForWindow: window];
}

- (BOOL) validateMenuItem: (NSMenuItem*) menuItem
{
	NSArray* availableTypes;
	id modelObject, pasteboard;

	pasteboard = [ULPasteboard appPasteboard];
	availableTypes = [pasteboard availableTypes];
	if([availableTypes count] != 1)
		return NO;

	if([pasteboard countOfObjectsForType: [availableTypes objectAtIndex: 0]] != 1)
		return NO;

	if([modelObjects containsObject: [availableTypes objectAtIndex: 0]])
		return YES;

	return NO;
}

- (void) properties: (id) sender
{
	id modelObject, dataType, pasteboard;

	pasteboard = [ULPasteboard appPasteboard];
	dataType = [[pasteboard availableTypes] 
			objectAtIndex: 0];
	modelObject = [pasteboard objectForType: dataType]; 
	[self displayMetadataForModelObject: modelObject
		allowEditing: YES];
}

- (void) windowWillClose: (NSNotification*) aNotification
{
	if([NSApp modalWindow] != nil)
		[NSApp stopModal];
}

@end
