/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <string.h>
#include <netdb.h>
#include <unistd.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <errno.h>
#include <stdio.h>
#include <netgroup.h>
#include <netsocket.h>

NetSocket::NetSocket(int size)
{
	// create socket
	sockfd=socket(AF_INET, SOCK_STREAM, 0);
	
	// set socket non-blocking (this seems to cause several problems in Linux)
	if(fcntl(sockfd, F_SETFL, O_NONBLOCK)!=0) printf("warning: could not set socket non-blocking\n");
	
	// set last error to an empty string
	lasterror="not connected";
	
	// set properties
	sr_size=size;
	read_timeout=NetGroup::getDefaultReadTimeout();
	write_timeout=NetGroup::getDefaultWriteTimeout();

	// not connected yet
	conn=false;
}

NetSocket::~NetSocket()
{
	// close first if connected
	if(isConnected()) close(sockfd);
}

void NetSocket::setReadTimeout(int timeout)
{
	read_timeout=timeout;
}

int NetSocket::getReadTimeout()
{
	return read_timeout;
}

void NetSocket::setBufferSize(int size)
{
	sr_size=size;
}

int NetSocket::getBufferSize()
{
	return sr_size;
}

bool NetSocket::connect(const char *host, int port, int timeout)
{
	struct sockaddr_in dest_addr;
	struct hostent *h;
	struct timeval tv;
	fd_set fdread, fdwrite, fdexcept;

	// set address options and select descriptors
	h=gethostbyname(host);
	if(!h)
	{
		// handle error
		switch(h_errno)
		{
			case HOST_NOT_FOUND:
				lasterror="host not found";
				break;
			case NO_ADDRESS:
				lasterror="host found, but no address was returned";
				break;
			case NO_RECOVERY:
				lasterror="a name server error has occured";
				break;
			case TRY_AGAIN:
				lasterror="a temporary name server error has occured, try again";
				break;
			default:
				lasterror="unknown gethostbyname() error";
		}
		
		return false;
	}
	
	dest_addr.sin_family=AF_INET;
	dest_addr.sin_port=htons(port);
	dest_addr.sin_addr=*((struct in_addr *)h->h_addr);
	bzero(&(dest_addr.sin_zero), 8);
	
	FD_ZERO(&fdread);
	FD_ZERO(&fdwrite);
	FD_ZERO(&fdexcept);
	FD_SET(sockfd, &fdread);
	FD_SET(sockfd, &fdwrite);
	FD_SET(sockfd, &fdexcept);
	
	tv.tv_sec=timeout;
	tv.tv_usec=0;
	
	// connect
	int iError = 0; //no error
	unsigned int uErrorSize = sizeof(int);
	::connect(sockfd, (struct sockaddr *)&dest_addr, sizeof(struct sockaddr));
	::select(sockfd+1, &fdread, &fdwrite, 0, &tv);
	// getsockopt will get us the connecting error
	getsockopt(sockfd, SOL_SOCKET, SO_ERROR, &iError, &uErrorSize);
  // if(FD_ISSET(sockfd, &fdread) || FD_ISSET(sockfd, &fdwrite))
	if( !iError )
	{
		conn=true;
		return true;
	}

	close(sockfd);
	lasterror="connection could not be established";
	return false;
}

void NetSocket::disconnect()
{
	close(sockfd);
	conn=false;
}

bool NetSocket::isConnected()
{
	if(!conn)
		return false;
	
	// reading 0 bytes should either return success (bytes available) or EAGAIN (no bytes available), otherwise it's not connected
	int res=recv(sockfd, 0, 0, 0);
	if(res==0 || (res==-1 && errno==EAGAIN)) return true;

	lasterror="not connected";
	return false;
}

bool NetSocket::isValid()
{
	if(sockfd!=-1)
		return true;
	return false;
}

bool NetSocket::bytesAvailable()
{
	fd_set fdread;
	struct timeval tv;
	
	FD_ZERO(&fdread);
	FD_SET(sockfd, &fdread);
	
	tv.tv_sec=read_timeout;
	tv.tv_usec=0;
	
	// using select() we are able to timeout on read, handy for slow connections
	if(select(sockfd+1, &fdread, 0, 0, &tv)>0)
		return true;
	return false;
}

bool NetSocket::canWrite()
{
	fd_set fdwrite;
	struct timeval tv;
	int iErr;
	
	FD_ZERO(&fdwrite);
	FD_SET(sockfd, &fdwrite);
	
	tv.tv_sec=write_timeout;
	tv.tv_usec=0;
	
	// using select() we are able to timeout on read, handy for slow connections
	if( (iErr=select(sockfd+1, 0, &fdwrite, 0, &tv))>0)
		return true;
	return false;
}

int NetSocket::write(const char *buffer)
{
	int offset=0, remaining=strlen(buffer), chunk=0;
		
	while(remaining!=0 && chunk!=-1)
	{
		chunk=send(sockfd, buffer+offset, remaining<sr_size?remaining:sr_size, 0);
		if(chunk!=-1)
		{
			offset+=chunk;
			remaining-=chunk;
		}
		// a small delay for controling the flood
		usleep(10000);
	}
	
	return offset;
}

int NetSocket::read(char *buffer, int maxlength)
{
	int received=0;
	
	if( bytesAvailable() )
		received=recv(sockfd, buffer+received, maxlength<sr_size?maxlength:sr_size, 0);
	
  if( received==-1 )
    return -1;

  buffer[received]=0;
	return received;
}

int NetSocket::readLine(char *buffer, int maxlength)
{
	int received=0, last=1;
	char lastchar=0;
	
	while(lastchar!='\n' && received<=maxlength && last)
	{
		last=read(buffer+received, 1);
		lastchar=buffer[received];
		received+=last;
	}
	
	return received;
}

const char *NetSocket::error()
{
	return lasterror;
}

void NetSocket::setSRSize(const int iSRSize)
{
  sr_size=iSRSize;
}

int NetSocket::srSize() const
{
  return sr_size;
}




