/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <pop3uidjar.h>
#include <pthread.h>
#include <semaphore.h>
#include <incomingauthstructure.h>
#include <string>
#include <vector>
#include <pop3handler.h>
#include <netpop3.h>
#include <algorithm>
#include <unistd.h>
#include <limits.h>
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#ifdef OS_LINUX
  #include <asm/errno.h>
#else 
  #ifdef OS_FREEBSD
    #include <errno.h>
  #endif
#endif

void pop3_signal_progress(string s)
{
  static const char buf = 0x0;
  
  POP3Handler::progressData=string("pop3:")+s;
  ::write(POP3Handler::progressSync[1], &buf, 1);
	
  sem_wait(&POP3Handler::threadLock);
}

void pop3_save_uids()
{
}

void *pop3_thread(void *)
{
  static const char buf=0x0;
	IncomingAuthStructure *auth=POP3Handler::incomingConnectionData;
	NetPOP3 client(auth->speed.c_str());
	bool uidsdirty=false;
	bool havemsgs=true;
	int validmsgs=0;
	unsigned nsindex=0;
	int dlindex=1;
	unsigned messagecount=0;
	unsigned totalsize=0;
	unsigned uPos=0;
	vector<int> messagesizes;
	vector<string> messageuids;
	vector<unsigned> todelete;
  POP3_UIDJar	*uidJar=POP3_UIDJar::ref();
	
	printf("pop3thread: starting\n");

	// debug
	printf("pop3thread: connecting to server %s:%d...\n", auth->server.c_str(), auth->port);
	pop3_signal_progress("connect");
	// connect to server
	if(!client.connect(auth->server, auth->port)) goto errexit;
	
	// debug
	printf("pop3thread: connected.\n");
	printf("pop3thread: sending user command...\n");
	// user
	if(!client.user(auth->user)) goto errexit;
	
	// debug
	printf("pop3thread: sending pass command...\n");

	// pass
	if(!client.pass(auth->password)) goto errexit;
	
	// debug
	printf("pop3thread: logged in.\n");

	pop3_signal_progress("list");

	if(!client.stat(messagecount, totalsize)) goto errexit;
	
	// debug
	printf("pop3thread: %d messages on the server\n", messagecount);
	printf("pop3thread: getting list of sizes and uids...\n");
	
	// get list of message sizes and uids
	if(messagecount)
	{
		messagesizes.resize(messagecount);
		messageuids.resize(messagecount);
		
		if(!client.list(messagesizes) || !client.uidl(messageuids)) goto errexit;
	
		char buf[100];
		sprintf(buf, "messages:%u", messagecount);
		pop3_signal_progress(string(buf));					
	}

	// set 0 size for messages we've already received
	uPos=0;
	for(vector<string>::iterator it=messageuids.begin(); it!=messageuids.end(); ++it)
	{
	  UIDMap::iterator itFind=uidJar->rcvList.find((*it));
	  if(  itFind!= uidJar->rcvList.end() )
			messagesizes[uPos]=0;
	  uPos++;
	}
	
	// debug
	printf("pop3thread: done checking.\n");

	// if server-side filtering is enabled get top of message and ask the handler for action
	if(auth->filtering)
	{
		// debug
		printf("pop3thread: server-side filtering enabled, proceeding...\n");

		for(unsigned i=0;i<messagecount;i++)
		{
			if(messagesizes[i])
			{
				if(!client.top(i+1, 0, POP3Handler::filterData)) goto errexit;
					
				write(POP3Handler::filterSync[1], buf, 1);
				sem_wait(&POP3Handler::threadLock);

				// delete or ignore
				if(POP3Handler::filterResponse==POP3Handler::Delete)
				{
					todelete.push_back(i);
					messagesizes[i]=0;
				}
				else if(POP3Handler::filterResponse==POP3Handler::Ignore)
				{
					messagesizes[i]=0;
				}
				else if(POP3Handler::filterResponse==POP3Handler::AlwaysIgnore)
				{
					uidJar->rcvInsert(messageuids[i]);
					messagesizes[i]=0;
					uidsdirty=true;
				}
			}
		}
	}

	// debug
	printf("pop3thread: server-side filtering finished (enabled: %s)\n", auth->filtering?"yes":"no");
	printf("pop3thread: checking message sizes (max download %d)...\n", auth->max_dl);

	// check messages sizes
	for(unsigned i=0;i<messagecount && auth->max_dl>0;i++)
	{
		if(messagesizes[i]>auth->max_dl)
		{
			if(auth->max_action==POP3Handler::Ask)
			{
				if(!client.top(i+1, 0, POP3Handler::dlData.header)) goto errexit;

				POP3Handler::dlData.size=messagesizes[i];
				write(POP3Handler::dlSync[1], buf, 1);
				sem_wait(&POP3Handler::threadLock);

				auth->max_action=POP3Handler::dlResponse;
			}

			// delete or ignore
			if(auth->max_action==POP3Handler::Delete)
			{
				todelete.push_back(i);
				messagesizes[i]=0;
			}
			else if(auth->max_action==POP3Handler::Ignore)
			{
				messagesizes[i]=0;
			}
			else if(auth->max_action==POP3Handler::AlwaysIgnore)
			{
				uidJar->rcvInsert(messageuids[i]);
				messagesizes[i]=0;
				uidsdirty=true;
			}
		}
	}

	// debug
	printf("pop3thread: size check done.\n");
	printf("pop3thread: computing number of messages to download...\n");
	// compute number of messages to download
	for(unsigned i=0;i<messagecount;i++) if(messagesizes[i]) validmsgs++;

	// debug
	printf("pop3thread: done. (%d messages)\n", validmsgs);

	// signal progress
	char total[100];
	if(!validmsgs)
	{
		pop3_signal_progress("nomessages");
	}
	else
	{
		sprintf(total, "messages:%d", validmsgs);
		pop3_signal_progress(total);
	}

	// debug
	printf("pop3thread: starting to download messages...\n");

  // get messages, sorted if necessary
	while(havemsgs && messagecount)
	{
		// debug
		printf("pop3thread: selecting message to download...\n");
		// select message to download
		int currentmessage=-1;
		if(auth->sorted)
		{
			int minsize=INT_MAX;
			for(unsigned i=0;i<messagecount;i++)
				if(messagesizes[i]>0 && messagesizes[i]<minsize)
				{	
					currentmessage=i;
					minsize=messagesizes[i];
				}
		}
		else
		{
			while(currentmessage==-1 && nsindex<messagecount)
			{
				if(messagesizes[nsindex]) currentmessage=nsindex;
				nsindex++;
			}
		}

		if(currentmessage!=-1)
			messagesizes[currentmessage]=0;
		else
			havemsgs=false;

		if(currentmessage!=-1)
		{
			// signal progress
			char current[100];
			sprintf(current, "%d", dlindex++);
			pop3_signal_progress(string("dl:")+current+":"+total);
		}

		// debug
		printf("pop3thread: message %d selected\n", havemsgs?currentmessage+1:-1);

		// check for abort signal
		if(!sem_trywait(&POP3Handler::stopThread))
			havemsgs=false;

		// debug
		printf("pop3thread: downloading message...\n");

		// download message
		if(havemsgs)
		{
			unsigned int currentTime=time(0);

			if(!client.retr(currentmessage+1, POP3Handler::incomingMessage.message)) goto errexit;

			POP3Handler::incomingMessage.uid=messageuids[currentmessage];
			POP3Handler::incomingMessage.rcvtime=currentTime;

			// debug
			// fprintf(log, "===> message %d\n%s", currentmessage+1, POP3Handler::incomingMessage.message.c_str());

			::write(POP3Handler::rcvSync[1], &buf, 1);
			sem_wait(&POP3Handler::threadLock);

			if(auth->leave)
			{
				uidJar->rcvInsert(messageuids[currentmessage], currentTime);
				uidsdirty=true;
			}
			else
			{
				todelete.push_back((unsigned)currentmessage);
			}
		}
	}

	// debug
	printf("pop3thread: syncing server...\n");
	printf("pop3thread: computing list of messages to delete...\n");

	// sync with server
	pop3_signal_progress("sync");
	
	for(vector<string>::iterator it=messageuids.begin(); it!=messageuids.end(); ++it)
	{
	  UIDMap::iterator itFind=uidJar->trashList.find((*it));
	  if( itFind!=uidJar->trashList.end() )
	  {
			todelete.push_back(uPos);
			uidJar->trashList.erase(itFind);
			uidsdirty=true;
	  }
	  uPos++;
	}
	
	if(auth->persistence)
	{
		unsigned int currentTime=time(0);
	  for(vector<string>::iterator itUID=messageuids.begin(); itUID!=messageuids.end(); ++itUID)
	  {
	    UIDMap::iterator itFind=uidJar->rcvList.find((*itUID));
	    if( itFind != uidJar->rcvList.end() && 
	        (*itFind).second.rcvTime - currentTime > auth->persistence )
	    {
				todelete.push_back(uPos);
				uidJar->rcvList.erase(itFind);
				uidsdirty=true;
	    }
	    uPos++;
	  }
	}

	// debug
	printf("pop3thread: deleting %u messages...\n", todelete.size());
	
	for(unsigned i=0; i<todelete.size(); i++)
		if(!client.dele(todelete[i]+1)) 
		  goto errexit;

	// save uids
	if(uidsdirty)
	{
		// debug
		printf("pop3thread: waiting for handler to save uids...\n");

		::write(POP3Handler::uidSync[1], &buf, 1);
		sem_wait(&POP3Handler::threadLock);
	}

	// disconnect
	pop3_signal_progress("completed");
	client.quit();

	// debug
	printf("pop3thread: process complete.\n");
		
	// unlock global pop3 semaphore and exit
	sem_post(&POP3Handler::pop3Lock);
	
	printf("pop3thread: exiting\n");
	
	pthread_exit(0);

errexit:	
	pop3_signal_progress("err:"+client.error());
	client.quit();
	sem_post(&POP3Handler::pop3Lock);
	pthread_exit(0);
}
