/***************************************************************************
                remotemailfolder.cpp  -  the remote mail folder class
                             -------------------
    begin                : Mon Mar  5 16:16:00 EET 2001
    copyright            : (C) 2001 by theKompany (www.thekompany.com>
    author               : Eugen Constantinescu
    email                : eug@thekompany.com
 ***************************************************************************/

#include <config.h>
#include <kconfig.h>

#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>

#include <remotemailfolder.h>
#include <indexclass.h>
#include <servernotifier.h>
#include <accounts.h>
#include <accountmanager.h>
#include <messagedevice.h>
#include <messagedescriptor.h>
#include <messagefactory.h>
#include <headerclass.h>
#include <orb.h>
#include <imap4handler.h>

extern KConfig *GlobalConfig;

RemoteMailFolder::RemoteMailFolder(const QString &_storageDevice) : MailFolder(_storageDevice)
{
  setupFiles();
  config();
	loadIndex();
	loadStatus();
	uidJar=new IMAP4_UIDJar(getStorageDevice());
}

RemoteMailFolder::RemoteMailFolder(const QString &_storageDevice, const QString account, const QString mailbox) : MailFolder(_storageDevice)
{
  setupFiles();
  makeConfig(account, mailbox);
	loadIndex();
	loadStatus();
	uidJar=new IMAP4_UIDJar(getStorageDevice());
}

RemoteMailFolder::~RemoteMailFolder()
{
  saveStatus();
  delete uidJar;
}

void RemoteMailFolder::setupFiles()
{
 	headersFileName=getStorageDevice()+"/headers";
 	configFileName=getStorageDevice()+"/config";
}

void RemoteMailFolder::config()
{
  QFile f( configFileName );
  f.open( IO_ReadOnly );
  QDataStream s( &f );
  s >> accountName;
  s >> mailboxName;
  f.close();

  printf("Account=%s||Mailbox=%s", accountName.latin1(), mailboxName.latin1());
  fflush(stdout);
}

void RemoteMailFolder::makeConfig(QString account, QString mailbox)
{
  accountName=account;
  mailboxName=mailbox;

  QFile f( configFileName );
  f.open( IO_WriteOnly );
  QDataStream s( &f );
  s << accountName;
  s << mailboxName;
  f.close();

  printf("Account=%s||Mailbox=%s", accountName.latin1(), mailboxName.latin1());
  fflush(stdout);
}

QString RemoteMailFolder::account() const
{
  return accountName;
}

QString RemoteMailFolder::mailbox() const
{
  return mailboxName;
}

QString RemoteMailFolder::getMessagesFileName() const
{
  return headersFileName;
}
		
IndexClass* RemoteMailFolder::createMessage(const QCString &text, const QCString &uid, const QDateTime &rcvtime, const QString &account=QString::null, MessageClass *parsedMessage=0, bool bSync, const unsigned flags)
{
  if( !uid || uid.isNull() || uid.isEmpty() )
    return (IndexClass*)0;

	// rfc-parse message header
	HeaderClass rfcHeader((const char*)text);
	// build the message from the header
	MessageClass rfcMessage(rfcHeader);
	
	// create index, set id and add to collection
	IndexClass *index=new IndexClass(this);
	syncIndexID(index);
	indexCollection.insert(index->getID(), index);
	
	// create message device
	MessageDevice *dev=new MessageDevice(index);
	
	// populate descriptor
	MessageDescriptor &descriptor=dev->getDescriptor();
	descriptor.load(rfcMessage);
	
	// add other data (account, uid, received time, index id etc)
	descriptor.account=account;
	descriptor.indexID=index->getID();
	descriptor.receivedDate=(QCString)DateClass(rcvtime);
	descriptor.uid=uid;
	
	// debug
	printf("remotemailfolder: created message with:\n");
	printf("\taccount %s\n", (const char *)descriptor.account);
	printf("\tuid %s\n", (const char *)descriptor.uid);
	printf("\treceived time %s\n", (const char *)descriptor.receivedDate);
	printf("\tindex id %s\n", (const char *)descriptor.indexID);
	
	// Update the uid file
	QString urlMessage=name()+"/"+index->getID();
	uidJar->rcvInsert(descriptor.uid.latin1(), urlMessage.latin1());
	
	// write message id and unread mark into index
	index->setMessageID(descriptor.messageID);
	if( flags&MailFolder::Seen )
	{
	  descriptor.status="Read";
	  index->setUnreadMark(false);
	}
	else
	{
	  descriptor.status="New";
	  index->setUnreadMark(true);
	}
	
	// update folder stats
	if(index->getUnreadMark()) incrementUnread();
	
	// save message
	QFile dataFile(getMessagesFileName());
	dataFile.open(IO_WriteOnly|IO_Append);

	// write MBOX information
	QCString mboxInfo;
	mboxInfo="From - " + descriptor.receivedDate + "\r\n";
	dataFile.writeBlock((const char *)mboxInfo, mboxInfo.length());
	
	// write uniblock data (index is uniblock by default)
	index->setUniblockOffset(dataFile.at());
	index->setUniblockLength(text.length());
	index->setMultipartOnly(false);
	
	dataFile.writeBlock((const char *)text, index->getUniblockLength());
	dataFile.writeBlock("\r\n", 2);
	
	dataFile.close();
	
	// create index part list and update offsets
	for(unsigned int i=0;i<rfcMessage.partList.count();i++)
		index->addPart(new MimePart(*rfcMessage.partList.at(i)), true);
	
	// save descriptor (that will save the index as well)
	dev->saveDescriptor();
	
	// cleanup
	delete dev;
	
	// add index and descriptor to hierarchy
	GlobalConfig->setGroup("Threading");
	if(GlobalConfig->readEntry("Enable")=="Yes")
	{
		reparentIndex(index);
		crossReferenceIndexSet(index);
	}
	
	// notify client or wait for a sync signal
	if( bSync )
    syncMessages.append(name()+"/"+index->getID());
	else
	{
  	ServerNotifier::thisInstance()->objectCreated(name()+"/"+index->getID());
  	ServerNotifier::thisInstance()->objectChanged(name());
	}
	
	return index;
}
		
bool RemoteMailFolder::updateMessage(const QCString &text, const QCString &uid)
{
  if( !uid || uid.isNull() || uid.isEmpty() )
    return false;

  IndexClass *index=(IndexClass*)0;
	MessageClass rfcMessage((const char*)text);
	// get the index reference from UID list
	UIDMap::iterator itFind=uidJar->rcvList.find((string)((const char*)uid));
	if( itFind!=uidJar->rcvList.end() )
    index=ObjectRequestBroker::thisInstance()->indexReference(QString((*itFind).second.info.c_str()));
  if( !index )
  {
//    printf("\nUid=%s, Url=%s\n", (const char*)uid, (*itFind).second.info.c_str());
//    fflush(stdout);
    return false;
  }

	// create message device
	MessageDevice *dev=new MessageDevice(index);
	dev->loadDescriptor();
	MessageDescriptor &descriptor=dev->getDescriptor();
	descriptor.load(rfcMessage);
	
	// debug
	printf("remotemailfolder: changed message with:\n");
	printf("\taccount %s\n", (const char *)descriptor.account);
	printf("\tuid %s\n", (const char *)descriptor.uid);
	printf("\treceived time %s\n", (const char *)descriptor.receivedDate);
	printf("\tindex id %s\n", (const char *)descriptor.indexID);
	
	// save message
	QFile dataFile(getMessagesFileName());
	dataFile.open(IO_WriteOnly|IO_Append);
	
	// write MBOX information
	QCString mboxInfo;
	mboxInfo="From - " + descriptor.receivedDate + "\r\n";
	dataFile.writeBlock((const char *)mboxInfo, mboxInfo.length());
	
	// write uniblock data (index is uniblock by default)
	index->setUniblockOffset(dataFile.at());
	index->setUniblockLength(text.length());
	index->setMultipartOnly(false);
	
	dataFile.writeBlock((const char *)text, index->getUniblockLength());
	dataFile.writeBlock("\r\n", 2);
	
	dataFile.close();
	
	// create index part list and update offsets
	for(unsigned int i=0;i<rfcMessage.partList.count();i++)
		index->addPart(new MimePart(*rfcMessage.partList.at(i)), true);
	
	// save descriptor (that will save the index as well)
	dev->saveDescriptor();
	
	// cleanup
	delete dev;

 	ServerNotifier::thisInstance()->objectChanged(name()+"/"+index->getID());
 	
	return true;	
}

void RemoteMailFolder::sync()
{
  // Add the messages from the list
  if( !syncMessages.count() )
  {
    printf("\nRemoteMailFolder::sync() : Nothing to sync!\n");
  }
  else
  {
    ServerNotifier::thisInstance()->objectCreated(syncMessages);
    syncMessages.clear();
  	ServerNotifier::thisInstance()->objectChanged(name());
  }

  // Delete the messages from trash list.
  if( !uidJar->trashList.size() )
  {
    for(UIDMap::iterator it=uidJar->trashList.begin(); it!=uidJar->trashList.end(); ++it)
    {
		  IndexClass *idx=ObjectRequestBroker::thisInstance()->indexReference(QString((*it).second.info.c_str()));
		  if( idx )
		    deleteMessage(idx);
    }
    uidJar->trashList.clear();
  }

  //Save the status
  uidJar->saveLists();
  if( !saveStatus() )
    printf("\nSaving status failed!\n");
}
	
void RemoteMailFolder::loadIndex()
{
	// debug
	printf("mailfolder: [%s] is reading index...\n", (const char *)name());
	
	IndexClass *tindex;
  setUnread(0);
  setPruneIndexCount(0);

	QFile f(getIndexFileName());
	QDataStream stream(&f);

  // check index version number
	if(f.open(IO_ReadOnly))
	{
		// debug
		printf("mailfolder: index file opened.\n");
		
  	QString idxExpectedVersion=QString::number(INDEX_VERSION);
		QString idxVersion;
		stream>>idxVersion;

  	if(idxExpectedVersion!=idxVersion)
  	{
    	printf("Folder %s attempted to read an older version or invalid index\
        	cache, discarding index...\n", (const char *)name());
  	}
  	else
  	{
	  	while(!stream.atEnd())
	  	{
		  	tindex=new IndexClass(this);

				// read index
      	stream>>(*tindex);

				// check to see if the index was already included
				IndexClass *pruneIndex;
				if((pruneIndex=indexCollection[tindex->getID()]))
				{
					// ...it was, increment the prune list
					incrementPruneIndexCount();

					// update folder stats
					if(pruneIndex->getUnreadMark()) decrementUnread();
				}
				else
				{
					// update the last index id
					unsigned long tlast=tindex->getID().mid(3).toULong();
					if(tlast>getLastIndexID())
						setLastIndexID(tlast+1);
				}
				// add index to collection
				indexCollection.replace(tindex->getID(), tindex);
				
				// debug
				// printf("localmailfolder: [loadIndex] loaded %s...\n", (const char *)tindex->getID());

				// update folder stats
		  	if(tindex->getUnreadMark()) incrementUnread();
	  	}
  	}
		
		f.close();
	}
}

bool RemoteMailFolder::shouldExpunge()
{
  return false;
}

bool RemoteMailFolder::deleteMessage(IndexClass *idx)
{
	// debug
	printf("\nremotemailfolder: message deleted\n");
	fflush(stdout);
	
	if(!idx || idx->getParentFolder()!=this)
	  return false;
	
	// remove the mail from the IMAP folder too
	if( !IMAP4Handler::ref()->deleteMessage(*idx) )
	  return false;
	
	QString path=name()+"/"+idx->getID();
	
	if(idx->getUnreadMark()) decrementUnread();	
	
	indexCollection.remove(idx->getID());

	delete idx;
	saveIndex();
		
	// notify the client
	ServerNotifier::thisInstance()->objectDeleted(path);
	ServerNotifier::thisInstance()->objectChanged(name());
	
	return true;
}

IndexClass* RemoteMailFolder::copyMessage(IndexClass *idx)
{
  if( !idx->isFullMessage() )
 	  IMAP4Handler::ref()->updateMessage(*idx);
  IMAP4Handler::ref()->addMessage(name(), idx);
  // return 0 because the new mail will be created with createMessage()
  return (IndexClass*)0;
}

IndexClass* RemoteMailFolder::moveMessage(IndexClass *idx)
{
	if(!idx)
	  return 0;
  IndexClass *newIndex=copyMessage(idx);
  idx->getParentFolder()->deleteMessage(idx);

  return newIndex;
}

bool RemoteMailFolder::expunge(bool force)
{
  printf("\nRemoteMailFolder::expunge is not implemented yet.\n");
  return force;
}

bool RemoteMailFolder::loadStatus()
{
	if( GlobalConfig->hasGroup(name()) )
	{
  	GlobalConfig->setGroup(name());
  	if( GlobalConfig->hasKey("MESSAGES") )
  	  status.messages=GlobalConfig->readUnsignedLongNumEntry("MESSAGES");
  	else
  	  status.messages=0;
  	if( GlobalConfig->hasKey("RECENT") )
  	  status.recent=GlobalConfig->readUnsignedLongNumEntry("RECENT");
  	else
  	  status.recent=0;
  	if( GlobalConfig->hasKey("UNSEEN") )
  	  status.unseen=GlobalConfig->readUnsignedLongNumEntry("UNSEEN");
  	else
  	  status.unseen=0;
  	if( GlobalConfig->hasKey("UIDNEXT") )
  	  status.uidNext=GlobalConfig->readUnsignedLongNumEntry("UIDNEXT");
  	else
  	  status.uidNext=0;
  	if( GlobalConfig->hasKey("UIDVALIDITY") )
  	  status.uidValidity=GlobalConfig->readUnsignedLongNumEntry("UIDVALIDITY");
  	else
  	  status.uidValidity=0;
  	return true;
	}
	else
	{
    status.messages=0;
    status.recent=0;
    status.unseen=0;
    status.uidNext=0;
    status.uidValidity=0;
	}
	
  return false;
}

bool RemoteMailFolder::saveStatus()
{
  	GlobalConfig->setGroup(name());
	  GlobalConfig->writeEntry("MESSAGES", status.messages);
	  GlobalConfig->writeEntry("RECENT", status.recent);
	  GlobalConfig->writeEntry("UNSEEN", status.unseen);
	  GlobalConfig->writeEntry("UIDNEXT", status.uidNext);
	  GlobalConfig->writeEntry("UIDVALIDITY", status.uidValidity);
  	return true;
}

unsigned long RemoteMailFolder::messages() const
{
  return status.messages;
}

unsigned long RemoteMailFolder::recent() const
{
  return status.recent;
}

unsigned long RemoteMailFolder::unseen() const
{
  return status.unseen;
}

unsigned long RemoteMailFolder::uidNext() const
{
  return status.uidNext;
}

unsigned long RemoteMailFolder::uidValidity() const
{
  return status.uidValidity;
}

void RemoteMailFolder::setStatus( unsigned long uidValidity, unsigned long uidNext,
                                  unsigned long messages, unsigned long unseen,
                                  unsigned long recent)
{
  status.uidValidity=uidValidity;
  status.uidNext=uidNext;
  status.messages=messages;
  status.unseen=unseen;
  status.recent=recent;
}


void RemoteMailFolder::showStatus() const
{
  printf("\nIMAP Folder Status\n");
  printf("UID Validity=%lu\n", status.uidValidity);
  printf("UID Next=%lu\n", status.uidNext);
  printf("Messages=%lu\n", status.messages);
  printf("Unseen=%lu\n", status.unseen);
  printf("Recent=%lu\n", status.recent);
  fflush(stdout);
}









