/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <orb.h>
#include <magellan.h>
#include <indexclass.h>

//#define DEBUG_URLMANAGER

URLManager::URLManager()
{
  // nothing here
}

URLManager::~URLManager()
{
  // nothing here
}

bool URLManager::isFolder(QString url)
{
	if(url.isEmpty()) return false;
	
	if(ObjectRequestBroker::thisInstance()->mailDict[url]) // check the mail folder dictionary
	{
	  return true;
	}
	if(ObjectRequestBroker::thisInstance()->collectionDict[url]) // check the collection dictionary
	{
	  return true;
	}

#ifdef DEBUG_URLMANAGER
  printf("server query: %s: no such interface\n", (const char *)url);
#endif

  return false;
}

bool URLManager::exists(QString url)
{
#ifdef DEBUG_URLMANAGER
	printf("calling exists <%s>\n",(const char *)url);
	fflush(stdout);
#endif
	if(url.isEmpty()) return false;
	
	// remove the query, if any
	if(url.contains('%')) url=url.left(url.findRev('%'));
	
  // check if the given url is a valid folder
  if(isFolder(url))
    return true;

  // debug
#ifdef DEBUG_URLMANAGER
  printf("urlmanager: exists query: %s is not a folder\n", (const char *)url);
#endif

	// check for subobject queries
	if(url.contains('.'))
	{
		if(url.right(13)=="/default.attr" || url.right(14)=="/default.fattr" || url.right(10)=="/dir.cache")
			return isFolder(url.left(url.find("/")));
		if(url.right(5)==".attr" || url.right(6)==".fattr")
			return exists(url.left(url.findRev('.')));
		
		// this should be a mail part query
		if(url.contains('.')>1)
			return false;
		
		IndexClass *idx=indexReference(url.left(url.find('.')));
		if(idx)
		{
			// the message exists, check the if the required part number is valid
			if(url.mid(url.find('.')+1).toUInt()<idx->getPartCount())
				return true;
			return false;
		}
	}
  else
  {
    // debug
#ifdef DEBUG_URLMANAGER
    printf("urlmanager: exists query: %s is not a subobject\n", (const char *)url);
#endif
  }
	
  // check if the given url is a valid object; this can happen only if the chunk before the last slash is a valid folder
  QString fpath=folderpath(url);
  if(isFolder(fpath))
  {
    // debug
#ifdef DEBUG_URLMANAGER
    printf("urlmanager: exists query: %s is a folder, searching\n", (const char *)fpath);
#endif

    // search for the object name in the folder
    QStringList t;
    if(type(url)==Magellan::Message)
      t=ObjectRequestBroker::thisInstance()->mailDict[fpath]->getEntryList();
    else
      t=ObjectRequestBroker::thisInstance()->collectionDict[fpath]->getEntryList();
#ifdef DEBUG_URLMANAGER
		for(int i=0;i<t.count();i++)
			printf("urlmanager: %d [%s]\n",i, (const char *)t[i]);
		printf("urlmanager: <%s> object.\n", (const char *)object(url));
		printf("urlmanager: <%s> URL.\n", (const char *)url);
    printf("urlmanager: %d objects.\n", t.grep(object(url)).count());
#endif
    if(t.grep(object(url)).count())
    {
      // debug
#ifdef DEBUG_URLMANAGER
      printf("urlmanager: got object\n");
#endif

      return true;
    }
  }

  printf("urlmanager: %s: no such device (authoritative)\n", (const char *)url);

  return false;
}

QString URLManager::parent(QString url)
{
	// folders don't have logical parents
	if(isFolder(url))
		return QString::null;

	// check for folder attributes
	if(url.mid(url.find('/')+1)=="default.attr" || url.mid(url.find('/')+1)=="default.fattr")
		return folderpath(url);
		
	// check for subobjects - they always have parents
	int k=url.findRev('.');
	if(k>-1)
		return url.left(k);
			
	if(subtype(url)==Magellan::Mail)
	{
    // have a look at the index relationship
    QString pidx=indexReference(url)->getParentID();
    return pidx.isEmpty()?folderpath(url):(folderpath(url)+'/'+pidx);
	}
	
	return folderpath(url);
}	

QString URLManager::folder(QString url)
{
  if(exists(url))
    return folderpath(url);

  return QString::null;
}

QStringList URLManager::children(QString url)
{
  QStringList tlist;

	// debug
	// printf("urlmanager: [children] on url %s\n", (const char *)url);
	
  if(isFolder(url))
  {
		// debug
		// printf("urlmanager: this is a folder\n");
		
		if(type(url)==Magellan::Message)
    {
			// debug
			// printf("urlmanager: this is a mail folder\n");
			
			tlist=mailfolderReference(url)->getEntryList();
			
			// debug
			// printf("urlmanager: response:\n\t%s\n", (const char *)tlist.join("\n\t"));
			
			return tlist;
    }
		else
    {
      // debug
#ifdef DEBUG_URLMANAGER
      printf("orb: entrylist query for %s\n", (const char *)url);
#endif

      ObjectRequestBroker::thisInstance()->clearStatus();
#ifdef DEBUG_URLMANAGER
			printf("<<<< Getting some entry list: %d\n",
				folderReference(url)->getEntryList().count());
#endif
    	return folderReference(url)->getEntryList();
    }
  }
  if(exists(url) && subtype(url)==Magellan::Mail)
  {
    // this is a mail object, scan the mail folder for possible index sons
//# warning implement children()
  }

  // return an empty list
  return tlist;
}

QStringList URLManager::related(QString url)
{
	QStringList rList;
	
	if(exists(url) && subtype(url)==Magellan::Mail)
	{
		// this is a mail object, query the related index id list
//#warning implement related()
	}
	
	return rList;
}

int URLManager::type(QString url)
{
	if(url.left(7)=="sysmail" || url.left(10)=="mailfolder" || url.left(5)=="imap_")
	{
		// debug
#ifdef DEBUG_URLMANAGER
		printf("urlmanager: type narrowed to mail\n");
#endif		
		return Magellan::Message;
	}
	
	if(url.left(11)=="syscontacts" || url.left(8)=="blockdev" || url.left(9)=="sysgroups")
		return Magellan::BlockObject;
	if(url.left(8)=="sysnotes" || url.left(4)=="file")
		return Magellan::FileObject;
	return Magellan::Unknown;
}

int URLManager::subtype(QString url)
{
	// remove all queries
	if(url.contains('%')) url=url.left(url.findRev('%'));
	
	if(url.right(6)==".cache")
	{
		return Magellan::FolderCache;
	}	
	
	if(url.right(5)==".attr" || url.right(6)==".fattr")
	{
		// an attribute type
		if(url.contains("default.attr") || url.contains("default.fattr"))
		{
			// debug
#ifdef DEBUG_URLMANAGER
			printf("urlmanager: type query narrowed to folderattribute\n");
#endif			
			return Magellan::FolderAttribute;
		}
		
		if(url.contains('.')>1)
		{
			// this should be a mail part attribute
			if(subtype(url.left(url.findRev('.')))==Magellan::Part)
			{
				// debug
#ifdef DEBUG_URLMANAGER
				printf("urlmanager: type query narrowed to partattribute\n");
#endif			
				
				return Magellan::PartAttribute;
			}
			else
			{
				// debug
#ifdef DEBUG_URLMANAGER
				printf("urlmanager: warning: type query could not be narrowed\n");
#endif			
				
				return Magellan::Unknown;
			}
		}
		
		switch(type(url.left(url.find('.'))))
		{
			case Magellan::Message:
				return Magellan::MailAttribute;
			case Magellan::BlockObject:
				return Magellan::BlockObjectAttribute;
			default:
				return Magellan::Attribute;
		}
	}
	
	if(url.find('.')>-1)
	{
		// this should be a mail part
		return Magellan::Part;
	}

	// return the storage device type	
	return type(url);
}

bool URLManager::setRelated(QString url, QStringList rList)
{
	if(subtype(url)!=Magellan::Mail)
		return false;
	
	IndexClass *idx=indexReference(url);
	if(idx)
	{
//#warning implement setRelated()
	}
	else
	{
		// this shouldn't occur
		printf("Warning: indexReference() returned null when a valid pointer was expected, bailing out\n");
		return false;
	}
	
	return true;
}

bool URLManager::setParent(QString url, QString parent)
{
	if(subtype(url)!=Magellan::Mail)
		return false;
	
	IndexClass *idx=indexReference(url);
	IndexClass *pdx=indexReference(folderpath(url)+"/"+parent);
	
	if(idx && pdx)
	{
//#warning implement setParent()
	}
	else
	{
		return false;
	}
	
	return true;
}

DataCollection *URLManager::folderReference(QString url)
{
	return ObjectRequestBroker::thisInstance()->collectionDict[url];
}

MailFolder *URLManager::mailfolderReference(QString url)
{
	return ObjectRequestBroker::thisInstance()->mailDict[url];
}

ObjectReference *URLManager::objectReference(QString url)
{
  DataCollection *t=ObjectRequestBroker::thisInstance()->collectionDict[folderpath(url)];
  if(t)
  {
    return t->getEntryDict()[object(url)];
  }
		
  return 0;
}

IndexClass *URLManager::indexReference(QString url)
{
  MailFolder *t=ObjectRequestBroker::thisInstance()->mailDict[folderpath(url)];
	
  if(t)
  {
		IndexClass *idx=t->indexIDict()[object(url)];
    return idx;
  }

  return 0;
}

QString URLManager::path(QString url)
{
  int k=url.find("://");
  if(k!=-1)
    return url.mid(k+3);

  return QString(url);
}

QString URLManager::object(QString url)
{
  QString t(url);
  if(t[t.length()-1]=='/')
    t.truncate(t.length()-1);

  int k=t.findRev('/');
  if(k!=-1)
    return t.mid(k+1);

  return QString::null;
}

QString URLManager::folderpath(QString url)
{
  int k=url.findRev('/');
  if(k!=-1)
    return url.left(k);

  return QString::null;
}


