/**
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <stdio.h>
#include <qstring.h>
#include <qlist.h>
#include <qvaluelist.h>

#define LET				100
#define PRINT			101
#define IF				102
#define INCLUDE		103

#define NO_ERROR			0
#define YES_ERROR			1
#define YES_WARNING		2


/***
*	LayoutScript
* @short Class parsing a script in order to generate a mail.
*/

class LayoutScript
{
	QValueList<Variable *> variables;
	QString content;
	QString script;
	QString result;
	Variable defaultVar;
	QString errorString;
	QString warningString;
	unsigned errorLine;
	unsigned errorCode;

	QString unfold(QString foldedText);
	QString getLine(QString text=QString::null, int position=0);
	Variable *getVariablePtr(QString name);
	void splitContent();
	QString internalParse(QString inputText);
	void multReplace(QString &str, QString oldS, QString newS);
	// methods for instructions...
	bool letFun(int lineno, QString line);
	bool declareFun(int lineno, QString line);
	bool inputFun(int lineno, QString line);
	bool printFun(int lineno, QString line);
	bool includeFun(int lineno, QString line);
	int ifFun(int lineno, QString line);
	// conversion funcs:
	QString itoa(unsigned i) { QString m; m.sprintf("%d",i); return m; }
public:
	/** 
	 *   The default constructor
	 * @param text the text of the file we want to parse
	 * it does not parse the file implicitely.
	 */
	LayoutScript(QString text=QString::null);
	/** 
	 *   This method is provided for creating the possibility of accessing 
	 * all the variables used/defined by the script.
	 * @return a QValueList<QString> containing all the names of variables
	 * used by the script.
	 * NOTE: It doesn't check the included files!
	 */
	QValueList<QString> getVariablesNames();
	/** 
	 *   This method provides the means to change the variable's content and/or
	 * type.
	 * @param name the name of the variable we want to change
	 * @param value the variable's new value
	 * @param type the variable's new type
	 * If any of the value or type parameters are QString::null then the old
	 * content is preserved for the corresponding field
	 * IMPORTANT: if the variable does not exist, it creates the variable and
	 * appends that variable to the variable list.
	 * Example:
	 *
	 * setVariable("test","New test string","String");
	 *
	 *   If the variable called "test" ($test$ in the script) exists, then the new
	 * type is "String" and the new value is "New test String".
	 *   If the variable called "test" does not exist, then it is created and
	 * appended to the list of variables.
	 *
	 *	This method can be used as a cast function to chang the variable's type.
	 */
	void setVariable(QString name=QString::null,
			QString value=QString::null,
			QString type=QString::null);
	/** 
	 *   This method is provided for specifying if a variable is to be entered
	 * by the user. (Mainly internal usage).
	 * @param name the variable's name
	 * @param isInput if true, then the variable is marked as a variable with
	 * the contents to be entered by the user. If false, then the variable is
	 * refreshed by the script.
	 */
	void setInputVariable(QString name=QString::null,bool isInput=false);
	/** 
	 *   This operator returns a reference to a variable identified by its name
	 * (almost useless).
	 * @param name the variable's name
	 * @return a reference to a Variable type for that name or an empty
	 * (name=QString::NULL, type=QString::null, value=QString::null) Variable.
	 */
	Variable &operator[](QString name);
	/** 
	 *   This method is used to replace the variables in the script. Is to be
	 * used before "parse()".
	 * @param all if TRUE replaces all the variables regardless of the content;
	 * if FALSE, then replaces only the variables with the non-void content.
	 */
	void replaceVariables(bool all=true);
	/** 
	 *   This method is used for unfolding the original text. If a line ends with
	 * \, then the following line is appended instead of the \ to the original
	 * line. Is like the Makefile multiple lines declarations or the multiline 
	 * #define in C/C++. The unfolded content is then placed in the internal
	 * variable "content" for further processing.
	 * @param text the source to be unfolded.
	 */
	void setText(QString text) { content=unfold(text); }
	/** 
	 *   This method looks in the source string  ("content") for variables and
	 * adds them to the variables list. It also emits a warning for the
	 * uninitialized variables (line + variable name).
	 */
	void generateVariables();
	/**
	 *	This method is used to replace stdout(...) or stderr(...) with the 
	 * actual content generated by the execution of the parameter.
	 * @return true always. Leaves place for enhancements.
	 */
	bool generateExec();
	/** 
	 *   This method is used for replacing control sequences (C style).
	 */
	void replaceWhiteSpaces();
	/** 
	 *   This method is provided for debug usage.
	 * @return the "content" (containing the almost final string).
	 */
	QString showContent() { return content; }
	/** 
	 *   This method is provided for parsed string accessing.
	 * @return the parsed string (with the included files too).
	 */
	QString showResult() { return result; }
	/** 
	 *   This method parses the string.
	 * The parsed string can then be accessed with showResult().
	 * Usage:
	 *	LayoutScript layout(scriptText);
	 *		layout.replaceVariables();
	 *		layout.parse();
	 *		layout.replaceWhiteSpaces();
	 *  Then acces the result (parsed text) with layout.showResult().
	 */
	QString parse() 
	{ 
		result=QString::null; 
		warningString="";
		return internalParse(content); 
	}
	/** 
	 *   This method is provided accessing the variables which have to be 
	 * entered by the user.
	 * @return a QValueList<Variable *> containing pointers to the variables from
	 * the object for ease of external refreshing.
	 */
	QValueList<Variable *> getInputVariables();
	/** 
	 *   This method is provided for debug purpose.
	 */
	void showValues();
	/** 
	 *   This method is provided post-parsing processing.
	 * @return if an error has occured during parsing, it will return the 
	 * line number, the error type and the line.
	 */
	QString getErrorString() 							{ return errorString; }
	/** 
	 *   This method is provided post-parsing processing.
	 * @return can be used for viewing all the warnings. The resulted string 
	 * will contain all the warnings (description, line (if possible)...).
	 */
	QString getWarningString() 						{ return warningString; }
	/** 
	 *   This method is provided post-parsing processing.
	 * @return the line where the error/last warning has occured.
	 */
	unsigned getErrorLine() 							{ return errorLine; }
	/** 
	 *   This method is provided post-parsing processing.
	 * @return the error code (NO_ERROR, YES_ERROR, YES_WARNING). It has to be
	 * processed "flag" style with `&` operator.
	 */
	int getErrorCode() 										{ return errorCode; }
	/** 
	 *   This method is provided post internal use.
	 * @param errstr the string containing an error description. Usually has the
	 * following format: "Error: description lineNo \n [line_content]"
	 */
	void setErrorString(QString errstr) 	{ errorString=errstr; }
	/** 
	 *   This method is provided post internal use.
	 * @param errstr the string containing an error description. Usually has the
	 * following format: "Warning: description lineNo \n [line_content]". 
	 * The difference to the @ref setErrorString is that it appends the string so 
	 * the user can see all the warnings( since the warnings are not fatal).
	 */
	void setWarningString(QString errstr) { warningString+=errstr+"\n"; }
	/** 
	 *   This method is provided post internal use.
	 * @param line the line number where the error/warning has occured.
	 */
	void setErrorLine(unsigned line) 			{ errorLine=line; }
	/** 
	 *   This method is provided post internal use.
	 * @param code the code (YES_ERROR, YES_WARNING).
	 */
	void setErrorCode(int code) 					{ errorCode|=code; }
// INPUT FUNCTIONS	
};








