// ---------------------------------------------------------------------------
// - Reactor.hpp                                                             -
// - standard object library - nuclear reactor class definition              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_REACTOR_HPP
#define  AFNIX_REACTOR_HPP

#ifndef  AFNIX_STRVEC_HPP
#include "Strvec.hpp"
#endif

namespace afnix {

  /// The Reactor class is a simple class used to produce quarks. A quark
  /// is a unique integer id for a string. When a string is interned in the
  /// reactor, a unique id is computed for that string. The obvious reason to
  /// use a quark instead of a string is the comparision speed benefits.
  /// Internally, the reactor is implemented as a hash table of integers.
  /// @author amaury darsch

  class Reactor {
  private:
    /// the hash table size
    long d_size;
    /// the number of elements
    long d_count;
    /// the threshold before resizing
    long d_thrs;
    /// the hash table is an array of quark node
    struct s_qnode** p_table;
    /// the string vector which remap the quark
    Strvec* p_vector;

  public:
    /// create a new reactor
    Reactor (void);
  
    /// destroy this reactor
    ~Reactor (void);

    /// intern a string and return the associated quark
    /// @param name the name to intern
    long intern (const String& name);

    /// remap a quark to a string array
    /// @param quark the quark to remap
    String qmap (const long quark) const;

  private:
    // make the copy constructor private
    Reactor (const Reactor&);
    // make the assignment operator private
    Reactor& operator = (const Reactor&);    
    // resize the hash table
    void resize (const long size);
  };
}

#endif
