// ---------------------------------------------------------------------------
// - HttpResponse.hpp                                                        -
// - afnix:nwg module - http response class definition                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_HTTPRESPONSE_HPP
#define  AFNIX_HTTPRESPONSE_HPP

#ifndef  AFNIX_INPUT_HPP
#include "Input.hpp"
#endif

#ifndef  AFNIX_PLIST_HPP
#include "Plist.hpp"
#endif

namespace afnix {

  /// The HttpResponse class is a http protocol class that handle the response
  /// following a http request. The response from a request generally
  /// contains a status line followed by a header and the body. The class
  /// is bound to an input stream which is used to read characters. The object
  /// encoding mode is set automatically from the header if it is available.
  /// @author amaury darsch

  class HttpResponse : public Input {
  protected:
    /// the input stream
    Input* p_is;
    /// the response status code
    long   d_code;
    /// the response header
    Plist d_head;
    /// the content length
    long  d_clen;
    /// the content counter
    long  d_ccnt;

  public:
    /// create a default http response
    HttpResponse (void);

    /// create a response by input stream
    /// @param is the input stream to bind
    HttpResponse (Input* is);
    
    /// destroy this http response
    ~HttpResponse (void);

    /// @return the class name
    String repr (void) const;

    /// reset the response and rescan
    void reset (void);

    /// set the http response input stream
    /// @param is the input stream to bind
    void setis (Input* is);

    /// @return the header length
    long hlength (void) const;
    
    /// @return true if a header property exists
    bool hexists (const String& name) const;
    
    /// @return a header property by index
    Property* hget (const long index) const;

    /// @return a header property by name
    Property* hfind (const String& name) const;

    /// @return a header property by name or throw an exception
    Property* hlookup (const String& name) const;

    /// @return a header property value by name
    String hgetval (const String& name) const;

    /// @return the response status code
    long getcode (void) const;

    /// @return the response media type
    String getmedia (void) const;

    /// @return true if the status is ok
    bool isok (void) const;

    /// @return true if the encoding mode is defined in the header
    bool isemod (void) const;

    /// @return true if there is something at another location
    bool ishloc (void) const;

    /// @retrun the header location value
    String gethloc (void) const;

    /// @return the stream descriptor
    int getsid (void) const;

    /// @return the next available character
    char read (void);

    /// @return true if we are at the eof
    bool iseof (void) const;

    /// check if we can read a character
    /// @param tout the timeout value
    bool valid (const long tout) const;

    /// push back a character in the input stream.
    /// @param value the character to push back
    void pushback (const char value);

    /// pushback a c-string
    /// @param s the string to pushback
    void pushback (const char* s);

    /// pushback a c-string by size
    /// @param s    the string to pushback
    /// @param size the string size
    void pushback (const char* s, const long size);

    /// pushback a string on this input stream
    void pushback (const String& value);

  private:
    // make the copy constructor private
    HttpResponse (const HttpResponse&);
    // make the assignment operator private
    HttpResponse& operator = (const HttpResponse&);

  public:
    /// create a new object in a generic object
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
