// ---------------------------------------------------------------------------
// - Xne.cpp                                                                 -
// - afnix:xml module - xne class implementation                             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Xne.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------
  
  // the expanded name separator
  static const t_quad XML_CHAR_DC = 0x0000003A; // :

  // the node selector type eval quarks
  static const long QUARK_ID    = String::intern ("ID");
  static const long QUARK_NAME  = String::intern ("NAME");
  static const long QUARK_INDEX = String::intern ("INDEX");
  static const long QUARK_XNE   = String::intern ("Xne");

  // map an item to a node selector type
  static inline Xne::t_xsel item_to_type (const Item& item) {
    // check for a key item
    if (item.gettid () != QUARK_XNE) {
      throw Exception ("item-error", "item is not a xne item");
    }
    // map the item to the enumeration
    long quark = item.getquark ();
    if (quark == QUARK_ID)    return Xne::XNE_ID;
    if (quark == QUARK_NAME)  return Xne::XNE_NAME;
    if (quark == QUARK_INDEX) return Xne::XNE_INDEX;
    throw Exception ("item-error", "cannot map item to xne selector type");
  }
  
  // -------------------------------------------------------------------------
  // - public section                                                        -
  // -------------------------------------------------------------------------

  // get a selector type by item

  Xne::t_xsel Xne::totype (const Item& item) {
    // check for a key item
    if (item.gettid () != QUARK_XNE) {
      throw Exception ("item-error", "item is not a xne item");
    }
    // map the item to the enumeration
    long quark = item.getquark ();
    if (quark == QUARK_ID)    return Xne::XNE_ID;
    if (quark == QUARK_NAME)  return Xne::XNE_NAME;
    if (quark == QUARK_INDEX) return Xne::XNE_INDEX;
    throw Exception ("item-error", "cannot map item to xne selector type");
  }
  
  // get a selector type by object
  
  Xne::t_xsel Xne::totype (Object* obj) {
    Item* iobj = dynamic_cast <Item*> (obj);
    if (iobj == nilp) {
      throw Exception ("type-error", "invalid object as a xne selector item",
		       Object::repr (obj));
    }
    return Xne::totype (*iobj);
  }
  
  // get an item by selector type
  
  Item* Xne::toitem (const t_xsel xsel) {
    switch (xsel) {
    case XNE_ID:
      return new Item (QUARK_XNE, QUARK_ID);
      break;
    case XNE_NAME:
      return new Item (QUARK_XNE, QUARK_NAME);
      break;
    case XNE_INDEX:
      return new Item (QUARK_XNE, QUARK_INDEX);
      break;
    }
    return nilp;
  }

  // return true if the name is a valid nc name

  bool Xne::isncnm (const String& name) {
    // get the name length
    long len = name.length ();
    for (long i = 0; i < len; i++) {
      if (name[i] == XML_CHAR_DC) return false;
    }
    return true;
  }

  // get the prefix from a name

  String Xne::getpnam (const String& name) {
    // the result buffer
    Buffer xbuf;
    // get the string length
    long len = name.length ();
    for (long i = 0; i < len; i++) {
      t_quad c = name[i];
      if (c == XML_CHAR_DC) {
        String result = xbuf.tostring ();
        return result;
      }
      xbuf.add (c);
    }
    // no prefix found
    return "";
  }

  // get the local name from a name

  String Xne::getlnam (const String& name) {
    // the result buffer
    Buffer xbuf;
    // get the string length
    long len = name.length ();
    bool flg = false;
    for (long i = 0; i < len; i++) {
      t_quad c = name[i];
      if ((c == XML_CHAR_DC) && (flg == false)) {
        xbuf.reset ();
        flg = true;
      } else {
        xbuf.add (c);
      }
    }
    // here is the local name
    return xbuf.tostring ();
  }

  // return the expanded name from a prefix and a local name
  
  String Xne::toexnm (const String& pnam, const String& lnam) {
    // check for nil namespace
    if (pnam.isnil () == true) return lnam;
    // check the names before expension
    if (Xne::isncnm (pnam) == false) {
      throw Exception ("name-error", "invalid prefix name to expand", pnam);
    }
    if (Xne::isncnm (lnam) == false) {
      throw Exception ("name-error", "invalid local name to expand", lnam);
    }
    // expand name
    String result = pnam;
    result += XML_CHAR_DC;
    result += lnam;
    return result;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // evaluate an object data member

  Object* Xne::meval (Runnable* robj, Nameset* nset, const long quark) {
    if (quark == QUARK_ID)    return new Item (QUARK_XNE, QUARK_ID);
    if (quark == QUARK_NAME)  return new Item (QUARK_XNE, QUARK_NAME);
    if (quark == QUARK_INDEX) return new Item (QUARK_XNE, QUARK_INDEX);
    throw Exception ("eval-error", "cannot evaluate member",
                     String::qmap (quark));
  }
}
