// ---------------------------------------------------------------------------
// - XneCond.cpp                                                             -
// - afnix:xml module - xne condition class implementation                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Boolean.hpp"
#include "Integer.hpp"
#include "XneCond.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the xml attributes names
  static const String XML_ID_ATTR = "id";

  // the condition structure
  
  struct t_cond {
    // the node selector type
    Xne::t_xsel d_type;
    // the condition name
    String d_name;
    // the cndition index
    long d_indx;
    // the next condition
    t_cond* p_next;
    // create a condition by type and name
    t_cond (const Xne::t_xsel type, const String& name) {
      d_type = type;
      d_name = name;
      d_indx = 0;
      p_next = nilp;
    }
    // create a condition by type and index
    t_cond (const Xne::t_xsel type, const long indx) {
      d_type = type;
      d_indx = indx;
      p_next = nilp;
    }
    // destroy this condition
    ~t_cond (void) {
      delete p_next;
    }
    // check if a node match a condition
    bool isvalid (const XmlNode* node) const{
      // check for nil node
      if (node == nilp) return false;
      // check for id
      if (d_type == Xne::XNE_ID) {
	return node->isattr (XML_ID_ATTR, d_name);
      }
      // check for name
      if (d_type == Xne::XNE_NAME) {
	return node->isname (d_name);
      }
      // check for index
      if (d_type == Xne::XNE_INDEX) {
	return node->isnidx (d_indx);
      }
      // invalid type
      throw Exception ("internal-error", "unknown xne condition type");
    }
  };

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create an empty xne condition

  XneCond::XneCond (void) {
    p_cond = nilp;
  }

  // destroy this condition

  XneCond::~XneCond (void) {
    delete p_cond;
  }

  // return the class name

  String XneCond::repr (void) const {
    return "XneCond";
  }

  // check if a node match a condition

  bool XneCond::isvalid (const XmlNode* node) const {
    rdlock ();
    try {
      // loop in the conditions
      t_cond* cond = p_cond;
      while (cond != nilp) {
	if (cond->isvalid (node) == false) {
	  unlock ();
	  return false;
	}
	cond = cond->p_next;
      }
      unlock ();
      return true;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a condition by type and name
  
  void XneCond::add (const Xne::t_xsel type, const String& name) {
    wrlock ();
    try {
      // create a new condition
      t_cond* cond = new t_cond (type, name);
      // link to root
      cond->p_next = p_cond;
      p_cond = cond;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a condition by type and index
  
  void XneCond::add (const Xne::t_xsel type, const long indx) {
    wrlock ();
    try {
      // create a new condition
      t_cond* cond = new t_cond (type, indx);
      // link to root
      cond->p_next = p_cond;
      p_cond = cond;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_ADD    = zone.intern ("add");
  static const long QUARK_VALIDP = zone.intern ("valid-p");

  // create a new object in a generic way

  Object* XneCond::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new XneCond;
    throw Exception ("argument-error",
                     "too many argument with xne condition constructor");
  }

  // return true if the given quark is defined

  bool XneCond::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* XneCond::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_VALIDP) {
        Object* obj = argv->get (0);
        XmlNode* node = dynamic_cast <XmlNode*> (obj);
        if ((obj != nilp) && (node == nilp)) {
          throw Exception ("type-error", "invalid object with valid-p",
                           Object::repr (obj));
        }
        return new Boolean (isvalid (node));
      }
    }
    if (argc == 2) {
      if (quark == QUARK_ADD) {
	// get the type
	Xne::t_xsel type = Xne::totype (argv->get(0));
	// check for a string
	Object*  obj = argv->get (1);
        String* sobj = dynamic_cast <String*> (obj);
	if (sobj != nilp) {
	  add (type, *sobj);
	  return nilp;
	}
	// check for an integer
        Integer* iobj = dynamic_cast <Integer*> (obj);
	if (iobj != nilp) {
	  add (type, iobj->tointeger ());
	  return nilp;
	}
	throw Exception ("type-error", "invalid object with add",
			 Object::repr (obj));
      }
    }
    // check the object
    return Object::apply (robj, nset, quark, argv);
  }
}
