/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#include <iostream>
#include <sstream>
#include <fstream>

#include <glibmm/error.h>
#include <gdkmm/cursor.h>

#include "gcs-paletteview.h"
#include "gcs-colorswatch.h"
#include "core/log-stream.h"
#include "gcs-i18n.h"
#include "gcs-conf.h"
#include "palettetreemodel.h"

namespace gcs
{
    namespace Widgets
    {
        const gint PaletteView::m_minCellSize = 20;
        const gint PaletteView::m_minPaletteWidth = 200;
        const gint PaletteView::m_maxPaletteHeight = 100;

        PaletteView::PaletteView(Glib::ustring filename) :
            Gtk::Expander(), m_table(1, 1, true)
        {
            m_refPaletteModel = PaletteTreeModel::create(m_palette);
            m_scrolledWindow.set_border_width(Conf::UI_SPACING_SMALL);
            m_scrolledWindow.set_policy(Gtk::POLICY_NEVER, Gtk::POLICY_AUTOMATIC);

            // we need to put the table into an event box so that we can
            // respond to GDK enter and leave signals which are used to change
            // the cursor to a hand and back to a regular pointer
            m_eventBox.add(m_table);
            m_scrolledWindow.add(m_eventBox);
            m_vbox.pack_start(m_scrolledWindow);
            m_vbox.pack_start(m_paletteSelector);

            add(m_vbox);
            set_label(_("Palette"));

            // set spacing between table cells
            m_table.set_spacings(0);
            
            // connect signals so that we can change the X cursor
            m_eventBox.signal_enter_notify_event().connect(
                    sigc::mem_fun(*this, &PaletteView::on_enter_palette));
            m_eventBox.signal_leave_notify_event().connect(
                    sigc::mem_fun(*this, &PaletteView::on_leave_palette));
            m_paletteSelector.signal_changed().connect(
                    sigc::mem_fun(*this, &PaletteView::on_palette_file_changed));

            m_paletteSelector.set_palette_file(filename);
        }


        void PaletteView::reset(void)
        {
            LOG("resetting PaletteView");
            // delete the ColorSwatches from the widget
            std::list<Gtk::Widget*> swatches = m_table.get_children();
            for (std::list<Gtk::Widget*>::iterator sw = swatches.begin();
                    sw != swatches.end(); sw++)
            {
                m_table.remove(**sw);
                //delete (dynamic_cast<ColorSwatch*>(*sw));
                delete (*sw);
            }
            m_rows = 0;
        }

        bool PaletteView::set_from_file(Glib::ustring fname)
        {
            LOG("Setting PaletteView from file " << fname);
            std::ifstream file(fname.c_str());
            reset();
            m_palette.parse(file);
            populate();
            return false;
        }

        void PaletteView::populate(void)
        {
            hide_all();
            LOG("Populating PaletteView");
            guint cols = m_palette.columns();
            if (!cols) // columns wasn't defined by the palette file
            {
                cols = m_minPaletteWidth / m_minCellSize;
            }
            m_rows = m_palette.size() / cols;
            // round up -- ugly, i know. FIXME
            if (m_palette.size() % cols)
            {
                m_rows += 1;
            }
            m_table.resize(m_rows, cols);
            gint height = m_rows * 25;
            if (height > m_maxPaletteHeight)
            {
                height = m_maxPaletteHeight;
            }
            m_scrolledWindow.set_size_request(-1, height);

            for (guint r = 0; r < m_rows; r++)
            {
                for (guint c = 0; c < cols; c++)
                {
                    guint index = c + r * cols;
                    if (index < m_palette.size())
                    {
                        ColorSwatch *pSwatch = (new
                                ColorSwatch(m_palette[index]));
                        pSwatch->set_border_width(0);
                        pSwatch->signal_selected().connect(
                                sigc::bind(sigc::mem_fun(*this,
                                        &PaletteView::on_color_selected), index));
                        m_table.attach(*pSwatch, c, c + 1, r, r + 1);
                    }
                }
            }
            //set_label(m_refPaletteModel->get_name());
            show_all();
        }


        bool PaletteView::on_color_selected(gint index)
        {
            signal_color_selected().emit(m_palette[index]);
            LOG("PaletteView::on_color_selected");
            return true;
        }


        bool PaletteView::on_enter_palette(GdkEventCrossing* event)
        {
            Glib::RefPtr<Gdk::Window> win = m_eventBox.get_window();
            if (win)
            {
                Glib::RefPtr<Gdk::Display> dpy = Gdk::Display::get_default();
                Gdk::Cursor cursor(dpy, Gdk::HAND2);
                win->set_cursor(cursor);
            }
            return true;
        }


        bool PaletteView::on_leave_palette(GdkEventCrossing* event)
        {
            Glib::RefPtr<Gdk::Window> win = m_eventBox.get_window();
            if (win)
            {
                win->set_cursor();
            }
            return true;
        }


        void PaletteView::on_palette_file_changed(void)
        {
            std::string filename = m_paletteSelector.get_palette_file();
            set_from_file(filename);
            populate();
        }

    } // namespace Widgets
} // namespace gcs
