#ifndef _MUD_H
#define _MUD_H

#include "strings.h"
#ifdef WIN32
#include "../win32/GameTimer.h"
#else
#include "timing.h"
#endif

#define STAT_FRESH            0
#define STAT_GAMEPLAY		  1
#define STAT_VERBS_GAMEVERBS  2
#define STAT_VERBS_ACTIONS    3
#define STAT_VERBS_ABILITIES  4
#define STAT_VERBS_BUILDVERBS 5
#define STAT_VERBS_CHATLINES  6
#define STAT_AREAS            7
#define STAT_LEVELS           8
#define STAT_GAMEREADY        9

#define STAGE_CLEAN       0
#define STAGE_ERRLOG      1
#define STAGE_SYSLOG      2
#define STAGE_SUGGLOG     3
#define STAGE_BUGLOG      4
#define STAGE_TYPOLOG     5
#define STAGE_BOOTSTRAP   6
#define STAGE_USERDBASE   7
#define STAGE_GAMEPORT    8
#define STAGE_BUILDPORT   9
#define STAGE_ACCESSLIST 10
#define STAGE_RUNGAME    11

class Access_List;
class Player;
class Builder;
class Connection;
class Verb_List;
class Object_List;
class Specials;
class Ability;
class Race;
class MudObject;
class ErrLog;
class SysLog;
class Data_Log;
class Bulletin;
class Port;
class User_Dbase;
class Level_List;
class Entity;

/* struct to hold an individual who is currently fighting, so we can loop
   through all fighters quickly to perform action on them */

struct fighting_ind {
   Individual   *the_fighter;
   int          turn;
   fighting_ind *next_fighter;
};


class Mud
{
public:
	void set_boot_status(int new_val);
	int get_boot_status();
   int switch_for_builder(Player *the_player);
   int are_ports_combined();

   Mud(bool quiet_mode); /* constructor, bootstraps mud and opens listening 
                           socket */
   ~Mud();       /* destructor, disconnects socket and closes down mud */

   /******* Player/Builder database functions ******/
   Player     *get_player(char *the_name);   /* gets a plr based on name */
   Builder    *get_builder(char *the_name); /* gets a bldr based on name */
   Player     *get_first_player();
   Builder    *get_first_builder();
   int        remove_player(char *the_name);
   int        remove_builder(char *the_name);
   int        handle_quits(void);
   Player     *add_player();
   Builder    *add_builder();
   Builder    *add_builder(Connection *the_conn);
   User_Dbase *get_user_database();
   void       save_player(Player *the_player);
   void       save_all_players();
   void       read_all_inventories();
   int        sort_playerlist(char *sort_by);
   Level_List *get_levels();


   /****** Verb/Action/Abilities database functions *****/
   Verb_List   *get_player_verb_list();        /* gets the verb list */
   Verb_List   *get_builder_verb_list();
   int         reload_verbs(void);
   int         load_abilities(void); 
   int         load_chatlines(void); 

   /****** MudObject database functions *******/
   MudObject   *get_object(char *obj_name); /* gets an obj from database */
   Object_List *get_dbase();
   int         delete_object(MudObject *the_object);
   int         reload_area(char *areaname);
   Specials    *get_special(char *obj_name);
   Ability     *get_ability(char *obj_name);
   Race        *get_race(char *obj_name);
   int         delete_entity(Entity *the_entity, int debug_check);
   int         delete_entity(Entity *the_entity);
   int         display_races(Player *the_player);
   int         display_inclinations(Player *the_player);
   int         reload_races(void);
   int         reload_inclinations(void);
   int         reload_talents(void);
   int         reload_abilities(void);
   int         reload_bulletins(void);
   int         reload_masks(void);

   /****** Log functions **********************/
   int    log_error(char *the_error, char *the_funct);
   int    log_event(char *the_event);
   ErrLog  *get_log();
   SysLog  *get_syslog();
   Data_Log *get_suggestlog();
   Data_Log *get_buglog();
   Data_Log *get_typolog();

   /****** Connection handling funtions ***********/
   Access_List *get_bldr_access_list();
   Access_List *get_game_access_list();
   int        check_sockets();
   Port		  *get_game_socket();
   Port		  *get_build_socket();

   //   int        get_game_socket();
   //   int        get_build_socket();

   void       increment_lagged_cycle();


   /****** general bootstrap/shutdown functions *******/
   int is_shutdown();
   int shutdown();   /* shuts down the mud */
   int reset_associations();
   int clear_associations();
   int reload_all(); 
   void startup(bool quiet_mode);
   int startup_stage(int stage, bool quiet_mode);
   int shutdown_stage(int stage);

   /****** communications functions *****************/
   int send_all_players(char *the_message, Player *the_player);
   int send_all_builders(char *the_message, Builder *the_builder);
   int send_all_players(char *the_message, Player *the_player, 
                                                        int with_adm_flag);
   int send_all_builders(char *the_message, Builder *the_builder, 
                                                        int with_adm_flag);
   int send_all_players(char *the_message);
   int send_all_builders(char *the_message);
   int send_all_outside(char *the_message);
   int flush_players(void);


   /***** cycling functions *****************/
   int cycle_world(void);
   int cycle_mudobjects(void);
   int cycle_players(void);
   int check_players();
   int check_builders();
   int start_timers();
   int check_events();
   int handle_fights(void);  

   /****** environment functions *********/
   TimeOfDay get_tod();
   int       get_doy();
   int       get_yoa();
   int       is_summer();
   void      increment_tod();
   void      set_tod(TimeOfDay the_time);
   void      set_doy(int the_day);
   void      set_yoa(int the_year);
   void      show_tod(Player *the_player);
   int       save_game_data(ErrLog *error_log);
   int       load_game_data(ErrLog *error_log);
   void      set_def_start_loc(char *new_start);
   char      *get_def_start_loc(void);


   /********** fighting functions *********/
   int          remove_fighter(Individual *the_ind);
   int          clear_fighter(Individual *the_ind);
   int          add_fighter(Individual *the_ind, int strike_now);
   void         reset_fight_list();
   fighting_ind *get_next_fighter();
   int          get_num_attacking(Individual *the_ind);

   int          display_mud_stats(Player *the_player);

   Bulletin     *has_bulletin(MudObject *the_obj);
   int          clear_bulletin_users();

   int       bootstrap(bool quiet_mode);  /* boots up the mud, loading mud data */
   int       bootstrap(bool quiet_mode, int stage);  
   int       shutdown_mud();
   int       shutdown_mud(int stage);

	int start_convert(char *filename);
	void unlock_bootstrap();
#ifdef WIN32
	CGameTimer * get_timer();
#else
	Timing *get_timer();
#endif

private:
   int boot_locked;
   int combined_ports;

   int remove_all_players(void);
   int remove_all_builders(void);
   int load_actions();

   int                status;        // Current mud status as defined above

   Port               *game_port;    /* the mud port */
   Port               *build_port;   /* the builder port */
   Access_List        *builder_access_list;  /* what hosts can access 
                                                the builder port */
   Access_List        *game_access_list; /* what hosts are denied from game
                                            access */

   Object_List        *mudobjects;   /* the list of objects */
   Verb_List          *verbs;        /* pointer to the verb list */
   Verb_List          *buildverbs;   /* verbs for the builders */
   Player             *player_list;  /* the linked list of connected players */
   Builder            *builder_list; /* linked list of connected builders */
   User_Dbase         *user_database; /* the user database stored on disk */
   Level_List         *game_levels;  /* the levels in the game */

   ErrLog             *error_log;    /* the error log */
   SysLog             *sys_log;      /* the syslog */
   Data_Log           *suggest_log;  /* the suggestions log */
   Data_Log           *bug_log;      /* a log full of bugs, big juicy ones */
   Data_Log           *typo_log;     /* log for user reported typos */

#ifdef WIN32
   CGameTimer		  the_timer;
#else
   Timing             the_timer;
#endif

   int                shutdwn;      /* are we shutting down the mud */ 

   /* environment variables */
   TimeOfDay          time_of_day;
   int                day_of_year;
   int                year_of_age;
   Strings            def_start_loc; /* the default start location */

   fighting_ind       *fight_list;
   fighting_ind       *cur_fighter;

   /* time to loop through */
   long num_cycle;
   long max_cycle;
   long avg_cycle;
   long min_cycle;
 
   long lagged_cycle;
};

#endif






