/**********************************************************************
 ** Action - contains methods and attributes for an action verb, which
 **          is a verb that is mostly for show, and doesn't really do
 **          anything except pass messages
 **
 ** Last reviewed: v.51
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the
 **   License, or any later version.
 **
 **   This program is distributed in the hope that it will be useful, but
 **   WITHOUT ANY WARRANTY; without even the implied warranty of
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 **   General Public License for more details.
 **
 **   You should have received a copy of the GNU General Public License
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **
 **********************************************************************/

#ifndef ACTION_C
#define ACTION_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "action.h"
#include "lexer.h"
#include "builder.h"
#include "parse.h"
#include "utils.h"
#include "location.h"
#include "player.h"
#include "actflags.h"
#include "objtype.h"
#include "newfuncts.h"
#include "flags.h"

/***********************************************************************
 ** Action (constructor) - loads action name and attributes
 **
 ** Parameters: action_name - this action name
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Action::Action(char *action_name)
{
   Strings tmp_name;

   tmp_name.assign_word(action_name, 1);

   if (tmp_name.str_len() > MAX_VERB_LEN)
      tmp_name.truncate(MAX_VERB_LEN);
   set_name(tmp_name.str_show());

   modified = 0;
   obj_type = OBJ_TYPE_ACTION;
   act_flags = new_Flags(1);
}


/***********************************************************************
 ** ~Action (destructor) - cleans up the action
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Action::~Action(void)
{
   delete_Flags(act_flags);
}


/***********************************************************************
 ** set_actor - sets the actor string for this action
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Action::set_actor(char *the_string)
{
   if (the_string == NULL)
      return -1;

   actor = the_string;
   return 1;
}

/***********************************************************************
 ** get_actor - gets the actor string for this action
 **
 ** Parameters: None
 **
 ** Returns: pointer to the actor string
 **
 ***********************************************************************/

char *Action::get_actor(void)
{
   return actor.str_show();
}



/***********************************************************************
 ** set_crowd - sets the crowd string for this action
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Action::set_crowd(char *the_string)
{
   if (the_string == NULL)
      return -1;

   crowd = the_string;
   return 1;
}


/***********************************************************************
 ** get_crowd - gets the crowd string for this action
 **
 ** Parameters: None
 **
 ** Returns: pointer to the crowd string
 **
 ***********************************************************************/

char *Action::get_crowd(void)
{
   return crowd.str_show();
}


/***********************************************************************
 ** set_target - sets the target string for this action
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Action::set_target(char *the_string)
{
   if (the_string == NULL)
      return -1;

   target = the_string;
   return 1;
}


/***********************************************************************
 ** get_target - gets the target string for this action
 **
 ** Parameters: None
 **
 ** Returns: pointer to the target string
 **
 ***********************************************************************/

char *Action::get_target(void)
{
   return target.str_show();
}



/***********************************************************************
 ** set_sender - sets the sender string for this action
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Action::set_sender(char *the_string)
{
   if (the_string == NULL)
      return -1;

   sender = the_string;
   return 1;
}


/***********************************************************************
 ** get_sender - gets the sender string for this action
 **
 ** Parameters: None
 **
 ** Returns: pointer to the sender string
 **
 ***********************************************************************/

char *Action::get_sender(void)
{
   return sender.str_show();
}



/***********************************************************************
 ** set_bystander - sets the bystander string for this action
 **
 ** Parameters: the_bystander - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Action::set_bystander(char *the_string)
{
   if (the_string == NULL)
      return -1;

   bystander = the_string;
   return 1;
}


/***********************************************************************
 ** get_bystander - gets the bystander string for this action
 **
 ** Parameters: None
 **
 ** Returns: pointer to the bystander string
 **
 ***********************************************************************/

char *Action::get_bystander()
{
   return bystander.str_show();
}

/***********************************************************************
 ** load_action - loads an action from a file into memory
 **
 ** Parameters: the_file - where we are getting the action from
 **
 ** Returns: pointer to the next action
 **
 ***********************************************************************/

int Action::load_action(FILE *the_file, ErrLog *error_log, int is_builder)
{
   token_record *the_token;
   char         *tmp_charholder;
   Flags        *tmp_actflags;
   Strings      holder;

   if (is_builder)
   {
       the_token = get_token(the_file,'\0');

       if (the_token->token_type != T_NUMERICAL)
       {
	 error_log->invalid_header("actions", "load_action");
	 return -1;
       }
       set_modified(atoi(the_token->the_string));
   }

   /* get the actor */
   tmp_charholder = read_desc_type(the_file, error_log, NULL);
   set_actor(tmp_charholder);
   delete tmp_charholder;

   /* get the crowd */
   tmp_charholder = read_desc_type(the_file, error_log, NULL);
   set_crowd(tmp_charholder);
   delete tmp_charholder;

   /* get the target */
   tmp_charholder = read_desc_type(the_file, error_log, NULL);
   set_target(tmp_charholder);
   delete tmp_charholder;

   /* get the sender */
   tmp_charholder = read_desc_type(the_file, error_log, NULL);
   set_sender(tmp_charholder);
   delete tmp_charholder;

   /* get the bystander */
   tmp_charholder = read_desc_type(the_file, error_log, NULL);
   set_bystander(tmp_charholder);
   delete tmp_charholder;

   /* read in the item flags */
   tmp_actflags = get_actflags();
   if (tmp_actflags->read_flags(the_file, error_log) <= 0)
   {
     error_log->invalid_attr_format(get_name(), "actions", "actflags", 
				    "load_action");
     return -1;
   }

   return 1;
}


/***********************************************************************
 ** describe - describes the action to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Action::describe(Builder *the_builder)
{
   the_builder->send_bldr(_("\n&+GAction: \t&+M%s&*\n"), get_name());
   the_builder->send_bldr(_("&+GActor: \n&+w%s&*\n"), get_actor());
   the_builder->send_bldr(_("&+GCrowd: \n&+w%s&*\n"), get_crowd());
   the_builder->send_bldr(_("&+GTarget: \n&+w%s&*\n"), get_target());
   the_builder->send_bldr(_("&+GSender: \n&+w%s&*\n"), get_sender());
   the_builder->send_bldr(_("&+GBystander: \n&+w%s&*\n"), get_bystander());
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the action to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Action::describe(Player *the_player)
{
   the_player->send_plr(_("\n&+GAction: \t&+M%s&*\n"), get_name());
   the_player->send_plr(_("&+GActor: \n&+w%s&*\n"), get_actor());
   the_player->send_plr(_("&+GCrowd: \n&+w%s&*\n"), get_crowd());
   the_player->send_plr(_("&+GTarget: \n&+w%s&*\n"), get_target());
   the_player->send_plr(_("&+GSender: \n&+w%s&*\n"), get_sender());
   the_player->send_plr(_("&+GBystander: \n&+w%s&*\n"), get_bystander());
   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Action::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr(_("You can set the following attributes on an action.\n"
               "   actor, crowd, target, sender, bystander and actflags\n"));
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "actor",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&actor) < 0)
      {
         the_builder->send_bldr(_("Error reading in input, failed!\n"));
         return -1;
      }
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "crowd",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&crowd) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "target",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&target) < 0)
      {
         the_builder->send_bldr(_("Error reading in input, failed!\n"));
         return -1;
      }
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "sender",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&sender) < 0)
      {
         the_builder->send_bldr(_("Error reading in input, failed!\n"));
         return -1;
      }
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "bystander",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&bystander) < 0)
      {
         the_builder->send_bldr(_("Error reading in input, failed!\n"));
         return -1;
      }
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "actflags",
                               strlen(the_parsed->get_target1())))
   {
      Flags *tmp_actflags;
      int flagnum;
      Strings holder;

      tmp_actflags = get_actflags();

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("Set which action flag?\n"));
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);

      if ((flagnum =
       tmp_actflags->get_by_name(holder.str_show(), actflagnames)) == -1)
      {
         the_builder->send_bldr(_("That is not an action flag.\n"));
         return -1;
      }

      holder.assign_word(the_parsed->get_speech(), 2);
      if (holder.str_show() == NULL)
      {
         the_builder->send_bldr(_("Set that flag to what?\n"
                                "Valid choices are: On or Off\n"));
         return -1;
      }

      if (holder.str_n_cmp(_("On"), holder.str_len()))
         tmp_actflags->set_flag(flagnum);
      else if (holder.str_n_cmp(_("Off"), holder.str_len()))
         tmp_actflags->clr_flag(flagnum);
      else
      {
         the_builder->send_bldr(_("That is not a valid setting.\n"
                             "Valid choices are: On or Off\n"));
         return -1;
      }
      the_builder->send_bldr(_("Flag &+M%s&* has been set to: %s\n"),
          actflagnames[flagnum], (tmp_actflags->get_flag(flagnum)) ?
			     _("&+GOn&*") : _("&+ROff&*"));
      return 1;
   }
   the_builder->send_bldr(_("The attribute '%s' is not an action attribute.\n"),
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the action to a specified file in specified
 **                format
 **
 ** Parameters: the_file - the file to write to
 **             build_format - shall we use builder format or not
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

void Action::write_object(FILE *the_file, int build_format)
{
   Flags *tmp_actflags;

   fprintf(the_file, "\n%s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   fprintf(the_file, "^%s^\n", (get_actor() == NULL) ? "" : get_actor());
   fprintf(the_file, "^%s^\n", (get_crowd() == NULL) ? "" : get_crowd());
   fprintf(the_file, "^%s^\n", (get_target() == NULL) ? "" : get_target());
   fprintf(the_file, "^%s^\n", (get_sender() == NULL) ? "" : get_sender());
   fprintf(the_file, "^%s^\n\n", (get_bystander() == NULL) ? "" :
                                                          get_bystander());

   tmp_actflags = get_actflags();
   tmp_actflags->write_flag(the_file);

   return;
}


/***********************************************************************
 ** is_modified - has this action been modified?
 **
 ** Parameters: None
 **
 ** Returns:  1 for yes, 0 for no
 **
 ***********************************************************************/

int Action::is_modified(void)
{
   return modified;
}



/***********************************************************************
 ** set_modified - shall we set this?
 **
 ** Parameters: the_num - the number to set it to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Action::set_modified(int the_num)
{
   modified = the_num;
}


/***********************************************************************
 ** execute_action - executes an action command
 **
 ** Parameters: the_parsed - parse information from user input
 **             the_player - who is executing the action
 **
 ** Returns: 1 if successful, -1 if failed
 **
 ***********************************************************************/

int Action::execute_action(Parse *the_parsed, Player *the_player)
{
   Location   *player_loc;
   Strings    formatted;
   Flags      *tmp_actflags;
   MudObject  *target_ind;
   int        the_num = 0;

   tmp_actflags = get_actflags();
   player_loc = the_player->get_loc();

   /* if this is a targeted action, show appropriate strings */
   if ((tmp_actflags->get_flag(ACTFLAG_TARGETED)) &&
       (the_parsed->get_target1() != NULL))
   {
      /* don't perform the action on objects, only individuals */
      if (((target_ind = player_loc->
             find_contained(the_parsed->get_target1(), &the_num)) == NULL) ||
               ((target_ind->get_type() != OBJ_TYPE_PLAYER) &&
                         (target_ind->get_type() != OBJ_TYPE_MOBILE)))
         return -2;

      /* if it is a player, show them the string */
      if (target_ind->get_type() == OBJ_TYPE_PLAYER)
      {
         formatted.format_for_actions(target.str_show(), the_player,
                                               (Individual *) target_ind);
         ((Player *) target_ind)->send_plr("%s", formatted.str_show());
      }

      /* send to the actor */
      formatted.format_for_actions(sender.str_show(), the_player,
                                               (Individual *) target_ind);
      the_player->send_plr("%s", formatted.str_show());

      /* send to the bystanders */
      formatted.format_for_actions(bystander.str_show(), the_player,
                                               (Individual *) target_ind);
      if (target_ind->get_type() == OBJ_TYPE_PLAYER)
         player_loc->send_location(formatted.str_show(), the_player,
                                                  (Player *) target_ind);
      else
         player_loc->send_location(formatted.str_show(), the_player, ( Player * )NULL);

      return 1;

   }

   /* if the action can go to all in the room, send to them */
   if (tmp_actflags->get_flag(ACTFLAG_ALL))
   {
      formatted.format_for_actions(actor.str_show(), the_player, NULL);
      the_player->send_plr("%s", formatted.str_show());

      formatted.format_for_actions(crowd.str_show(), the_player, NULL);
      player_loc->send_location(formatted.str_show(), the_player);
   }


   return 1;
}



/***********************************************************************
 ** get_actflags - gets the actions flag structure
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the actions flag
 **
 ***********************************************************************/

Flags *Action::get_actflags()
{
   return act_flags;
}


/***********************************************************************
 ** copy_object - copies the action to an action of a different name
 **
 ** Parameters: copy_from - copy attributes from this object
 **
 ** Returns:  1 if succeeded
 **           0 if failed
 **
 ***********************************************************************/
int Action::copy_object(Entity *copy_obj)
{
   Action *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_ACTION)
      return 0;

   copy_from = (Action *) copy_obj;

   /******* set the action attributes *****/
   set_actor(copy_from->get_actor());
   set_crowd(copy_from->get_crowd());
   set_target(copy_from->get_target());
   set_sender(copy_from->get_sender());
   set_bystander(copy_from->get_bystander());

   act_flags->copy_flags(copy_from->get_actflags());

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Action::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Action::get_mem_size_dynamic()
{
   int  size = 0;

   size += get_mem_size_entity();
   size += actor.get_mem_size_dynamic();
   size += crowd.get_mem_size_dynamic();
   size += target.get_mem_size_dynamic();
   size += sender.get_mem_size_dynamic();
   size += bystander.get_mem_size_dynamic();
   size += act_flags->get_mem_size();

   return size;
}

#endif

